package gov.va.vamf.service.clio.flowsheet.cache;

import com.google.common.base.Throwables;
import com.google.common.cache.*;
import com.google.common.util.concurrent.*;
import gov.va.vamf.service.clio.application.config.*;
import gov.va.vamf.service.clio.flowsheet.cache.flowsheet.FlowsheetException;

import java.util.concurrent.*;

/**
 * Generic Cache that supports dynamic loading of data (R) for a given key (T).  It also supports reloading the data (R)
 * asynchronously after a given time. Each concrete cache supports a Loader that must be thread safe and is used to load
 * the data into the cache.
 *
 * Refresh interval should be less than the expiration interval.  Reload happens on an get of the item after the refresh time.
 * The get returns the value in the cache then reload happens asynchronously and item is replaced in the cache after it
 * is successfully loaded.  Reload does not through an exception to the client.
 *
 * @see com.google.common.cache.CacheLoader
 * @see gov.va.vamf.service.clio.flowsheet.cache.Loader
 * @see gov.va.vamf.service.clio.flowsheet.cache.BackgroundExecutorService
 * @see gov.va.vamf.service.clio.flowsheet.cache.CacheKey
 *
 * @see gov.va.vamf.service.clio.flowsheet.cache.location.LocationsCache
 * @see gov.va.vamf.service.clio.flowsheet.cache.flowsheet.FlowsheetListCache
 * @see gov.va.vamf.service.clio.flowsheet.cache.flowsheet.FlowsheetCache
 */
public abstract class ClioCache<T, R> {
    private static CacheTime cacheExpiration = new CacheTime("48h");
    private static CacheTime cacheRefresh = new CacheTime("24h");

    private LoadingCache<T, R> cacheItems;

    public static void init(ClioFlowsheetsConfig config) {
        cacheExpiration = config.cacheExpiration();
        cacheRefresh = config.cacheRefresh();
    }

    protected ClioCache(final Loader<T, R> loader) {
        cacheItems = CacheBuilder.newBuilder()
                .expireAfterWrite(cacheExpiration.duration(), cacheExpiration.timeUnit())
                .refreshAfterWrite(cacheRefresh.duration(), cacheRefresh.timeUnit())
                .build(new CacheLoader<T, R>() {
                           public R load(T key) throws FlowsheetException {
                               return loader.load(key);
                           }

                           @Override
                           public ListenableFuture<R> reload(final T key, R oldValue) throws FlowsheetException {
                               ListenableFutureTask<R> task = ListenableFutureTask.create(new Callable<R>() {
                                   public R call() {
                                       return loader.load(key);
                                   }
                               });
                               BackgroundExecutorService.executeTask(task);
                               return task;
                           }
                       }
                );
    }

    public R get(T key) {
        try {
            return cacheItems.get(key);
        } catch (RuntimeException re) {
            throw new FlowsheetException(500, "Unable to retrieve requested information.", Throwables.getRootCause(re).getMessage());
        } catch (Throwable t) {
            Throwables.propagateIfPossible(
                    t.getCause(), FlowsheetException.class);
            throw new FlowsheetException(500, "Unable to retrieve requested information.", Throwables.getRootCause(t).getMessage());
        }
    }
}