package gov.va.vamf.service.clio.observation.resources;

import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import gov.va.vamf.service.clio.ResourceTest;
import gov.va.vamf.service.clio.application.config.ClioFlowsheetsConfig;
import gov.va.vamf.service.clio.infrastructure.vista.ClioResultRecord;
import gov.va.vamf.service.clio.infrastructure.vista.ClioResultRecords;
import gov.va.vamf.service.clio.infrastructure.vista.mdws.ClioMdwsService;
import gov.va.vamf.service.clio.observation.representations.Observation;
import gov.va.vamf.service.clio.observation.representations.ObservationSet;

import java.io.File;
import java.io.IOException;

import org.aeonbits.owner.ConfigFactory;
import org.junit.Test;

import com.google.common.base.Charsets;
import com.google.common.io.Files;
import com.google.common.io.Resources;
import com.jayway.restassured.http.ContentType;
import com.jayway.restassured.response.Response;

public class ObservationResourcesTests  extends ResourceTest {

    public static final String OBSERVATION_URI = "/ClioResources/rest/v1/clio-service/dfn-777/observation";

    @Test
    public void saveTemperatureObservation() throws IOException {
        File file = new File(Resources.getResource("TemperatureObservation.json").getFile());

        ObservationSet observationSet = getRequestSpecWithBearerToken()
                .body(Files.toString(file, Charsets.UTF_8)).with().contentType(ContentType.JSON).when()
                .post(OBSERVATION_URI + "/d2f312e5-d0de-4caa-955e-d55ee5a7add4")
                .then()
                .body("id", notNullValue())
                .body("enteredBy", notNullValue())
                .body("observedBy", notNullValue())
                .body("observationDateTime", notNullValue())
                .extract().body().as(ObservationSet.class);

        assertEquals(observationSet.enteredDateTime, observationSet.observationDateTime);

        ClioResultRecord record = getObservationRecordFromMock(observationSet.observations.get(0));

        Observation observation = observationSet.observations.get(0);

        verifyObservation(observationSet, record, observation);

        String qualifiers = record.getString("QUALIFIERS");
        assertTrue(qualifiers.contains("{1964CB4C-0584-DA80-C7A9-E7400DCD817D}"));
        assertTrue(qualifiers.contains("{5076C291-4EE7-7F9B-E7A5-3BB0A78639B7}"));
        assertTrue(qualifiers.contains("{5842A4D9-56A1-7A5A-13C9-F481699C3A09}"));
    }

    @Test
    public void saveBloodPressureObservation() throws IOException {
        File file = new File(Resources.getResource("BloodPressureObservation.json").getFile());

        ObservationSet observationSet = getRequestSpecWithBearerToken()
                .body(Files.toString(file, Charsets.UTF_8)).with().contentType(ContentType.JSON).when()
                .post(OBSERVATION_URI + "/2b41adaf-76de-4083-85c7-b379974ac9e0")
                .then()
                .body("id", notNullValue())
                .body("enteredBy", notNullValue())
                .body("observedBy", notNullValue())
                .body("observationDateTime", notNullValue())
                .body("observations.size()", is(3))
                .extract().body().as(ObservationSet.class);

        for (Observation observation : observationSet.observations) {
            ClioResultRecord resultRecord = getObservationRecordFromMock(observation);
            verifyObservation(observationSet, resultRecord, observation);
        }
    }

    @Test
    public void savingBloodPressureObservationVistaError() throws IOException {
        File file = new File(Resources.getResource("BloodPressureVistaError.json").getFile());

        Response response = getRequestSpecWithBearerToken()
                .body(Files.toString(file, Charsets.UTF_8)).with().contentType(ContentType.JSON).when()
                .post(OBSERVATION_URI + "/2b41adaf-76de-4083-85c7-b379974ac9e0")
                .then()
                .body("detailedDescription", containsString("Source: clio"))
                .extract().response();
    }

    @Test
    public void savingBloodPressureObservationMdwsError() throws IOException {
        File file = new File(Resources.getResource("BloodPressureMdwsError.json").getFile());

        Response response = getRequestSpecWithBearerToken()
                .body(Files.toString(file, Charsets.UTF_8)).with().contentType(ContentType.JSON).when()
                .post(OBSERVATION_URI + "/2b41adaf-76de-4083-85c7-b379974ac9e0")
                .then()
                .body("detailedDescription", containsString("Source: interface"))
                .extract().response();
    }

    private ClioResultRecord getObservationRecordFromMock(Observation observation) {
        ClioFlowsheetsConfig config = ConfigFactory.create(ClioFlowsheetsConfig.class, System.getProperties());

        ClioMdwsService clioMdwsService = new ClioMdwsService("123", config.mockClioMdwsUrl(),config.mdwsConnectionTimeOut(),config.mdwsReadTimeOut());

        ClioResultRecords result = clioMdwsService.makeClioCall("GetObservationByID", observation.id);
        return result.next();
    }

    private void verifyObservation(ObservationSet observationSet, ClioResultRecord record, Observation observation) {
        assertEquals(observationSet.location.id, record.getLong("HOSPITAL_LOCATION_ID"));
        assertEquals(observationSet.observationDateTime, record.getString("OBSERVED_DATE_TIME"));
        assertEquals(observationSet.observedBy.uniqueId, record.getString("OBSERVED_BY_ID"));
        assertEquals(observationSet.patient.uniqueId, record.getString("PATIENT_ID"));
        assertEquals(observationSet.enteredDateTime, record.getString("ENTERED_DATE_TIME"));
        assertEquals(observationSet.enteredBy.uniqueId, record.getString("ENTERED_BY_ID"));
        assertEquals(observationSet.id, record.getString("SETS/SET/VALUE"));
        assertEquals(observation.id, record.getString("OBS_ID"));
        assertEquals(observation.parentId, record.getString("PARENT_ID"));
        assertEquals(observation.uniqueTermId.id, record.getString("TERM_ID"));
        assertEquals("1", record.getString("STATUS"));
        assertEquals(observation.value, record.getString("SVALUE"));
    }

    @Test
    public void saveInvalidObservation() throws IOException {
        File file = new File(Resources.getResource("InvalidObservation.json").getFile());

        getRequestSpecWithBearerToken()
                .body(Files.toString(file, Charsets.UTF_8)).with().contentType(ContentType.JSON).when()
                .post(OBSERVATION_URI + "/d2f312e5-d0de-4caa-955e-d55ee5a7add4")
                .then()
                .body("message", notNullValue())
                .body("uuid", notNullValue())
                .body("detailedDescription", notNullValue());
    }

    //Tests below here only for AgileX Vista verification


//    @Test
//    public void savingBloodPressureObservationVista() throws IOException {
//        File file = new File(Resources.getResource("BloodPressureVista.json").getFile());
//
//        Response response = getRequestSpecWithBearerToken()
//                .body(Files.toString(file, Charsets.UTF_8)).with().contentType(ContentType.JSON).when()
//                .post(OBSERVATION_URI + "/3E610D66-6AB2-4543-80ED-2B55869C2D63")
//                .then()
//                //.body("detailedDescription", containsString("Source: interface"))
//                .extract().response();
//
//        int i = 0;
//    }

//    @Test
//    public void saveNewFlowsheetObservation() throws IOException {
//        File file = new File(Resources.getResource("NewFlowsheetObservation.json").getFile());
//
//        ObservationSet observationSet = getRequestSpecWithBearerToken()
//                .body(Files.toString(file, Charsets.UTF_8)).with().contentType(ContentType.JSON).when()
//                .post(OBSERVATION_URI + "/0CF60069-BA57-44F8-ABAE-EAB2D163A246")
//                .then()
//                .body("id", notNullValue())
//                .body("enteredBy", notNullValue())
//                .body("observedBy", notNullValue())
//                .body("observationDateTime", notNullValue())
//                //.body("observations.size()", is(3))
//                .extract().body().as(ObservationSet.class);
//
//        int x=0;
//
//    }
//
//    @Test
//    public void saveRestraintFlowsheetObservation() throws IOException {
//        File file = new File(Resources.getResource("RestraintFlowsheetObservation.json").getFile());
//
//        ObservationSet observationSet = getRequestSpecWithBearerToken()
//                .body(Files.toString(file, Charsets.UTF_8)).with().contentType(ContentType.JSON).when()
//                .post(OBSERVATION_URI + "/4A42D756-593F-4CC7-91AD-A4F3FBAF27E8")
//                .then()
//                .body("id", notNullValue())
//                .body("enteredBy", notNullValue())
//                .body("observedBy", notNullValue())
//                .body("observationDateTime", notNullValue())
//                        //.body("observations.size()", is(3))
//                .extract().body().as(ObservationSet.class);
//
//        int x=0;
//
//    }
}
