package gov.va.vamf.service.clio.flowsheet.cache;

import gov.va.vamf.service.clio.application.config.*;
import gov.va.vamf.service.clio.application.representations.UniqueTermId;
import gov.va.vamf.service.clio.flowsheet.cache.flowsheet.*;
import gov.va.vamf.service.clio.flowsheet.representations.*;
import org.junit.*;

import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.*;

public class ClioCacheTests {

    private Loader loader;
    private FlowsheetItem item;
    private ClioFlowsheetsConfig config;
    private CacheKey<FlowsheetItem> key;

    @Before
    public void initBuilder() {
        loader = mock(FlowsheetRetrievalExecutive.class);
        item = new FlowsheetItem(new UniqueTermId("1", ""));
        key = new CacheKey<FlowsheetItem>(new FlowsheetItem(new UniqueTermId("1", "")), null);

        config = mock(ClioFlowsheetsConfig.class);
        when(config.cacheExpiration()).thenReturn(new CacheTime("4s"));
        when(config.cacheRefresh()).thenReturn(new CacheTime("1s"));

        FlowsheetCache.init(config);
    }

    @After
    public void cleanupCache() {
        FlowsheetCache.clearCache();
    }

    @Test
    public void cacheItemUsed() {
        when(loader.load(key)).thenReturn(new Flowsheet(item));
        FlowsheetCache cache = FlowsheetCache.getInstance(loader);

        cache.get(key);
        Flowsheet flowsheet = cache.get(key);

        assertEquals("1", flowsheet.uniqueTermId.id);
        verify(loader, times(1)).load(any(FlowsheetItem.class));
    }

    @Test
    public void repullAfterCacheExpires() throws InterruptedException {
        when(loader.load(key)).thenReturn(new Flowsheet(item));

        FlowsheetCache cache = FlowsheetCache.getInstance(loader);

        cache.get(key);

        Thread.sleep(5000);  //let cache expire

        Flowsheet flowsheet = cache.get(key);

        assertEquals("1", flowsheet.uniqueTermId.id);
        verify(loader, times(2)).load(any(FlowsheetItem.class));
    }

    @Test
    public void refreshCache() throws InterruptedException {
        Flowsheet first = new Flowsheet(item);
        first.comment = "first";

        Flowsheet second = new Flowsheet(item);
        second.comment = "second";

        when(loader.load(key)).thenReturn(first, second);

        FlowsheetCache cache = FlowsheetCache.getInstance(loader);

        cache.get(key);
        Flowsheet flowsheet = cache.get(key);

        Thread.sleep(2000);  //let cache pass refresh time

        //refresh should occur in back ground but return current item
        assertEquals("first", flowsheet.comment);

        flowsheet = cache.get(key);

        assertEquals("second", flowsheet.comment);
        verify(loader, times(2)).load(any(FlowsheetItem.class));

    }

    @Test(expected = FlowsheetException.class)
    public void exceptionRetrieveCacheItem() {
        when(loader.load(key)).thenThrow(new RuntimeException("Ahhhhhhhh"));

        FlowsheetCache cache = FlowsheetCache.getInstance(loader);
        cache.get(key);
    }
}
