package gov.va.vamf.service.clio.mdws.service;

import com.google.common.base.*;
import com.google.common.cache.*;
import com.google.common.collect.*;

import java.util.*;

public class ObservationCache {
    private final String XML_RECORD_TEMPLATE = "<![CDATA[<?xml version=\"1.0\" standalone=\"yes\"?><RESULTS>%s</RESULTS>]]>";

    private Cache<String, String[]> observations = CacheBuilder.newBuilder()
                                                                .maximumSize(1000)
                                                                .removalListener(new RemovalListener<String, String[]>() {
                                                                    @Override
                                                                    public void onRemoval(RemovalNotification<String, String[]> notification) {
                                                                        sets.remove(notification.getKey());
                                                                        qualifiers.removeAll(notification.getKey());
                                                                    }
                                                                }).build();

    private Map<String, String> sets = Maps.newConcurrentMap();
    private Multimap<String, String> qualifiers = ArrayListMultimap.create();

    public void add(String observation) {
        String[] values = Iterables.toArray(splitValues(observation), String.class);

        if (values.length > 0)
            observations.put(values[0], values);
    }

    public void addToSet(String observationSet) {
        String[] ids = getIds(observationSet);

        if (!Strings.isNullOrEmpty(ids[0]) && !Strings.isNullOrEmpty(ids[1]))
            sets.put(ids[1], ids[0]);
    }

    public void addQualifier(String qualifierForObseration) {
        String[] ids = getIds(qualifierForObseration);

        if (!Strings.isNullOrEmpty(ids[0]) && !Strings.isNullOrEmpty(ids[1]))
            qualifiers.put(ids[0], ids[1]);
    }

    private String[] getIds(String value) {
        String[] ids = new String[2];

        if (Strings.isNullOrEmpty(value))
            return ids;

        Iterator<String> values = splitValues(value).iterator();

        if (values.hasNext())
            ids[0] = values.next();

        if (values.hasNext())
            ids[1] = values.next();

        return ids;
    }

    private Iterable<String> splitValues(String values) {
        if (values == null)
            return Lists.newArrayList();

        return Splitter.onPattern("\\^").split(values);
    }

    public String getObservationRecord(String observationId) {
        String[] observation = observations.getIfPresent(observationId);

        if (observation == null)
            return "";

        String setId = Strings.nullToEmpty(sets.get(observationId));
        String qualifierIds = Joiner.on(",").skipNulls().join(qualifiers.get(observationId));

        observation = ObjectArrays.concat(observation, setId);
        observation = ObjectArrays.concat(observation, qualifierIds);

        String template = FlowsheetRecordTemplates.getTemplate("ObservationRecordTemplate.txt");

        return String.format(XML_RECORD_TEMPLATE, String.format(template, observation));
    }
}
