package gov.va.vamf.service.clio.mdws.service;

import com.google.common.base.*;
import com.google.common.collect.*;
import com.google.common.io.*;

import gov.va.vamf.service.clio.mdws.service.model.*;

import java.io.*;
import java.net.URL;
import java.util.*;

public class FlowsheetsCache {
    private String flowsheetList;
    private Multimap<String, View> viewsByFlowsheetId;
    private Map<String, View> viewsByViewId;

    private List<Unit> units = Lists.newArrayList();
    private List<Qualifier> qualifiers = Lists.newArrayList();

    private final String XML_RECORD_TEMPLATE = "<![CDATA[<?xml version=\"1.0\" standalone=\"yes\"?><RESULTS>%s</RESULTS>]]>";

    public void loadFlowsheets() {
        FlowsheetCacheBuilder builder = new FlowsheetCacheBuilder();

        builder.loadFlowsheets();

        flowsheetList = CharMatcher.JAVA_ISO_CONTROL.removeFrom(String.format(XML_RECORD_TEMPLATE, builder.getFlowsheetList()));
        viewsByFlowsheetId = builder.getViewsByFlowsheetId();
        viewsByViewId = builder.getViewsByViewId();
        units = builder.getUnits();
        qualifiers = builder.getQualifiers();
    }

    public String getFlowsheetList() {
        return flowsheetList;
    }

    public String getFlowsheetPageList(final String flowsheetId) {
        Collection<View> views = viewsByFlowsheetId.get(flowsheetId);

        if (views == null)
            return String.format(XML_RECORD_TEMPLATE, "");

        Collection<String> xmlViews = FluentIterable.from(views)
                                                    .transform(new Function<View, String>() {
                                                        @Override
                                                        public String apply(View input) {
                                                            return input.getPageView(flowsheetId);
                                                        }
                                                    })
                                                    .toList();

        return CharMatcher.JAVA_ISO_CONTROL.removeFrom(String.format(XML_RECORD_TEMPLATE, Joiner.on("").join(xmlViews)));
    }

    public String getFlowsheetViewTermList(String viewId) {
        View view = viewsByViewId.get(viewId);

        if (view == null)
            return String.format(XML_RECORD_TEMPLATE, "");

        Collection<String> xmlTerms = FluentIterable.from(view.viewTerms)
                                                    .transform(new Function<ViewTerm, String>() {
                                                        @Override
                                                        public String apply(ViewTerm input) {
                                                            return input.toString();
                                                        }
                                                    })
                                                    .toList();

        return CharMatcher.JAVA_ISO_CONTROL.removeFrom(String.format(XML_RECORD_TEMPLATE, Joiner.on("").join(xmlTerms)));
    }

    public String getFlowsheetViewFilterList(String viewId) {
        View view = viewsByViewId.get(viewId);

        if (view == null)
            return String.format(XML_RECORD_TEMPLATE, "");

        Collection<String> xmlTerms = FluentIterable.from(view.viewTerms)
                .transform(new Function<ViewTerm, String>() {
                    @Override
                    public String apply(ViewTerm input) {
                        return input.getFilters();
                    }
                })
                .toList();

        return CharMatcher.JAVA_ISO_CONTROL.removeFrom(String.format(XML_RECORD_TEMPLATE, Joiner.on("").join(xmlTerms)));
    }

    public String getTermUnits(final String termId) {
        Collection<String> xmlUnits = FluentIterable.from(units)
                                                    .filter(new Predicate<Unit>() {
                                                        @Override
                                                        public boolean apply(Unit input) {
                                                            return input.termId.equals(termId);
                                                        }
                                                    })
                                                    .transform(new Function<Unit, String>() {
                                                        @Override
                                                        public String apply(Unit input) {
                                                            return input.toString();
                                                        }
                                                    })
                                                    .toList();

        return CharMatcher.JAVA_ISO_CONTROL.removeFrom(String.format(XML_RECORD_TEMPLATE, Joiner.on("").join(xmlUnits)));
    }

    public String getTermQualifiers(String termIdAndTermType) {
        Iterable<String> parameters = Splitter.on("^")
                .trimResults()
                .omitEmptyStrings()
                .split(termIdAndTermType);

        Iterator<String> parameterIterator = parameters.iterator();
        final String termId = parameterIterator.next();
        final int qualifierType = Integer.parseInt(parameterIterator.next());

        Collection<String> xmlQualifiers = FluentIterable.from(qualifiers)
                .filter(new Predicate<Qualifier>() {
                    @Override
                    public boolean apply(Qualifier input) {
                        return input.termId.equals(termId) && input.type == qualifierType;
                    }
                })
                .transform(new Function<Qualifier, String>() {
                    @Override
                    public String apply(Qualifier input) {
                        return input.toString();
                    }
                })
                .toList();

        return CharMatcher.JAVA_ISO_CONTROL.removeFrom(String.format(XML_RECORD_TEMPLATE, Joiner.on("").join(xmlQualifiers)));

//        for (Flowsheet flowsheet : flowsheets) {
//            Optional<ViewTerm> viewTerm = flowsheet.findTerm(termId);
//            if (viewTerm.isPresent())
//                return String.format(XML_RECORD_TEMPLATE, viewTerm.get().getQualifiers(termId, termType));
//        }
//
//        return String.format(XML_RECORD_TEMPLATE, "");
    }

    public String getTerm(String termId) {
        TermRecordTemplate termRecordTemplate = new TermRecordTemplate();

        return String.format(XML_RECORD_TEMPLATE, termRecordTemplate.getTerm(termId));
    }

    public String getLocations() {

        URL resource = Resources.getResource("AvailableLocations.xml");
        try {
            return "<![CDATA[" + Resources.toString(resource, Charsets.UTF_8).replaceAll("\t", "").replaceAll("\n", "") + "]]>";
        } catch (IOException e) {
            System.out.println(e.getMessage());
            e.printStackTrace();
        }

        return String.format(XML_RECORD_TEMPLATE, "");
    }
}
