package gov.va.vamf.service.clio.mdws.service;

import gov.va.vamf.service.clio.mdws.service.model.Flowsheet;
import gov.va.vamf.service.clio.mdws.service.model.Qualifier;
import gov.va.vamf.service.clio.mdws.service.model.Unit;
import gov.va.vamf.service.clio.mdws.service.model.View;

import java.io.IOException;
import java.net.URL;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.codehaus.jackson.map.ObjectMapper;

import com.google.common.base.Function;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.FluentIterable;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.io.Resources;

public class FlowsheetCacheBuilder {
    ObjectMapper mapper = new ObjectMapper();

    private String flowsheetList;
    private Multimap<String, View> viewsByFlowsheetId;
    private Map<String, View> viewsByViewId;

    public void loadFlowsheets() {
    	URL resource = Resources.getResource("AvailableFlowsheets.json");

        List<Flowsheet> flowsheets = null;

        try {
            flowsheets = Arrays.asList(mapper.readValue(resource, Flowsheet[].class));
        } catch (IOException e) {
            System.out.println(e.getMessage());
            e.printStackTrace();
        }

        createFlowsheetListXml(flowsheets);
        createViewMaps(flowsheets);
    }

    private void createFlowsheetListXml(List<Flowsheet> flowsheets) {
        StringBuilder builder = new StringBuilder();

        for (Flowsheet flowsheet : flowsheets) {
            builder.append(flowsheet.toString());
        }

        flowsheetList =  builder.toString();
    }

    private void createViewMaps(List<Flowsheet> flowsheets) {
        viewsByFlowsheetId = ArrayListMultimap.create();

        for (Flowsheet flowsheet : flowsheets)
            viewsByFlowsheetId.putAll(flowsheet.id, flowsheet.views);

        viewsByViewId = FluentIterable.from(flowsheets)
                                    .transformAndConcat(new Function<Flowsheet, Iterable<View>>() {
                                        @Override
                                        public Iterable<View> apply(Flowsheet input) {
                                            return input.views;
                                        }
                                    })
                                    .uniqueIndex(new Function<View, String>() {
                                        @Override
                                        public String apply(View input) {
                                            return input.id;
                                        }
                                    });
    }

    public String getFlowsheetList() {
        return flowsheetList;
    }

    public List<Qualifier> getQualifiers() {
        
        URL resource = Resources.getResource("Qualifiers.json");

        try {
            return Arrays.asList(mapper.readValue(resource, Qualifier[].class));
        } catch (IOException e) {
            System.out.println(e.getMessage());
            e.printStackTrace();
        }

        return Lists.newArrayList();
    }

    public List<Unit> getUnits() {
        URL resource = Resources.getResource("Units.json");

        try {
            return Arrays.asList(mapper.readValue(resource, Unit[].class));
       } catch (IOException e) {
            System.out.println(e.getMessage());
            e.printStackTrace();
        }

        return Lists.newArrayList();
    }

    public Multimap<String, View> getViewsByFlowsheetId() {
        return viewsByFlowsheetId;
    }

    public Map<String, View> getViewsByViewId() {
        return viewsByViewId;
    }
}
