package gov.va.vamf.service.clio.mdws.service;

import static com.xebialabs.restito.builder.stub.StubHttp.whenHttp;
import static com.xebialabs.restito.semantics.Action.contentType;
import static com.xebialabs.restito.semantics.Action.custom;
import static com.xebialabs.restito.semantics.Action.stringContent;
import static com.xebialabs.restito.semantics.Condition.alwaysTrue;
import static com.xebialabs.restito.semantics.Condition.endsWithUri;
import static com.xebialabs.restito.semantics.Condition.withPostBodyContaining;

import java.io.IOException;
import java.net.URL;

import org.glassfish.grizzly.http.server.Response;

import com.google.common.base.CharMatcher;
import com.google.common.base.Charsets;
import com.google.common.base.Function;
import com.google.common.io.Resources;
import com.xebialabs.restito.semantics.Action;
import com.xebialabs.restito.server.StubServer;

public class ClioMdwsMockService {
    private static String soapTemplate;
    private static FlowsheetsCache flowsheetsCache = new FlowsheetsCache();
    private static ObservationCache observationCache = new ObservationCache();

    public static void main(String[] args) throws Exception {
        flowsheetsCache.loadFlowsheets();
        initSoapTemplate();

        StubServer server = new StubServer(8088).run();

        getSoapFault(server);

        getTermDetails(server);
        getTermQualifierPairs(server);
        getUnitsForTerm(server);
        getViewFilters(server);
        getViewTerms(server);
        getViewPageList(server);
        getFlowsheetList(server);
        getLocations(server);

        addSet(server);
        addObservation(server);
        addObservationToSet(server);
        addQualifier(server);

        getObservation(server);

        while (true) { Thread.sleep(10000); }
    }

    public static void addSet(StubServer server) {
        whenHttp(server).match(endsWithUri("/ClioSvc.asmx"), withPostBodyContaining("AddSet"))
                .then(contentType("text/xml"), stringContent(String.format(soapTemplate, "1^success")));
    }

    public static void addObservation(StubServer server) {
        Function<Response, Response> addObservation = new Function<Response, Response>() {
            @Override
            public Response apply(Response input) {
                GuidParameterExtractor extractor = new GuidParameterExtractor();

                try {
                    String observationValues = extractor.fromSoapMessage(input.getRequest().getInputStream());
                    observationCache.add(observationValues);

                    input.getWriter().write(String.format(soapTemplate, "1^success"));
                    input.setContentType("text/xml");
                } catch (IOException e) {
                    System.out.println(e);
                }

                return input;
            }
        };

        whenHttp(server).match(endsWithUri("/ClioSvc.asmx"), withPostBodyContaining("AddObservation"))
                .then(contentType("text/xml"), custom(addObservation));

        Function<Response, Response> returnVistaFailure = new Function<Response, Response>() {
            @Override
            public Response apply(Response input) {
                GuidParameterExtractor extractor = new GuidParameterExtractor();

                try {
                    String observationValues = extractor.fromSoapMessage(input.getRequest().getInputStream());
                    observationCache.add(observationValues);

                    input.getWriter().write(String.format(soapTemplate, "-1^Unspecified Error"));
                    input.setContentType("text/xml");
                } catch (IOException e) {
                    System.out.println(e);
                }

                return input;
            }
        };

        whenHttp(server).match(endsWithUri("/ClioSvc.asmx"), withPostBodyContaining("AddObservation"), withPostBodyContaining("{B15F2DF6-CE99-B847-FE6B-3D5F174A2BCD}"), withPostBodyContaining("100/100"))
                .then(contentType("text/xml"), custom(returnVistaFailure));

        Function<Response, Response> getFault = new Function<Response, Response>() {
            @Override
            public Response apply(Response input) {
                try {
                    URL resource = Resources.getResource("SoapFault.xml");
                    input.getWriter().write(CharMatcher.JAVA_ISO_CONTROL.removeFrom(Resources.toString(resource, Charsets.UTF_8)));
                    input.setContentType("text/xml");
                } catch (IOException e) {
                    System.out.println(e);
                }

                return input;
            }
        };

        whenHttp(server).match(endsWithUri("/ClioSvc.asmx"), withPostBodyContaining("AddObservation"), withPostBodyContaining("{B15F2DF6-CE99-B847-FE6B-3D5F174A2BCD}"), withPostBodyContaining("0/0"))
                .then(contentType("text/xml"), custom(getFault));
    }

    public static void addObservationToSet(StubServer server) {
        Function<Response, Response> addObservationToSet = new Function<Response, Response>() {
            @Override
            public Response apply(Response input) {
                GuidParameterExtractor extractor = new GuidParameterExtractor();

                try {
                    String observationValues = extractor.fromSoapMessage(input.getRequest().getInputStream());
                    observationCache.addToSet(observationValues);

                    input.getWriter().write(String.format(soapTemplate, "1^success"));
                    input.setContentType("text/xml");
                } catch (IOException e) {
                    System.out.println(e);
                }

                return input;
            }
        };

        whenHttp(server).match(endsWithUri("/ClioSvc.asmx"), withPostBodyContaining("AddObservationToSet"))
                .then(contentType("text/xml"), custom(addObservationToSet));
    }

    public static void addQualifier(StubServer server) {
        Function<Response, Response> addQualifier = new Function<Response, Response>() {
            @Override
            public Response apply(Response input) {
                GuidParameterExtractor extractor = new GuidParameterExtractor();

                try {
                    String observationValues = extractor.fromSoapMessage(input.getRequest().getInputStream());
                    observationCache.addQualifier(observationValues);

                    input.getWriter().write(String.format(soapTemplate, "1^success"));
                    input.setContentType("text/xml");
                } catch (IOException e) {
                    System.out.println(e);
                }

                return input;
            }
        };

        whenHttp(server).match(endsWithUri("/ClioSvc.asmx"), withPostBodyContaining("AddQualifier"))
                .then(contentType("text/xml"), custom(addQualifier));
    }

    public static void getObservation(StubServer server) {
        Function<Response, Response> getObsById = new Function<Response, Response>() {
            @Override
            public Response apply(Response input) {
                GuidParameterExtractor extractor = new GuidParameterExtractor();

                try {
                    String obsId = extractor.fromSoapMessage(input.getRequest().getInputStream());
                    input.getWriter().write(String.format(soapTemplate, observationCache.getObservationRecord(obsId)));
                    input.setContentType("text/xml");
                } catch (IOException e) {
                    System.out.println(e);
                }

                return input;
            }
        };

        whenHttp(server).match(endsWithUri("/ClioSvc.asmx"), withPostBodyContaining("GetObservationByID"))
                .then(contentType("text/xml"),  Action.custom(getObsById));
    }

    public static void getSoapFault(StubServer server) {
        Function<Response, Response> getFault = new Function<Response, Response>() {
            @Override
            public Response apply(Response input) {
                try {
                    URL resource = Resources.getResource("SoapFault.xml");
                    input.getWriter().write(CharMatcher.JAVA_ISO_CONTROL.removeFrom(Resources.toString(resource, Charsets.UTF_8)));
                    input.setContentType("text/xml");
                } catch (IOException e) {
                    System.out.println(e);
                }

                return input;
            }
        };

        whenHttp(server).match(alwaysTrue()).then(contentType("text/xml"), custom(getFault));
    }

    private static void getTermDetails(StubServer server) {
        Function<Response, Response> getTerm = new Function<Response, Response>() {
            @Override
            public Response apply(Response input) {
                GuidParameterExtractor extractor = new GuidParameterExtractor();

                try {
                    String termId = extractor.fromSoapMessage(input.getRequest().getInputStream());
                    input.getWriter().write(String.format(soapTemplate, flowsheetsCache.getTerm(termId)));
                    input.setContentType("text/xml");
                } catch (IOException e) {
                    System.out.println(e);
                }

                return input;
            }
        };

        whenHttp(server).match(endsWithUri("/ClioSvc.asmx"), withPostBodyContaining("GetTermByID"))
                .then(contentType("text/xml"),  Action.custom(getTerm));
    }

    private static void getTermQualifierPairs(StubServer server) {
        Function<Response, Response> getQualifiers = new Function<Response, Response>() {
            @Override
            public Response apply(Response input) {
                GuidParameterExtractor extractor = new GuidParameterExtractor();

                try {
                    String termIdAndTermType = extractor.fromSoapMessage(input.getRequest().getInputStream());
                    input.getWriter().write(String.format(soapTemplate, flowsheetsCache.getTermQualifiers(termIdAndTermType)));
                    input.setContentType("text/xml");
                } catch (IOException e) {
                    System.out.println(e);
                }

                return input;
            }
        };

        whenHttp(server).match(endsWithUri("/ClioSvc.asmx"), withPostBodyContaining("GetTermQualifierPairsByID"))
                .then(contentType("text/xml"),  Action.custom(getQualifiers));
    }

    private static void getUnitsForTerm(StubServer server) {
        Function<Response, Response> getUnits = new Function<Response, Response>() {
            @Override
            public Response apply(Response input) {
                GuidParameterExtractor extractor = new GuidParameterExtractor();

                try {
                    String termId = extractor.fromSoapMessage(input.getRequest().getInputStream());
                    input.getWriter().write(String.format(soapTemplate, flowsheetsCache.getTermUnits(termId)));
                    input.setContentType("text/xml");
                } catch (IOException e) {
                    System.out.println(e);
                }

                return input;
            }
        };

        whenHttp(server).match(endsWithUri("/ClioSvc.asmx"), withPostBodyContaining("GetTermUnitPairsByID"))
                .then(contentType("text/xml"),  Action.custom(getUnits));
    }


    private static void getViewFilters(StubServer server) {
        Function<Response, Response> getViewFilterList = new Function<Response, Response>() {
            @Override
            public Response apply(Response input) {
                GuidParameterExtractor extractor = new GuidParameterExtractor();

                try {
                    String viewId = extractor.fromSoapMessage(input.getRequest().getInputStream());
                    input.getWriter().write(String.format(soapTemplate, flowsheetsCache.getFlowsheetViewFilterList(viewId)));
                    input.setContentType("text/xml");
                } catch (IOException e) {
                    System.out.println(e);
                }

                return input;
            }
        };

        whenHttp(server).match(endsWithUri("/ClioSvc.asmx"), withPostBodyContaining("GetFlowsheetViewFilters"))
                .then(contentType("text/xml"),  Action.custom(getViewFilterList));
    }

    private static void getViewTerms(StubServer server) {
        Function<Response, Response> getViewTermList = new Function<Response, Response>() {
            @Override
            public Response apply(Response input) {
                GuidParameterExtractor extractor = new GuidParameterExtractor();

                try {
                    String viewId = extractor.fromSoapMessage(input.getRequest().getInputStream());
                    input.getWriter().write(String.format(soapTemplate, flowsheetsCache.getFlowsheetViewTermList(viewId)));
                    input.setContentType("text/xml");
                } catch (IOException e) {
                    System.out.println(e);
                }

                return input;
            }
        };

        whenHttp(server).match(endsWithUri("/ClioSvc.asmx"), withPostBodyContaining("GetFlowsheetViewTerms"))
                .then(contentType("text/xml"),  Action.custom(getViewTermList));
    }

    private static void getViewPageList(StubServer server) {
        Function<Response, Response> getPageList = new Function<Response, Response>() {
            @Override
            public Response apply(Response input) {
                GuidParameterExtractor extractor = new GuidParameterExtractor();

                try {
                    String flowsheetId = extractor.fromSoapMessage(input.getRequest().getInputStream());
                    input.getWriter().write(String.format(soapTemplate, flowsheetsCache.getFlowsheetPageList(flowsheetId)));
                    input.setContentType("text/xml");
                } catch (IOException e) {
                    System.out.println(e);
                }

                return input;
            }
        };

        whenHttp(server).match(endsWithUri("/ClioSvc.asmx"), withPostBodyContaining("GetFlowsheetPageList"))
                .then(contentType("text/xml"),  Action.custom(getPageList));
    }

    private static void getFlowsheetList(StubServer server) {
        whenHttp(server).match(endsWithUri("/ClioSvc.asmx"), withPostBodyContaining("GetFlowsheetList"))
                .then(contentType("text/xml"), stringContent(String.format(soapTemplate, flowsheetsCache.getFlowsheetList())));
    }

    private static void getLocations(StubServer server) {
        whenHttp(server).match(endsWithUri("/ClioSvc.asmx"), withPostBodyContaining("GetFacilityLocations"))
                .then(contentType("text/xml"), stringContent(String.format(soapTemplate, flowsheetsCache.getLocations())));
    }

    private static void initSoapTemplate() {
        if (soapTemplate != null)
            return;

        URL resource = Resources.getResource("SoapResponseTemplate.txt");
        try {
            soapTemplate = Resources.toString(resource, Charsets.UTF_8);
        } catch (IOException e) {
            System.out.println(e);
        }
    }
}
