REM
REM The script will create tablespace, schema, and table that are required by the Metrics Service.
REM The script is expected to be run by a Database Admin sysdba account.
REM The script expects three input values: 
REM the first value is the password for METRIC_AUDIT user, 
REM the seond value is the password for USERDB user,
REM and third value is the password for METRICS_USER user.
REM Example: "sqlplus / as sysdba @metrics_service_dbinstall_r1.sql metricAuditPwd userDbPwd metricsUserPwd"
REM

set pagesize 0
set linesize 500
set trimspool on
spool metrics_service_dbinstall_r1.log

WHENEVER SQLERROR EXIT SQL.SQLCODE
-- create encrypted tablespace
CREATE TABLESPACE METRIC_AUDIT_DATA 
  DATAFILE '/u01/apps/oracle/oradata/metric_audit_data01.dbf' SIZE 1024M autoextend on next 1024M
  encryption using 'AES256' default storage (encrypt);
CREATE TABLESPACE METRIC_AUDIT_IDX 
  DATAFILE '/u01/apps/oracle/oradata/metric_audit_idx01.dbf' SIZE 1024M autoextend on next 1024M
  encryption using 'AES256' default storage (encrypt);

CREATE TABLESPACE USERDB_DATA 
 DATAFILE '/u01/apps/oracle/oradata/userdb_data01.dbf' SIZE 16M autoextend on next 4M
  encryption using 'AES256' default storage (encrypt);

-- create user 
CREATE USER METRIC_AUDIT IDENTIFIED BY &1
 DEFAULT TABLESPACE METRIC_AUDIT_DATA
 QUOTA unlimited ON METRIC_AUDIT_DATA
 QUOTA unlimited ON METRIC_AUDIT_IDX;
 
grant create session to METRIC_AUDIT;
grant create table to METRIC_AUDIT;
grant CREATE MATERIALIZED VIEW to METRIC_AUDIT;
grant QUERY REWRITE to METRIC_AUDIT;

CREATE USER USERDB IDENTIFIED BY &2
  DEFAULT TABLESPACE USERDB_DATA
  QUOTA unlimited ON USERDB_DATA;

grant create session to USERDB;
grant create table to USERDB;

CREATE USER METRICS_USER IDENTIFIED BY &3
    DEFAULT TABLESPACE METRIC_AUDIT_DATA
    QUOTA unlimited ON METRIC_AUDIT_DATA
    QUOTA unlimited ON METRIC_AUDIT_IDX;
grant create session to METRICS_USER;
                       
-- create table
CREATE TABLE METRIC_AUDIT.METRIC 
   (ID VARCHAR2(40) PRIMARY KEY, 
    METRIC_NAME VARCHAR2(1000) not null, 
    START_DATE TIMESTAMP (6) not null, 
    END_DATE TIMESTAMP (6), 
    TIMING NUMBER(11,0), 
    START_TICKS NUMBER(20,0), 
    END_TICKS NUMBER(20,0), 
    RESULT_STATUS VARCHAR2(10),
    USER_ID varchar2(50),
    ACCESS_TOKEN varchar2(255),
    APP_NAME varchar2(50),
    PARENT_EVENT_ID VARCHAR2(40)
   ) TABLESPACE METRIC_AUDIT_DATA 
   STORAGE (INITIAL 500M);

CREATE INDEX METRIC_AUDIT.METRIC_START_DT_IDX ON METRIC_AUDIT.METRIC (START_DATE) TABLESPACE METRIC_AUDIT_IDX;

CREATE TABLE METRIC_AUDIT.METRIC_DATA 
   (ID RAW(16) PRIMARY KEY, 
    METRIC_ID VARCHAR2(40), 
    TAG VARCHAR2(100), 
    TAG_VALUE VARCHAR2(1000),
    CONSTRAINT METRIC_ID_FK FOREIGN KEY (METRIC_ID) REFERENCES METRIC_AUDIT.METRIC(ID)
   )TABLESPACE METRIC_AUDIT_DATA 
   STORAGE (INITIAL 128M);
   
CREATE INDEX METRIC_AUDIT.METRIC_ID_IDX ON METRIC_AUDIT.METRIC_DATA (METRIC_ID, TAG)
      TABLESPACE METRIC_AUDIT_IDX;

-- MATERIALIZED VIEW for data extraction
CREATE MATERIALIZED VIEW LOG ON METRIC_AUDIT.METRIC 
WITH ROWID, SEQUENCE(START_DATE, USER_ID, ACCESS_TOKEN, APP_NAME), PRIMARY KEY
   INCLUDING NEW VALUES; 
   
CREATE MATERIALIZED VIEW METRIC_AUDIT.USER_LOGIN_MV 
  REFRESH FAST
  NEXT trunc(sysdate+1)+1/24
  AS select min(start_date) as LOGIN_DATE, user_id, access_token, APP_NAME
    from metric_audit.metric
    where ACCESS_TOKEN is not null 
    and user_id is not null
    and APP_NAME is not null
    group by user_id, access_token, APP_NAME;

CREATE TABLE USERDB.CLIENT_ACCOUNT 
  ( CLIENT_ID VARCHAR2(30) PRIMARY KEY, 
    SECRET VARCHAR2(30) not null, 
    EXPIRED_DATE DATE
   )TABLESPACE userdb_data 
  STORAGE (INITIAL 4M);

CREATE OR REPLACE PUBLIC SYNONYM CLIENT_ACCOUNT FOR USERDB.CLIENT_ACCOUNT;
GRANT SELECT ON USERDB.CLIENT_ACCOUNT TO METRICS_USER;

CREATE OR REPLACE PUBLIC SYNONYM METRIC FOR METRIC_AUDIT.METRIC;
GRANT INSERT, SELECT ON METRIC_AUDIT.METRIC TO METRICS_USER;

CREATE OR REPLACE PUBLIC SYNONYM METRIC_DATA FOR METRIC_AUDIT.METRIC_DATA;
GRANT INSERT, SELECT ON METRIC_AUDIT.METRIC_DATA TO METRICS_USER;

CREATE OR REPLACE PUBLIC SYNONYM USER_LOGIN_MV FOR METRIC_AUDIT.USER_LOGIN_MV;
GRANT SELECT ON METRIC_AUDIT.USER_LOGIN_MV TO METRICS_USER;

@metrics_service_dml.sql

spool off;
