package com.agilex.vamf.metrics.aopaspect;

import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Set;

import javax.annotation.Resource;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.UriInfo;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.oauth2.common.DefaultOAuth2AccessToken;
import org.springframework.security.oauth2.provider.authentication.OAuth2AuthenticationDetails;

import com.agilex.healthcare.mobilehealthplatform.domain.DomainTransferObjectCollection;
import com.agilex.healthcare.mobilehealthplatform.domain.MhpUser;
import com.agilex.healthcare.mobilehealthplatform.healthcheck.SystemProperties;
import com.agilex.healthcare.mobilehealthplatform.security.MhpUserFactory;
import com.agilex.healthcare.utility.Hasher;
import com.agilex.healthcare.utility.NullChecker;
import com.agilex.vamf.metrics.MetricMessage;
import com.agilex.vamf.metrics.MetricMessagePublisher;
import com.agilex.vamf.metrics.domain.ServiceTiming;
import com.agilex.vamf.metrics.domain.ServiceTimingDetail;
import com.agilex.vamf.metrics.domain.ServiceTimingDetails;
import com.agilex.vamf.metrics.enumeration.ClosingStatus;
import com.agilex.vamf.metrics.enumeration.MetricsKey;
import com.agilex.vamf.metrics.exception.MetricsPublishException;

/**
 * Created with IntelliJ IDEA.
 * User: patelh
 * Date: 12/27/13
 * Time: 9:06 AM
 * To change this template use File | Settings | File Templates.
 */
public abstract class AbstractMetricsAspect {

	private static final Log serviceLogger = LogFactory.getLog(AbstractMetricsAspect.class);

	private static final String emptyString = "";
	private static final boolean logSystemProperties = false;
	private static boolean enableHashSensitiveData = false;
	
	@Resource
	MetricMessagePublisher messagePublisher;

	protected AbstractMetricsAspect() {}

	public abstract void restServicePointcut();

	@Around("restServicePointcut()")
	public Object recordTimingMetric(ProceedingJoinPoint joinPoint) throws Throwable{
	    return recordTiming(joinPoint);
	}

	public Object recordTiming(ProceedingJoinPoint joinPoint) throws Throwable{
	    String appName = null;
	    String accessToken = null;
	    String className = joinPoint.getTarget().getClass().toString();
	    String methodName = joinPoint.getSignature().getName();
	    String eventName = className + "." + methodName;

	    ServiceTiming serviceTiming = new ServiceTiming();

	    serviceTiming.setStartTicks(getCurrentTicks());
	    serviceTiming.setStartDate(getCurrentDateTime());
	    serviceTiming.appendTag(MetricsKey.START_TIME_KEY.toString(), Long.toString(getCurrentTicks()));
	
	    serviceTiming.setEvent(eventName);
	    appendUserInfo(serviceTiming);
		
	    serviceTiming.appendTag(MetricsKey.CLASS_NAME_KEY.toString(), className);
	    serviceTiming.appendTag(MetricsKey.METHOD_NAME_KEY.toString(), methodName);

    	    if (className.contains(MetricsKey.APPOINTMENT_RESOURCE_KEY.toString())) {
    	        appName = MetricsKey.APPOINTMENT_KEY.toString();
    	        Authentication authentication = getAuthentication();
    	        if (authentication.getDetails() instanceof OAuth2AuthenticationDetails) {
                    accessToken = ((OAuth2AuthenticationDetails)authentication.getDetails()).getTokenValue();
                    serviceTiming.setAccessToken(accessToken);
    	        }
    	    }
            if (className.contains(MetricsKey.MHP_USER_RESOURCE_KEY.toString())) {
                appName = MetricsKey.MHP_ROA_KEY.toString();
                Authentication authentication = getAuthentication();
                if (authentication.getDetails() instanceof OAuth2AuthenticationDetails) {
                    accessToken = ((OAuth2AuthenticationDetails)authentication.getDetails()).getTokenValue();
                    serviceTiming.setAccessToken(accessToken);
                }
            }
            if (className.contains(MetricsKey.MBB_CONTOLLER_KEY.toString())) {
                appName = MetricsKey.MBB_KEY.toString();
                Authentication authentication = getAuthentication();
                if (authentication.getDetails() instanceof OAuth2AuthenticationDetails) {
                    accessToken = ((OAuth2AuthenticationDetails)authentication.getDetails()).getTokenValue();
                    serviceTiming.setAccessToken(accessToken);
                }
            }
    	    if (methodName.equalsIgnoreCase(MetricsKey.LOGOUT_KEY.toString())) {
    	        appName = methodName;
    	    }
    	    if (methodName.equalsIgnoreCase(MetricsKey.VET_AUTH_KEY.toString()) || methodName.equalsIgnoreCase(MetricsKey.PROV_AUTH_KEY.toString())) {
		appName = MetricsKey.LOGIN_KEY.toString();
    	    }
    	    if (logSystemProperties) {
    	        serviceTiming.appendTag(MetricsKey.SYSTEM_PROPERTIES_KEY.toString(), System.getProperties().toString());
    	    }

    	    java.net.URI uri = null;
    	    for (Object argument : joinPoint.getArgs()) {
		if (argument instanceof UriInfo) {
		    UriInfo uriinfo = (UriInfo) argument;
		    uri = uriinfo.getRequestUri();
		    serviceTiming.appendTag(MetricsKey.REQUEST_KEY.toString(), uri);
		    SystemProperties.setValue(MetricsKey.LAST_TIMING_URI_KEY.toString(), uri.toString());
		}
		if (argument instanceof HttpHeaders) {
		    HttpHeaders headers = (HttpHeaders) argument;
		    Set<String> headerKeys = headers.getRequestHeaders().keySet();
		    for (String k : headerKeys) {
		        List<String> valueList = headers.getRequestHeader(k);
			for (String value : valueList) {
			    serviceTiming.appendTag(MetricsKey.HTTP_HEADER_KEY.toString() + k, value);
			    if (NullChecker.isNullish(accessToken) && MetricsKey.AUTHORIZATION_KEY.toString().equalsIgnoreCase(k)) {
				accessToken = stripChars(value, "Bearer ");
			    }						
			    if (MetricsKey.REFERER_KEY.toString().equalsIgnoreCase(k) && !StringUtils.isEmpty(value)) {
			        appName = extractAppNameFromReferer(value);
			        if (value.contains("?token=")) {
			            accessToken = extractTokenFromReferer(value);    
			        }
			    } else if (NullChecker.isNullish(appName) && MetricsKey.USER_AGENT_KEY.toString().equalsIgnoreCase(k) && !StringUtils.isEmpty(value)) {
				appName = extractAppNameFromUserAgent(value);
			    }
			}
		    }
		}
    	    }
    	    serviceTiming.setAppName(appName);
    	    if (appName != null && appName.length() > 50) {
    	        serviceTiming.setAppName(appName.substring(0, 50));
    	    }
    	    String logTemplate = "Service invoked [service=%s][uri=%s]";
    	    serviceLogger.debug(String.format(logTemplate, eventName, uri));
    	    Object result = null;
    	    int resultCount = 1;
    	    try {
    	        result = joinPoint.proceed();
		if (result instanceof DomainTransferObjectCollection) {
		    @SuppressWarnings("unchecked")
		    DomainTransferObjectCollection<Object> resultObjects = (DomainTransferObjectCollection<Object>) result;
		    resultCount = resultObjects.getSize();
		    serviceTiming.appendTag(MetricsKey.RESULT_COUNT_KEY.toString(), resultCount);
		}
		if (result instanceof ResponseEntity) {
		    DefaultOAuth2AccessToken defaultOAuth2AccessToken = (DefaultOAuth2AccessToken)((ResponseEntity)result).getBody();
		    accessToken = defaultOAuth2AccessToken.getValue();
		    appName = MetricsKey.LOGIN_KEY.toString();
		    serviceTiming.setAppName(appName);
		}
		if (result != null)
		    serviceTiming.appendTag(MetricsKey.RESULT_CLASS_KEY.toString(), result.getClass().getCanonicalName());
		else {
		    serviceTiming.appendTag(MetricsKey.RESULT_CLASS_KEY.toString(), emptyString);
		    resultCount = 0;
		}
		markComplete(serviceTiming, ClosingStatus.Success);

    	    } catch(Throwable e) {
		markComplete(serviceTiming, ClosingStatus.Exception);
		serviceTiming.setAccessToken(accessToken);
		publishMetrics(serviceTiming);			
		logTemplate = "Service invocation failed [service=%s][uri=%s][timing=%s]";
		serviceLogger.error(String.format(logTemplate, eventName, uri, getTiming(serviceTiming)), e);
		throw e;
    	    }
    	    if (methodName.equalsIgnoreCase(MetricsKey.DELETE_TOKEN_KEY.toString())) {
    	        Authentication authentication = getAuthentication();
		accessToken = ((OAuth2AuthenticationDetails)authentication.getDetails()).getTokenValue();
		appName = MetricsKey.LOGOUT_KEY.toString();
		serviceTiming.setAppName(appName);
    	    }
    	    serviceTiming.setAccessToken(accessToken);
    	    publishMetrics(serviceTiming);
    	    logTemplate = "Service invocation complete [service=%s][uri=%s][resultcount=%s][timing=%s]";
    	    serviceLogger.debug(String.format(logTemplate, eventName, uri, resultCount, getTiming(serviceTiming)));
    	    return result;
	}

	private String extractAppNameFromUserAgent(String value) {
	    return value.substring(0, value.toString().indexOf('/'));
	}

	private String extractAppNameFromReferer(String value) {
	    StringBuilder sb = new StringBuilder();
	    if ( value.substring(value.length() - 1).equalsIgnoreCase("/" )) {
	        sb.append(value.substring(0, value.length() - 1));
	    } else {
		sb.append(value);
	    }
	    String appName = sb.substring(sb.toString().lastIndexOf('/') + 1, sb.length());
	    if (appName.contains("?") || appName.contains(";")) {
		int lastIndex = sb.toString().lastIndexOf('/');
		int penultimateIndex = sb.toString().lastIndexOf('/', lastIndex -1);
		appName = sb.substring(penultimateIndex + 1, lastIndex);
	    }
	    return appName;
	}

        private String extractTokenFromReferer(String value) {
            StringBuilder sb = new StringBuilder();
            if ( value.substring(value.length() - 1).equalsIgnoreCase("/" )) {
                sb.append(value.substring(0, value.length() - 1));
            } else {
                sb.append(value);
            }
            String appName = sb.substring(sb.toString().lastIndexOf('/') + 1, sb.length());
            if (appName.contains("?token=")) {
                appName = appName.substring(appName.indexOf("=")+1);
            }
            return appName;
        }
	
	private void appendUserInfo(ServiceTiming serviceTiming) {
	    MhpUser currentUser = getCurrentUser();
	    String userId = currentUser == null ? "anonymous" : currentUser.getUserIdentifier().toString();
	    String username = currentUser == null ? "anonymous" : currentUser.getUserName();

	    serviceTiming.setUserId(userId);
	    serviceTiming.appendTag(MetricsKey.LOGGED_IN_USER_ID_KEY.toString(), userId);
	    serviceTiming.appendTag(MetricsKey.LOGGED_IN_USER_NAME_KEY.toString(), username);
	    if (currentUser != null) {
		serviceTiming.appendTag(MetricsKey.USERID_KEY.toString(), currentUser.getUserIdentifier().toString());   
	    } else {
	        serviceTiming.appendTag(MetricsKey.USERID_KEY.toString(), userId);
	    }

	}

	private MhpUser getCurrentUser(){
	    return MhpUserFactory.createFromSecurityContext();
	}

	private Authentication getAuthentication() {
	    SecurityContext ctx = SecurityContextHolder.getContext();
	    return ctx.getAuthentication();
	}
	
	private long getCurrentTicks() {
	    return Calendar.getInstance().getTimeInMillis();
	}

	private Date getCurrentDateTime() {
	    return new Date();
	}

	private void markComplete(ServiceTiming serviceTiming, ClosingStatus status){

	    serviceTiming.setEndDate(getCurrentDateTime());
	    serviceTiming.setEndTicks(getCurrentTicks());
	    serviceTiming.appendTag(MetricsKey.END_TIME_KEY.toString(), Long.toString(getCurrentTicks()));
	    serviceTiming.appendTag(MetricsKey.STATUS_KEY.toString(), status.toString());

	    hashSensitiveData(serviceTiming.getTags());
	}

	private static void hashSensitiveData(ServiceTimingDetails tags) {
	    Hasher hasher = new Hasher();
	    for (ServiceTimingDetail tag : tags) {
		if (enableHashSensitiveData && isSensitiveData(tag)) {
		    tag.setValue(hasher.hashStringToString(tag.getValue()));
		}
	    }
	}

	private static boolean isSensitiveData(ServiceTimingDetail serviceTimingDetail) {
	    if (serviceTimingDetail.getKey().equalsIgnoreCase(MetricsKey.URI_KEY.toString()))
		return true;
	    else if (serviceTimingDetail.getKey().equalsIgnoreCase(MetricsKey.URL_KEY.toString()))
		return true;
	    else if (serviceTimingDetail.getKey().equalsIgnoreCase(MetricsKey.REQUEST_PATH_KEY.toString()))
		return true;
	    else if (serviceTimingDetail.getKey().equalsIgnoreCase(MetricsKey.REQUEST_QS_KEY.toString()))
		return true;
	    else if (serviceTimingDetail.getKey().equalsIgnoreCase(MetricsKey.USER_NAME_KEY.toString()))
		return true;
	    else if (serviceTimingDetail.getKey().equalsIgnoreCase(MetricsKey.PATIENT_URI_KEY.toString()))
		return true;
	    else if (serviceTimingDetail.getKey().equalsIgnoreCase(MetricsKey.LAB_TEST_URI_KEY.toString()))
		return true;
	    else if (serviceTimingDetail.getKey().equalsIgnoreCase(MetricsKey.MEDICATION_URI_KEY.toString()))
		return true;
	    else
		return false;
	}

	public long getTiming(ServiceTiming serviceTiming) {
	    long endTicks = 0;
	    long startTicks = 0;
	    if (Long.valueOf(serviceTiming.getEndTicks()) != null && Long.valueOf(serviceTiming.getStartTicks()) != null){
	        endTicks =  serviceTiming.getEndTicks();
		startTicks = serviceTiming.getStartTicks();
		return (endTicks - startTicks);
	    }
	    return 0;
	}

	private void publishMetrics(ServiceTiming serviceTiming) {
	    assert(messagePublisher != null);
	    try {
	        messagePublisher.publishMessage(new MetricMessage(serviceTiming));
	    } catch (MetricsPublishException e) {
	        serviceLogger.error(new StringBuilder().append("Error, failed to publish event : ").append(serviceTiming.getEvent()).toString(), e);
	    }
	}

	private String stripChars(String input, String strip) {
	    if (input.length() > 7) {
	        StringBuilder result = new StringBuilder(input);
	        String strippedStr = result.substring(7, result.length());
	        return strippedStr;
	    }
	    return null;
	}
	  
	@Around("restServicePointcut()")
	public Object catchServiceException(ProceedingJoinPoint joinPoint) throws Throwable {
	    Object result;
	    try {
	      result = joinPoint.proceed();
	    } catch (Throwable e) {
	      serviceLogger.debug("Caught exception in metrics aspect", e);
	      throw e;
	    }
	
	    return result;
	}	
}
