package com.agilex.vamf.metrics;

import java.util.Calendar;
import java.util.Date;
import java.util.Map;
import java.util.Stack;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.agilex.vamf.metrics.domain.ServiceTiming;
import com.agilex.vamf.metrics.domain.ServiceTimingDetail;
import com.agilex.vamf.metrics.domain.ServiceTimingDetails;
import com.agilex.vamf.metrics.enumeration.ClosingStatus;
import com.agilex.vamf.metrics.exception.MetricsPublishException;
import com.agilex.vamf.metrics.utility.Hasher;

public class Timing extends ServiceTiming {

	private static final Log logger = LogFactory.getLog(Timing.class);
	
	private static final long serialVersionUID = -6103180049233345363L;
	private static boolean enableHashSensitiveData = false;
	
        @Resource
	MetricMessagePublisher publisher;

	static private Map<Long, Stack<Timing>> allTimings = new ConcurrentHashMap<Long, Stack<Timing>>();
    
	public Timing() {}
    	
	public static Timing getInstance(String event) {
		long myThreadId = Thread.currentThread().getId();
		Stack<Timing> relatedTimings = allTimings.get(myThreadId);

		String timingParentId = null;
		if (relatedTimings == null) {
			relatedTimings = new Stack<Timing>();
			allTimings.put(myThreadId, relatedTimings);
		} else if (relatedTimings.size() > 0) {
			timingParentId = relatedTimings.peek().getId();
		}

		Timing t = new Timing(event, timingParentId);
		t.appendTag("threadId", Long.toString(myThreadId));
		relatedTimings.add(t);

		return t;
	}

	public Timing(String event, String parentId) {
		this.setStartTicks(getCurrentTicks());
		this.setStartDate(getCurrentDateTime());
		this.parentId = parentId;
		this.setEvent(event);
	}

	private long getCurrentTicks() {
		return Calendar.getInstance().getTimeInMillis();
	}

	private void markCompleteInternal(ClosingStatus status) {
		this.setEndTicks(getCurrentTicks());
		this.setEndDate(getCurrentDateTime());
		appendTag("status", status.toString());

		boolean continuePopping = true;
		long myThreadId = Thread.currentThread().getId();
		Stack<Timing> relatedTimings = allTimings.get(myThreadId);

		if (relatedTimings != null) {
			if (!relatedTimings.contains(this)) {
				// appendTag("s2", ClosingStatus.Orphaned.toString());
			} else {
				do {
					if (relatedTimings.size() == 0)
						continuePopping = false;
					else {
						Timing poppedItem = relatedTimings.pop();
						if (poppedItem == this)
							continuePopping = false;
						else {
							poppedItem.markComplete(ClosingStatus.Orphaned);
						}
					}
				} while (continuePopping);
			}
		}
	}

	private Date getCurrentDateTime() {
		return new Date();
	}

	public void markComplete(ClosingStatus status) {
		markCompleteInternal(status);
		logEvent();
	}

	public static void markComplete(Timing t, ClosingStatus status) {
		if (t != null)
			t.markComplete(status);
	}

	public void logEvent() {
		logEvent(this);
	}

	public  void logEvent(ServiceTiming event) {
	    hashSensitiveData(event.getTags());
	    try {
                publishMetrics(event);
	    } catch (MetricsPublishException e) {
        	logger.error("Error happened in Publishing Metrics info : " + e.getMessage().toString());
	    }		
	}

        private void publishMetrics(ServiceTiming event) throws MetricsPublishException {
            if (event.getAppName() != null && event.getAppName().length() > 50) {
                event.setAppName(event.getAppName().substring(0, 50));
            }
    	    publisher.publishMessage(new MetricMessage(event));
        }
    
        public boolean isEnableHashSensitiveData() {
            return enableHashSensitiveData;
        }

        public void setEnableHashSensitiveData(boolean enableHashSensitiveData) {
            Timing.enableHashSensitiveData = enableHashSensitiveData;
        }
        
	private static void hashSensitiveData(ServiceTimingDetails tags) {
		Hasher hasher = new Hasher();

		for (ServiceTimingDetail tag : tags) {
			if (enableHashSensitiveData && isSensitiveData(tag))
				tag.setValue(hasher.hashStringToString(tag.getValue()));
		}
	}

	private static boolean isSensitiveData(ServiceTimingDetail tag) {
		if (tag.getKey().equalsIgnoreCase("URI"))
			return true;
		else if (tag.getKey().equalsIgnoreCase("URL"))
			return true;
		else if (tag.getKey().equalsIgnoreCase("request-path"))
			return true;
		else if (tag.getKey().equalsIgnoreCase("request-qs"))
			return true;
		else if (tag.getKey().equalsIgnoreCase("User.name"))
			return true;
		else if (tag.getKey().equalsIgnoreCase("Patient.uri"))
			return true;
		else if (tag.getKey().equalsIgnoreCase("labTest"))
			return true;
		else if (tag.getKey().equalsIgnoreCase("Medication.uri"))
			return true;
		else
			return false;

	}

}
