define(['InputVideo'], function () {
    'use strict';

    describe('Input Video Directive', function () {
        var scope,
            timeout,
            el,
            clickMock,
            globalDelayTrackerMock,
            file,
            attrs = {
                'video-object-id': 'view-video'
            };
            
        beforeEach(function () {
            module('angularTemplateApp');

            clickMock = jasmine.createSpyObj('clickMock', ['click']);
            globalDelayTrackerMock = jasmine.createSpyObj('globalDelayTracker', ['add', 'remove']);

            spyOn(window.FileReader.prototype, 'readAsDataURL').and.callFake(function (file) {
                this.onload({
                    target: {
                        result: file.src
                    }
                });
            });

            spyOn(URL, 'createObjectURL').and.callFake(function () {
                return 'someurl';
            });

            module(function ($provide) {
                $provide.value('globalDelayTracker', globalDelayTrackerMock);
            });

            inject(function ($compile, $rootScope, $timeout, inputVideoDirective) {
                el = angular.element('<input-video></input-video>');

                timeout = $timeout;
                scope = $rootScope.$new();

                scope.ngModel = {};

                file = {
                    src: 'file.mov'
                };

                spyOn(el, 'find').and.callFake(function () {
                    return [
                        {
                            click: clickMock.click,
                            files: [file]
                        }
                    ]
                });

                inputVideoDirective[0].link(scope, el, attrs);
                scope.onSelect = function () {};

                spyOn(scope, 'onSelect');
            });
        });

        describe('captureVideo function', function () {
            it('should call a click on the file input element', function () {
                scope.captureVideo();

                expect(el.find).toHaveBeenCalledWith('input[type="file"]');
                expect(clickMock.click).toHaveBeenCalled();
            });
        });

        describe('onFileSelect function', function () {
            beforeEach(function () {
                spyOn(scope, 'getVideoSrc').and.callFake(function (file, callback) {
                    callback('file.mov');
                });
            });

            it('should not execute the body of the function if the file was not selected', function () {
                file = undefined;
                scope.onFileSelect();

                expect(scope.ngModel.fileObj).toBeFalsy();
                expect(scope.onSelect).not.toHaveBeenCalled();
            });

            it('should set the fileObj on the ngModel', function () {
                scope.onFileSelect();
                expect(scope.ngModel.fileObj).toEqual({
                    src: 'file.mov'
                });
            });

            it('should call add on the globalDelayTracker before getting the video source', function () {
                scope.onFileSelect();
                expect(globalDelayTrackerMock.add).toHaveBeenCalledWith('read-file');
                expect(scope.getVideoSrc).not.toHaveBeenCalled();

                timeout.flush();
                expect(scope.getVideoSrc).toHaveBeenCalled();
                expect(scope.getVideoSrc.calls.argsFor(0)[0]).toEqual(file);
            });

            it('should call remove on the globalDelayTracker after getting the video source', function () {
                scope.onFileSelect();
                expect(scope.getVideoSrc).not.toHaveBeenCalled();
                expect(globalDelayTrackerMock.remove).not.toHaveBeenCalled();

                timeout.flush();
                expect(scope.getVideoSrc).toHaveBeenCalled();
                expect(scope.getVideoSrc.calls.argsFor(0)[0]).toEqual(file);
                expect(globalDelayTrackerMock.remove).toHaveBeenCalledWith('read-file');
            });

            it('should call onSelect if defined', function () {
                scope.onFileSelect();
                timeout.flush();
                expect(scope.onSelect).toHaveBeenCalled();
            });
        });

        describe('getVideoSrc', function () {
            beforeEach(function () {
                spyOn(angular, 'element').and.callFake(function (selector) {
                    return [{'src':'','load': function(){}, 'data': function(){}}];
                });
            });

            afterEach(function () {
                angular.element.and.callThrough();
            });

            it('should create an object url with the passed file', function () {
                scope.getVideoSrc(file);

                expect(URL.createObjectURL).toHaveBeenCalledWith(file);
            });

            it('should call the callback with the result of reading the file as a data URL', function () {
                var callbackMock = jasmine.createSpyObj('callbackMock', ['callback']);
                
                scope.getVideoSrc(file, callbackMock.callback);

                expect(callbackMock.callback).toHaveBeenCalled();
            });
        });
    });
});