define(['app', 'angular'], function (app, angular) {

	app.directive('inputRange', function () {
		return {
			restrict: 'E',
			require: '^form',
			scope: {
				'formName': '@',  //required(used for custom validation)
				'label': '@',
				'name': '@',
				'ngDisabled': '=',
				'ngRequired': '=',
				'ngReadonly': '=',
				'min': '@',
				'max': '@',
				'ngModel': '=',
				'minLabel': '@',
				'maxLabel': '@',
				'step': '@'
			},
			link: function (scope, element, attrs, ngModelCtrl){
				scope.$watch('ngModel', function (newVal, oldVal) {
					if (newVal !== oldVal) {
						ngModelCtrl.$setDirty();
					}
				});
			},
			controller: function ($scope, $window, $element, $swipe, $timeout) {
				//CONSTANTS
				var THUMB_WIDTH = 20;
				var AVG_DIGIT_WIDTH = 8;

				var angularizedInputElement = $element.find("input");
				var inputElement = angularizedInputElement[0];
				//bottom, height, left, right, top, width
				var inputElementPosition = inputElement.getBoundingClientRect();
				var inSwipeEvent = false;
				var minVal = parseFloat($scope.min);
				var maxVal = parseFloat($scope.max);
				var range = maxVal - minVal;

				$scope.inputModel = {"value": $scope.ngModel || $scope.min};

				(function estimatedPositions() {
					var elementWidth = $element.width();
					var currentValueDisplacementLeft = (1 - (AVG_DIGIT_WIDTH * $scope.inputModel.value.length / elementWidth)) * (parseFloat($scope.inputModel.value) - minVal) / range;

					$scope.thumbDisplacement = {'left': 100 * (1 - (THUMB_WIDTH / elementWidth)) * (parseFloat($scope.inputModel.value) - minVal) / range + "%"};
					$scope.currentValueDisplacement = {'left': 100 * currentValueDisplacementLeft + "%"};

					$scope.hideMin = $scope.min.length * AVG_DIGIT_WIDTH / elementWidth > currentValueDisplacementLeft;
					$scope.hideMax = 1 - (AVG_DIGIT_WIDTH * ($scope.max.length + $scope.inputModel.value.length) / elementWidth) < currentValueDisplacementLeft;

				})();

				$scope.iOS = ( navigator.userAgent.match(/(iPad|iPhone|iPod)/g) ? true : false );
				$scope.liveLabel = {'inputValue': $scope.inputModel.value,
					'labelReadable': false,
					'valueReadable': false};


				$scope.updateNgModelAndSlider = function () {
					$timeout(function () {
						$scope.ngModel = $scope.inputModel.value;


						$scope.formName && $scope.$parent[$scope.formName][$scope.name].$setViewValue($scope.inputModel.value);
						
						if (!inSwipeEvent) {
							updateSliderWithModel();
						}
						$scope.liveLabel.labelReadable = false;
						$scope.liveLabel.valueReadable = true;
					});
				};

				$scope.$watch('ngModel', function (newVal, oldVal) {
					if (newVal !== oldVal) {
						$scope.inputModel.value = $scope.ngModel || $scope.min;
						updateSliderWithModel();
					}

					$scope.formName && $scope.$parent[$scope.formName][$scope.name].$setValidity($scope.name, newVal !== undefined && newVal !== null);

				});

				$scope.$watch('parseFloat(max) - parseFloat(min)', function (newVal, oldVal) {
					if (newVal !== oldVal) {
						range = newVal;
						$scope.$apply(function () {
							minVal = parseFloat($scope.min);
							maxVal = parseFloat($scope.max);
							updateSliderWithModel();
						});
					}

				});

				$scope.$watch(function () {
					return $element.find("input").is(':visible');
				}, function (isVisible) {
					if (isVisible) {
						$timeout(function () {
							inputElementPosition = inputElement.getBoundingClientRect();
							updateSliderWithModel();
						});
					}
				});

				$scope.$watch(function () {
					return inputElement.getBoundingClientRect().left;
				}, function (currentLeft, prevLeft) {
					if (currentLeft !== prevLeft) {
						$timeout(function () {
							inputElementPosition = inputElement.getBoundingClientRect();
							updateSliderWithModel();
						});
					}
				});

				angularizedInputElement.bind("focus", function () {
					$scope.liveLabel.inputValue = $scope.inputModel.value;
					$scope.liveLabel.labelReadable = true;
					$scope.liveLabel.valueReadable = true;
				});

				angularizedInputElement.bind("blur", function () {
					$scope.liveLabel.labelReadable = false;
					$scope.liveLabel.valueReadable = false;
				});


				$element.bind('resize', function () {
					$timeout(function () {
						inputElementPosition = inputElement.getBoundingClientRect();
						updateSliderWithModel();
					});
				});

				(function bindSwipe() {


					function updateModelAndSlider(withPos) {
						$timeout(function () {
							var percentLeft = (withPos.x - inputElementPosition.left) / inputElementPosition.width;
							if (percentLeft > 1) {
								percentLeft = 1;
							} else if (percentLeft < 0) {
								percentLeft = 0;
							}

							var currentValueWidth = elementWidth(".value");
							var currentValueDisplacement = (1 - currentValueWidth / inputElementPosition.width) * percentLeft;

							$scope.inputModel.value = (Math.round(range * percentLeft) + minVal).toString();
							$scope.updateNgModelAndSlider();

							$scope.thumbDisplacement.left = 100 * percentLeft*(1-(THUMB_WIDTH/inputElementPosition.width)) + "%";
							$scope.currentValueDisplacement.left = 100 * currentValueDisplacement + "%";


							$scope.hideMin = elementWidth(".min") / inputElementPosition.width > currentValueDisplacement;
							$scope.hideMax = 1 - ((elementWidth(".max") + currentValueWidth) / inputElementPosition.width) < currentValueDisplacement;
						});
					};

					function onStart(pos, event) {
						inSwipeEvent = true;
						updateModelAndSlider(pos);
					}

					function onEnd(pos, event) {
						inSwipeEvent = false;
						$scope.updateNgModelAndSlider();
					}

					function onMove(pos, event) {
						updateModelAndSlider(pos);
					}

					$swipe.bind(angularizedInputElement, {
						start: onStart,
						move: onMove,
						end: onEnd,
						cancel: onEnd
					});
				})();


				function updateSliderWithModel() {
					if (!inSwipeEvent && inputElementPosition.width !== 0) {

						var currentValueWidth = elementWidth(".value");
						var percentLeft = (parseFloat($scope.inputModel.value) - minVal) / range;
						var currentValueDisplacementLeft = (1 - currentValueWidth/inputElementPosition.width) * percentLeft;

						$scope.thumbDisplacement.left = 100 * percentLeft * (1-(THUMB_WIDTH/inputElementPosition.width)) + "%";
						$scope.currentValueDisplacement.left = 100 * currentValueDisplacementLeft + "%";

						$scope.hideMin = elementWidth(".min")/inputElementPosition.width > currentValueDisplacementLeft;
						$scope.hideMax = 1 - ((elementWidth(".max") + currentValueWidth) / inputElementPosition.width) < currentValueDisplacementLeft;

					}
				};

				function elementWidth(selector) {
					return selectElement(selector).offsetWidth;
				};

				function selectElement(selector) {
					return document.querySelector("[name='" + $scope.name + "'] " + selector);
				};


			},
			templateUrl: 'modules/ui-components/form/controls/simple/input-range/input-range_template.html'
		};
	});

});
