define(['InputPhoto'], function () {
    'use strict';

    describe('Input Photo Directive', function () {
        var scope,
            timeout,
            element,
            clickMock,
            globalDelayTrackerMock,
            file;
            
        beforeEach(function () {
            module('angularTemplateApp');

            clickMock = jasmine.createSpyObj('clickMock', ['click']);
            globalDelayTrackerMock = jasmine.createSpyObj('globalDelayTracker', ['add', 'remove']);

            spyOn(window.FileReader.prototype, 'readAsDataURL').and.callFake(function (file) {
                this.onload({
                    target: {
                        result: file.src
                    }
                });
            });

            module(function ($provide) {
                $provide.value('globalDelayTracker', globalDelayTrackerMock);
            });

            inject(function ($compile, $rootScope, $timeout, inputPhotoDirective) {
                element = angular.element('<input-photo></input-photo>');

                timeout = $timeout;
                scope = $rootScope.$new();

                scope.ngModel = {};

                file = {
                    src: 'file.jpg'
                };

                spyOn(element, 'find').and.callFake(function () {
                    return [
                        {
                            click: clickMock.click,
                            files: [file]
                        }
                    ]
                });

                inputPhotoDirective[0].link(scope, element);
                scope.onSelect = function () {};

                spyOn(scope, 'onSelect');
            });
        });

        describe('captureImage function', function () {
            it('should call a click on the file input element', function () {
                scope.captureImage();

                expect(element.find).toHaveBeenCalledWith('input[type="file"]');
                expect(clickMock.click).toHaveBeenCalled();
            });
        });

        describe('onFileSelect function', function () {
            beforeEach(function () {
                spyOn(scope, 'getImageSrc').and.callFake(function (file, callback) {
                    callback('file.jpg');
                });
            });

            it('should not execute the body of the function if the file was not selected', function () {
                file = undefined;
                scope.onFileSelect();

                expect(scope.ngModel.fileObj).toBeFalsy();
                expect(scope.ngModel.src).toBeFalsy();
                expect(scope.onSelect).not.toHaveBeenCalled();
            });

            it('should set the fileObj on the ngModel', function () {
                scope.onFileSelect();
                expect(scope.ngModel.fileObj).toEqual({
                    src: 'file.jpg'
                });
            });

            it('should call add on the globalDelayTracker before getting the image source', function () {
                scope.onFileSelect();
                expect(globalDelayTrackerMock.add).toHaveBeenCalledWith('read-file');
                expect(scope.getImageSrc).not.toHaveBeenCalled();

                timeout.flush();
                expect(scope.getImageSrc).toHaveBeenCalled();
                expect(scope.getImageSrc.calls.argsFor(0)[0]).toEqual(file);
            });

            it('should call remove on the globalDelayTracker after getting the image source', function () {
                scope.onFileSelect();
                expect(scope.getImageSrc).not.toHaveBeenCalled();
                expect(globalDelayTrackerMock.remove).not.toHaveBeenCalled();

                timeout.flush();
                expect(scope.getImageSrc).toHaveBeenCalled();
                expect(scope.getImageSrc.calls.argsFor(0)[0]).toEqual(file);
                expect(globalDelayTrackerMock.remove).toHaveBeenCalledWith('read-file');
            });

            it('should set the src of the ngModel to the the result of parsing the file', function () {
                scope.onFileSelect();
                timeout.flush();
                expect(scope.ngModel.src).toEqual('file.jpg');
            });

            it('should call onSelect if defined', function () {
                scope.onFileSelect();
                timeout.flush();
                expect(scope.onSelect).toHaveBeenCalled();
            });
        });

        describe('getImageSrc', function () {
            it('should read the file as a data URL', function () {
                scope.getImageSrc(file);

                expect(window.FileReader.prototype.readAsDataURL).toHaveBeenCalledWith(file);
            });

            it('should call the callback with the result of reading the file as a data URL', function () {
                var callbackMock = jasmine.createSpyObj('callbackMock', ['callback']);
                
                scope.getImageSrc(file, callbackMock.callback);

                expect(callbackMock.callback).toHaveBeenCalledWith('file.jpg');
            });
        });
    });
});