define(['angular', 'app', 'BaseCRUDService'], function(angular, app) {
	"use strict";
	app.service('CollectionCRUDService', function($rootScope, $http, $q, $parse, $filter, BaseCRUDService) {
		//expected use
		//fetch > save/delete
		//list will not be initialized properly if a save is done without a fetch

		var sort = function(list, sortComparator) {
			if (angular.isFunction(sortComparator)) {
				return list.sort(sortComparator);
			}
			else if (angular.isArray(sortComparator) && typeof list !== 'undefined' && list.length) {

				var isUndefined = angular.isUndefined;

				return list.sort(function(left, right) {
					var resolvedOrder = 0;
					for (var i = 0; i < sortComparator.length; i++) {
						var getter = $parse(sortComparator[i].key);
						var leftVal = getter(left);
						var rightVal = getter(right);

						if (angular.isString(leftVal) && angular.isString(rightVal)) {
							leftVal = leftVal.toLowerCase();
							rightVal = rightVal.toLowerCase();
						}

						if (leftVal === rightVal) {
							resolvedOrder = 0;
						}
						else {
							if (angular.isUndefined(leftVal) && angular.isDefined(rightVal)) {
								resolvedOrder = -1;
							}
							else if (angular.isDefined(leftVal) && angular.isUndefined(rightVal)) {
								resolvedOrder = 1;
							}
							else {
								resolvedOrder = leftVal > rightVal ? 1 : -1;
								resolvedOrder = sortComparator[i].reverse ? -1 * resolvedOrder : resolvedOrder;
							}
							break;
						}
					}
					return resolvedOrder;
				});
			}

			return list;
		};

		//Constructor
		var CollectionCRUDService = function(config) {

			var that = this;

			that.config = config;

			that.BaseCRUDService = new BaseCRUDService(config);

			that.dataPointers = {
				list: null,
				filters: null,
				link: null,
				unfilteredItemCount: null
			};

			that.compiledSorters = [];

			$rootScope.$on("$stateChangeSuccess", function(event, toState, toParams, fromState) {
				if (fromState.name === "" || toState.data.appletName !== fromState.data.appletName) {
					that.reset();
				}
			});
		};

		//Methods
		CollectionCRUDService.prototype = {};

		CollectionCRUDService.prototype.transformResponse = function(item) {
			return this.config.object.responseTransform ? this.config.object.responseTransform(item) : item;
		};

		CollectionCRUDService.prototype._onRequestComplete = function(promise) {
			var that = this;
			var defer = $q.defer();

			promise.then(function(success) {
				var response = success.data;
				var code = success.status;
				var method = success.config.method;

				if (response !== undefined && code != 204) {
					if (response['object-type'] === that.config.collection.objectType) {
						if (response[that.config.collection.name]) {
							response[that.config.collection.name].forEach(function(item) {
								that.transformResponse(item);
							});
						}
						else {
							response[that.config.collection.name] = [];
						}
					}
					else {
						response = that.transformResponse(response);
					}
				}
				else if (code == 204) {
					response = {};
					response[that.config.collection.name] = [];
					response.size = 0;
				}
				defer.resolve(response);
			}, function(error) {
				defer.reject(error.data);
			});

			return defer.promise;
		};

		CollectionCRUDService.prototype._deleteFromLocalList = function(item) {
			var list = this.dataPointers.list;
			var expectedId = this.idGetter(item);
			if (expectedId && list) {
				var deleteIndex = list.indexOf(this.localGetById(expectedId));
				if (deleteIndex !== -1) {
					list.splice(deleteIndex, 1);
				}
			}
		};

		CollectionCRUDService.prototype.idGetter = function(item){
			return this.BaseCRUDService.idGetter(item);
		};

		CollectionCRUDService.prototype.localGetById = function(id) {
			var that = this;
			var list = that.dataPointers.list;
			var listLength = list ? that.dataPointers.list.length : 0;

			for (var i = 0; i < listLength; i++) {
				if (angular.equals(id, that.idGetter(list[i]))) {
					return list[i];
				}
			}
		};

		CollectionCRUDService.prototype.createEmpty = function() {
			return this.BaseCRUDService.createEmpty();
		};


		CollectionCRUDService.prototype.fetch = function(queryParams) {
			var that = this;
			var requestPromise = that._onRequestComplete(that.BaseCRUDService.fetch(queryParams));
			requestPromise.then(function(response) {
				that.dataPointers.filters = angular.copy(queryParams);
				that.dataPointers.list = sort(response[that.config.collection.name], that.config.collection.sortComparator);

				that.dataPointers.link = response.link;
				if (angular.isDefined(response.unfilteredCount)) {
					that.dataPointers.unfilteredItemCount = response.unfilteredCount;
				}
			});
			return requestPromise;
		};

		CollectionCRUDService.prototype.save = function(item) {
			var that = this;
			var requestPromise = that._onRequestComplete(that.BaseCRUDService.save(angular.copy(item)));

			requestPromise.then(function(responseItem) {
				if (that.dataPointers.list) {
					that._deleteFromLocalList(item);
					var scopedList = angular.copy(that.dataPointers.list);
					scopedList.push(responseItem);
					scopedList = that.config.collection.filter ? that.config.collection.filter(scopedList, that.dataPointers.filters) : scopedList;
					that.dataPointers.list = sort(scopedList, that.config.collection.sortComparator);
					if (!item.link && that.dataPointers.unfilteredItemCount !== null) {
						that.dataPointers.unfilteredItemCount++;
					}
				}
			});
			return requestPromise;
		};

		CollectionCRUDService.prototype.delete = function(item) {
			var that = this;

			var requestPromise = that.BaseCRUDService.delete(item);
			requestPromise.then(function() {
				that._deleteFromLocalList(item);
				if (that.dataPointers.unfilteredItemCount !== null && that.dataPointers.unfilteredItemCount > 0) {
					that.dataPointers.unfilteredItemCount--;
				}
			});
			return requestPromise;
		};

		CollectionCRUDService.prototype.reset = function() {
			this.dataPointers.list = null;
			this.dataPointers.filters = null;
			this.dataPointers.link = null;
			this.dataPointers.unfilteredItemCount = null;
		};

		return CollectionCRUDService;
	});
});