define(['angular', 'app', 'lodash', 'formatter'], function (angular, app, _) {
	"use strict";
	app.service('BaseCRUDService', function ($http, $q, $parse) {

		//Constructor
		var BaseCRUDService = function(config) {

			this.config = config;
			
			this.requests = {
				change : [],
				retrieve : []
			};

			this.idGetter = $parse(this.config.object.uidKey);

		};

		//Methods
		BaseCRUDService.prototype = {};

		BaseCRUDService.prototype.transformRequest = function(item) {
			return this.config.object.requestTransform ? this.config.object.requestTransform(item) : item;
		};

		BaseCRUDService.prototype._makeRequest = function (httpConfig) {
			var that = this;
			var defer = $q.defer();

			function sendRequest(httpConfig) {
				if(httpConfig.method === "POST" || httpConfig.method === "PUT") {
					httpConfig.data = that.transformRequest(httpConfig.data);
					delete httpConfig.data.link;
				} else if(httpConfig.method === "GET") {
				    httpConfig.params = that.transformRequest(httpConfig.params);
				}

				var waitForIt = httpConfig.method === "GET" ? that.requests.change : that.requests.retrieve;

				$q.all(waitForIt).finally(function(){
					var resultsSet = null;
					var sendRequest = function(httpConfig) {
						var request = $http(httpConfig).success(function(data, status, headers, config, statusText) {
							// Notify success to the serivce that needs to work with data processed post-success
							//  Example: A service should implement this method if it needs to update its list after
							//  a change has been made and a re-query is not desired or necessary
							if (that.config.object.requestSuccess) {
								that.config.object.requestSuccess(httpConfig.data, httpConfig.method);
							}

							// Next check if the request returned a results set of maximum length and has additional results
							//  if so, re-invoke the get function and append the results set
							if(!resultsSet) resultsSet = data;
							else{
								resultsSet.entry = resultsSet.entry.concat(data.entry);
								resultsSet.total += data.total;
							}

							var hasMoreItems = -1;
							if(typeof data.link == 'object' && data.link.length) {
								hasMoreItems = _.findIndex(data.link, function(item) {
									return item.relation == 'next';
								});
							}
							if(hasMoreItems >= 0){
								var nextQueryUrl = data.link[hasMoreItems].url;
								httpConfig.url = httpConfig.url.substring(0, httpConfig.url.lastIndexOf('/') + 1);
								httpConfig.url += nextQueryUrl.substring(nextQueryUrl.indexOf('?'));
								sendRequest(httpConfig);
							}
							else {
								defer.resolve({
									data: resultsSet,
									status: status,
									headers: headers,
									config: config,
									statusText: statusText
								});
							}
						}).error(function(data, status, headers, config, statusText) {
							defer.reject({
								data: data,
								status: status,
								headers: headers,
								config: config,
								statusText: statusText
							});
						}).finally(function() {
							waitForIt.splice(waitForIt.indexOf(request), 1);
						});

						if (httpConfig.method === "GET") {
							that.requests.retrieve.push(request);
						}
						else {
							that.requests.change.push(request);
						}
					};

					sendRequest(httpConfig);
				});
			}

			if (httpConfig.url) {
				sendRequest(httpConfig);
			} else {
				that.config.dependency.service.fetch().then(
					function (resource) {
						var linkTitle = that.config.dependency.linkTitles[httpConfig.method] || that.config.dependency.linkTitles["GET"];
						httpConfig.url = resource.link ? resource.link[linkTitle] : resource[linkTitle];
						httpConfig.url += httpConfig.pathParamString || "";
						sendRequest(httpConfig);
					});
			}

			return defer.promise;
		};

		//Requires Override
		BaseCRUDService.prototype.createEmpty = function() {
			return {};
		};

		BaseCRUDService.prototype.fetch = function (queryParams, preventLocalUpdate) {
			return this._makeRequest({method: "GET", params: queryParams});		
		};

		BaseCRUDService.prototype.save = function (item) {
			var method = !item.link && !(item.id && this.config.dependency.linkTitles.PUT !== "none") ? "POST" : "PUT";

			return this._makeRequest({method: method, data: item,
				url : (method === "PUT" ?
					(item.link ? item.link[0].href : //for legacy shared services
						(this.config.dependency.service[this.config.object.key] + "/" + item.id) //for FHIR-related PUT
				) : (!item.link ? this.config.dependency.service[this.config.object.key] : null))}); //for FHIR-related POST
		};

		BaseCRUDService.prototype.delete = function (item) {
			if(this.idGetter(item) || item.id) {
				return this._makeRequest({method: "DELETE",
					url : (item.link ? item.link[0].href : //for legacy shared services
						(this.config.dependency.service[this.config.object.key] + "/" + item.id || null) ) //for FHIR-related items
				});
			} else if(item.pathParamString || item.queryParams) {
				return this._makeRequest({method: "DELETE", params: item.queryParams, pathParamString: item.pathParamString});
			} else {
				var defer = $q.defer();
				defer.reject();
				return defer.promise;
			}

		};

		return BaseCRUDService;
	});
});