define(['angular', 'app', 'localResourceDirectoryService', 'authentication_service'], function (angular, app) {
	"use strict";
	app.service('fhirPatient', function ($http, $q, $timeout, localResourceDirectoryService, authenticationService, patient) {

		var fhirPatientResources = {};
		var fhirPatient = {};

		// Fetch FHIR patient information based on the logged in user
		fhirPatientResources.fetch = function () {
			var deferred = $q.defer();

			patient.fetch().then(function(patient){
				resolvePatientByUserService(patient).then(function(response){
					deferred.resolve(response);
				});
			});

			return deferred.promise;
        };
        
        fhirPatientResources.update = function (updatedFhirPatient) {
            var deferred = $q.defer();

            localResourceDirectoryService.fetch().then(function (resources) {
                $http({
                    method: 'PUT',
                    url: resources['patient'] + '/' + updatedFhirPatient.id,
                    data: JSON.stringify(updatedFhirPatient)
                }).success(function () {
                    deferred.resolve();
                }).error(function (error) {
                    deferred.reject(error);
                });
            });

            return deferred.promise;
        };

		// Resolve patient by internal FHIR ID reference, retries up to 5 times, once per second to wait for elastic indexing
		// Parameters:
		//  -reference: Patient/<FHIR unique id> or <FHIR unique id>
		var resolvePatientByReference = function (reference) {
			var deferred = $q.defer();
			localResourceDirectoryService.fetch().then(function (resources) {
                var refId = reference.indexOf('/') >= 0 ? reference.split("/")[1] : reference;

                var retryCount = 0;

                var requestPatient = function(patient){
                    $http.get(resources['patient'] + '/' + refId).success(function(response){
                        if(!response.id){
                            if(retryCount < 5){
                                retryCount++;
                                $timeout(function(){ 
                                    requestPatient(patient) 
                                }, 1000);
                            } else{
                                deferred.reject("Expected patient not found");
                                deferred = null;
                            }
                        } else{
                            deferred.resolve(response);
                        }
                    }).error(function (error) {
                        deferred.reject(error);
                        deferred = null;
                    });
                };

                requestPatient(reference);
            });

			return deferred.promise;
		};

		// Resolve patient by internal VA identifier
		// If the FHIR-based patient record does not exist, it will be created using the necessary data points using known JWT payload
		// Parameters:
		//  -patient: response from User Service's patient object (in NextGen at /users/v1/session/)
		var resolvePatientByUserService = function (patient){
			//determine unique ID and assigning authority based on available fields
			var assigningAuthority, uniqueId;
			if(patient['icn']){
				assigningAuthority = 'ICN';
				uniqueId = patient['icn'];
			} else if(patient['edipid']){
				assigningAuthority = 'EDIPI';
				uniqueId = patient['edipid'];
			}

			var lastName = patient.lastName;
			var firstName = patient.firstName;

			return requestFhirPatient(assigningAuthority, uniqueId, lastName, firstName);
		};

		function requestFhirPatient(assigningAuthority, uniqueId, lastName, firstName){
			var deferred = $q.defer();
			localResourceDirectoryService.fetch().then(function (resources) {
                $http.get(resources['patient'] + '?identifier=' + uniqueId).success(function (patientResponse) {
                    if(!patientResponse.total){
                        //if no item returned then create a new reference with the supplied parameters and return that
                        var baseUrl = resources['patient'];
                        var system = resources['pgd-identifier-system'];
                        createFhirPatient(baseUrl, system, assigningAuthority, uniqueId, lastName, firstName).success(function(response, staus, headers){
                            //get response header's returned location
                            var patientResource = headers()['content-location'];
                            patientResource = patientResource.substring(patientResource.indexOf("Patient"));

                            //now resolve patient by the content location
                            patientResource = patientResource.split('/');
                            resolvePatientByReference(patientResource[0] + '/' + patientResource[1]).then(function(response){
                                deferred.resolve(response);
                            })
                        });
                    } else {
                        deferred.resolve(patientResponse.entry[0].resource);
                    }
                }).error(function (error) {
                    deferred.reject(error);
                    deferred = null;
                });
            });

			return deferred.promise;
		}

		var createFhirPatient = function (baseUrl, system, assigningAuthority, uniqueId, lastName, firstName) {
			return $http({
				method: 'POST',
				url: baseUrl,
				data: {
					"resourceType": "Patient",
					"identifier": [{
						"system": system,
						"type": {
							"coding": [{
								"code": assigningAuthority
							}]
						},
						"value": uniqueId
					}],
					"name": [{
						"use": "usual",
						"family": [ lastName ],
						"given": [ firstName ]
					}]
				}
			});
		};

		return fhirPatientResources;
	});
});