define(['fhirPatientService'], function () {
	'use strict';

	describe('FHIR Patient Service', function () {
		var $scope,
			$httpBackend,
			$q,
			$filter,
			resourceJson,
			authenticationServiceMock,
			localResourceDirectoryMock,
			patientServiceMock,
			patientMock,
			service;

		beforeEach(function () {
			module('angularTemplateApp');

			patientMock = {
				firstName: 'John',
				lastName: 'Doe',
				dateOfBirth: '01/01/1980',
				icn: '1234'
			};

			resourceJson = {
				"patient": "/pgd-fhir-services/fhir/Patient",
				"pgd-identifier-system": "urn:uuid:2.16.840.1.113883.4.349"
			};

			authenticationServiceMock = jasmine.createSpyObj('authenticationService', ['checkAuthStatus']);
			localResourceDirectoryMock = jasmine.createSpyObj('localResourceDirectoryService', ['fetch']);
			patientServiceMock = jasmine.createSpyObj('patient', ['fetch']);

			module(function ($provide) {
				$provide.value('patient', patientServiceMock);
				$provide.value('authenticationService', authenticationServiceMock);
				$provide.value('localResourceDirectoryService', localResourceDirectoryMock);
			});

			inject(function ($rootScope, _$httpBackend_, _$q_, fhirPatient) {
				$scope = $rootScope;
				$httpBackend = _$httpBackend_;
				$q = _$q_;
				service = fhirPatient;

				var patientDeferred = $q.defer();
				patientDeferred.resolve(patientMock);
				patientServiceMock.fetch.and.returnValue(patientDeferred.promise);

				var authDeferred = $q.defer();
				authDeferred.resolve(true);
				authenticationServiceMock.checkAuthStatus.and.returnValue(authDeferred.promise);

				var resourceDeferred = $q.defer();
				resourceDeferred.resolve(resourceJson);
				localResourceDirectoryMock.fetch.and.returnValue(resourceDeferred.promise);
			});
		});

		afterEach(function () {
			$httpBackend.verifyNoOutstandingExpectation();
			$httpBackend.verifyNoOutstandingRequest();
		});

		it('should return an existing patient', function () {
			$httpBackend.whenGET(resourceJson.patient + '?identifier=' + patientMock.icn)
				.respond(200, {
					total: 1,
					entry: [{
						resource: {
							id: 333
						}
					}]
				});

			service.fetch().then(function (response) {
				expect(response.id).toEqual(333);
			});
			$httpBackend.flush();
			$scope.$apply();

			expect(localResourceDirectoryMock.fetch).toHaveBeenCalled();
		});

		it('should create a FHIR patient using the system from the config', function () {
			$httpBackend.whenGET(resourceJson.patient + '?identifier=' + patientMock.icn)
				.respond(200, { total: 0 });

			$httpBackend.expectPOST(resourceJson.patient, {
				"resourceType": "Patient",
				"identifier": [{
					"system": "urn:uuid:2.16.840.1.113883.4.349",
					"type": {
						"coding": [{
							"code": "ICN"
						}]
					},
					"value": "1234"
				}],
				"name": [{
					"use": "usual",
					"family": ["Doe"],
					"given": ["John"]
				}]
			}).respond(201, 'test success', { 'content-location': resourceJson.patient + '/333' });

			$httpBackend.whenGET(resourceJson.patient + '/333')
				.respond(200, { id: 333 });

			service.fetch().then(function (response) {
				expect(response.id).toEqual(333);
			});
			$httpBackend.flush();
			$scope.$apply();

			expect(localResourceDirectoryMock.fetch).toHaveBeenCalled();
		});

		describe("update function", function () {
			it('should update the fhirPatient with the given object', function () {
				var data;

				$httpBackend.expectPUT(resourceJson.patient + '/444')
					.respond(200);

				service.update({
					id: 444,
					name: {
						family: [ 'userLast01' ],
						given: [ 'userFirst01' ],
						use: 'usual'
					}
				}).then(function () {
					data = 'success!';
				});
				$httpBackend.flush();

				expect(localResourceDirectoryMock.fetch).toHaveBeenCalled();
				expect(data).toEqual('success!');
			});

			it('should throw en error when the update call is unsuccessful', function () {
				var data;

				$httpBackend.expectPUT(resourceJson.patient + '/444')
					.respond(400, 'error!');

				service.update({
					id: 444,
					name: {
						family: [ 'userLast01' ],
						given: [ 'userFirst01' ],
						use: 'usual'
					}
				}).then(function () {
					//empty success block
				}, function (response) {
					data = response;
				});
				$httpBackend.flush();

				expect(localResourceDirectoryMock.fetch).toHaveBeenCalled();
				expect(data).toEqual('error!');
			});
		});
	});
});