define(['httpInterceptor'], function () {
    'use strict';

    describe('HTTP Interceptor', function () {
        var httpInterceptor,
            authenticationServiceMock,
            connectionErrorServiceMock,
            configMock,
            rejectionMock;

        var getEpochTime = function (minutesToAdd) {
            minutesToAdd = (typeof minutesToAdd === 'number') ? minutesToAdd : 0;

            return Math.round(((new Date()).getTime() + (minutesToAdd * 60000)) / 1000);
        };
        
        beforeEach(function () {
            module('angularTemplateApp');

            authenticationServiceMock = jasmine.createSpyObj('authenticationService', ['parseJwt', 'getJwtFromCookie', 'authenticate', 'isAuthenticated']);
            connectionErrorServiceMock = jasmine.createSpyObj('connectionErrorService', ['showAuthorizationErrorMsg', 'showServerErrorMsg']);
            configMock = {
                vamf_user: {
                    COOKIE: 'vamfjwtv1'
                }
            };
            rejectionMock = {
                status: 401
            };

            module(function ($provide) {
                $provide.value('authenticationService', authenticationServiceMock);
                $provide.value('connectionErrorService', connectionErrorServiceMock);
                $provide.value('config', configMock);
            });

            inject(function(_httpInterceptor_) {
				httpInterceptor = _httpInterceptor_;
			});
        });

        describe('when an HTTP 401 error is returned', function () {
            it ('should redirect to login if no JWT present', function () {
                var emptyJwt = '';
                authenticationServiceMock.parseJwt.and.returnValue(emptyJwt);

                httpInterceptor.responseError(rejectionMock);

                expect(authenticationServiceMock.authenticate).toHaveBeenCalled();
            });

            it ('should redirect to login if an unauthenticated JWT is present', function () {
                var unauthenticatedJwt = {
                    "authenticated": false,
                    "sub": null,
                    "nbf": getEpochTime(),
                    "iss": "gov.va.vamf.userservice.v1",
                    "exp": getEpochTime(),
                    "jti": "51aaae1e-ceb2-414a-94fe-d2fac7c5c811"
                };
                authenticationServiceMock.parseJwt.and.returnValue(unauthenticatedJwt);

                httpInterceptor.responseError(rejectionMock);

                expect(authenticationServiceMock.authenticate).toHaveBeenCalled();
            });

            it ('should show an authorization error message if a valid JWT is present', function () {
                var validJwt = {
                    "lastName": "userLast01",
                    "sub": "1006088937V099668",
                    "authenticated": true,
                    "authenticationAuthority": "gov.va.iam.ssoe.v1",
                    "idType": "ICN",
                    "iss": "gov.va.vamf.userservice.v1",
                    "vamf.auth.resources": [
                      "^.*(\\/)?patient[s]?\\/(ICN\\/)?1006088937V099668\\/.*$",
                      "^.*(\\/)?patient[s]?\\/EDIPI\\/1113138327\\/.*$"
                    ],
                    "firstName": "userFirst01",
                    "nbf": getEpochTime(),
                    "patient": {
                      "firstName": "userFirst01",
                      "lastName": "userLast01",
                      "middleName": "",
                      "dateOfBirth": "1960-12-01",
                      "gender": "MALE",
                      "ssn": "999999901",
                      "icn": "1006088937V099668",
                      "edipid": "1113138327",
                      "dob": "1960-12-01"
                    },
                    "middleName": "",
                    "attributes": {
                      "va_eauth_csid": "DSLogon",
                      "eulaAccepted": "true"
                    },
                    "vamf.auth.roles": [
                      "veteran"
                    ],
                    "rightOfAccessAccepted": true,
                    "exp": getEpochTime(15),
                    "jti": "25b18d7a-f66f-4bcf-aba0-741d71db79b7",
                    "loa": 2
                };
                authenticationServiceMock.parseJwt.and.returnValue(validJwt);

                httpInterceptor.responseError(rejectionMock);

                expect(connectionErrorServiceMock.showAuthorizationErrorMsg).toHaveBeenCalled();
            });
        });
    });
});