define(['angular', 'app'], function (angular, app) {
	"use strict";
	app.service('questionnaireResponseService', function ($q, fhirResources, localResourceDirectoryService, fhirPatient, questionnaireService) {

		return {
			resourceLinks: {},
			currQuestionnaire: {},
			currResponse: {},
			currEvaluation: {},
			currPatient: {},
			questionnaireMap: {},
			questionnaireType:'',

			getEvaluation: function () {
				return this.currEvaluation;
			},
			setEvaluation: function (evaluation) {
				this.currEvaluation = evaluation;
			},
			getCurrResponse: function () {
				return this.currResponse;
			},
			setCurrResponse: function (response) {
				this.currResponse = response;
			},
			getPatient: function () {
				return this.currPatient;
			},
			getLinkId: function (text) {
				return this.questionnaireMap[text] && this.questionnaireMap[text][0] ? this.questionnaireMap[text][0] : '';
			},
			getLinkText: function (linkId) {
				var linkTextToReturn;

				for (var currentLinkText in this.questionnaireMap) {
					if (_.contains(this.questionnaireMap[currentLinkText], linkId)) {
						linkTextToReturn = currentLinkText;

						break;
					}
				}

				return linkTextToReturn;
			},
			getQuestionnaireMap: function () {
				return this.questionnaireMap;
			},
			initQuestionnaireMap: function (type) {
				if ((type === 'MyTelederm' && typeof(this.questionnaireMap.MYTELEDERM_QUESTIONNAIRE) === 'undefined')
					|| (type === 'MyVideo' && typeof(this.questionnaireMap.MYVIDEO_QUESTIONNAIRE) === 'undefined')) {
					this.questionnaireMap = {};
					this.parseQuestionnaire(this.currQuestionnaire.resource.group);
				}
				return this.questionnaireMap;
			},
			parseQuestionnaire: function (jsonObject) {
				if (jsonObject && jsonObject.linkId && (jsonObject.text)) {
					this.questionnaireMap[jsonObject.text] = this.questionnaireMap[jsonObject.text] || [];
					this.questionnaireMap[jsonObject.text].push(jsonObject.linkId);
				}
				if (typeof(jsonObject) === 'object' && typeof(jsonObject.group) !== 'undefined') {
					for (var i = 0; i < jsonObject.group.length; i++) {
						this.parseQuestionnaire(jsonObject.group[i]);
					}
				}
				if (typeof(jsonObject) === 'object' && typeof(jsonObject.question) !== 'undefined') {
					for (var i = 0; i < jsonObject.question.length; i++) {
						this.parseQuestionnaire(jsonObject.question[i]);
					}
				}
				if (typeof(jsonObject) === 'object' && typeof(jsonObject.length) !== 'undefined') {
					for (var i = 0; i < jsonObject.length; i++) {
						this.parseQuestionnaire(jsonObject[i]);
					}
				}
			},
			createResponse: function (responseType, mediaRequestId) {

				var response = {
					resourceType: 'QuestionnaireResponse',
					status: 'in-progress',
					identifier: {value: mediaRequestId},
					questionnaire: {reference: 'Questionnaire/' + this.currQuestionnaire.resource.id},
					subject: {reference: 'Patient/' + this.currPatient.id},
					group: {
						title: responseType + ' Questionnaire Response',
						group: []
					}
				};

				return response;
			},
			createAnswer: function (linkId, answerType, value, index){
				var answer = {};
				switch(answerType) {
					case "string": 	{ answer = {valueString : value}; break; }
					case "int": 	{ answer = {valueInteger : parseInt(value)}; break;}
					case "bool": 	{ answer = {valueBoolean : value}; break;}
					case "choice": 	{ answer = {valueCoding : {code: index.toString(), display: value, system : fhirResources.getSystem()}}; break;}
				}

				return {
					linkId : linkId,
					answer : [answer]
				};
			},
			getValue :  function(jsonObject, linkId) {
				var that = this;
				var answer = _.find(jsonObject, {linkId: linkId});

				if(answer)
				{
					if(answer.answer[0].hasOwnProperty("valueString"))
						return answer.answer[0].valueString;
					else if(answer.answer[0].hasOwnProperty("valueInteger"))
						return answer.answer[0].valueInteger;
					else if(answer.answer[0].hasOwnProperty("valueBoolean"))
						return answer.answer[0].valueBoolean;
					else if(answer.answer[0].hasOwnProperty("valueCoding"))
						return answer.answer[0].valueCoding.display;
				}
				else if (typeof(jsonObject) === 'object' && typeof(jsonObject.group) !== 'undefined') {
					return that.getValue(jsonObject.group, linkId);
				}
				else if (typeof(jsonObject) === 'object' && typeof(jsonObject.question) !== 'undefined') {
					return that.getValue(jsonObject.question, linkId);
				}
				else if (typeof(jsonObject) === 'object' && typeof(jsonObject.length) !== 'undefined') {
					for(var i=0; i < jsonObject.length; i++) {
						var val = that.getValue(jsonObject[i], linkId);
						if(val) return val;
					}
				}
				else
					return null;

			},
			findElement: function (jsonObject, linkId) {
				var that = this;
				var obj = _.find(jsonObject, {linkId: linkId});

				if (obj)
					return obj;
				else if (typeof(jsonObject) === 'object' && typeof(jsonObject.group) !== 'undefined') {
					return that.findElement(jsonObject.group, linkId);
				}
				else if (typeof(jsonObject) === 'object' && typeof(jsonObject.question) !== 'undefined') {
					return that.findElement(jsonObject.question, linkId);
				}
				else if (typeof(jsonObject) === 'object' && typeof(jsonObject.length) !== 'undefined') {
					for (var i = 0; i < jsonObject.length; i++) {
						var elem = that.findElement(jsonObject[i], linkId);
						if (elem) return elem;
					}
				}
				else
					return null;

			},
			initEvaluation: function (questionnaireType, mediaRequestId, questionnaireResponseId) {
				this.questionnaireType = questionnaireType;
				var deferred = $q.defer();
				var that = this;

				fhirPatient.fetch().then(function (patient) {
					that.currPatient = patient;
					var idStr = questionnaireResponseId ? "/" + questionnaireResponseId : '?source=Patient/' + patient.id;

					$q.all({
						resourceLinks: localResourceDirectoryService.fetch(),
						questionnaire: questionnaireService.getQuestionnaire(that.questionnaireType),
						questionnaireResponse: fhirResources.resolveFhirResourceByReference("QuestionnaireResponse" + idStr)
					}).then(function (responses) {
						that.resourceLinks = responses.resourceLinks;
						that.currQuestionnaire = responses.questionnaire;
						that.currResponse = (responses.questionnaireResponse && responses.questionnaireResponse.resourceType == 'QuestionnaireResponse') ? responses.questionnaireResponse : that.createResponse(that.questionnaireType, mediaRequestId);

						that.initQuestionnaireMap(that.questionnaireType);

						that.currEvaluation = that.transformResponse(that.currResponse);
						deferred.resolve();
					});
				});

				return deferred.promise;
			},
			saveEvaluation: function (pageName) {
				var deferred = $q.defer();
				var that = this;
				this.updateResponse(pageName);

				var method = this.currResponse.id ? "PUT" : "POST";
				var resourceName = this.currResponse.id ? 'QuestionnaireResponse/' + this.currResponse.id : 'QuestionnaireResponse';


				if (this.currEvaluation.status === 'NOT_STARTED') {
					this.currEvaluation.status = 'STARTED';
					this.currResponse.status = 'in-progress';
				}

				if(this.currEvaluation.status === 'COMPLETED'){
					this.currResponse.status = 'completed';
				}

				var resourceConfig = {
					method: method,
					resourceName: resourceName,
					data: this.currResponse
				};

				fhirResources.createUpdateFhirResource(resourceConfig).then(function (response) {
					var questionnaireResponseId = response.split('/')[1];
					that.currResponse.id = questionnaireResponseId;
					deferred.resolve(questionnaireResponseId);
				}, function (error) {
					deferred.reject(error);
				});

				return deferred.promise;
			},
			deleteMedia: function(mediaId){
				var deferred = fhirResources.deleteFhirResourceByReference("Media/"+ mediaId);
				return deferred.promise;
			}
		}
	});
});
