define(['questionnaireResponseService'], function () {
    'use strict';

    describe('questionnaire Response Service', function () {
        var service,
            httpMock,
            localResourceDirectoryServiceMock,
			fhirResourcesMock,
			fhirPatientMock,
			questionnaireServiceMock,
			medTypes = ['TOPICAL', 'ORAL', 'INJECTED'],
			$q,
			$scope;

        beforeEach(function () {
            module('angularTemplateApp');

            localResourceDirectoryServiceMock = jasmine.createSpyObj('localResourceDirectoryServiceMock', ['fetch']);
			fhirResourcesMock = jasmine.createSpyObj('fhirResourcesMock', ['resolveFhirResourceByReference','createUpdateFhirResource', 'deleteFhirResourceByReference','getSystem']);
			fhirPatientMock = jasmine.createSpyObj('fhirPatientMock', ['fetch']);
			questionnaireServiceMock = jasmine.createSpyObj('questionnaireServiceMock', ['fetch','getQuestionnaire']);


			localResourceDirectoryServiceMock.fetch.and.returnValue({
                then: function (callback) {
                    callback({
                        'image-sections': '/media-request-service/v1/media/sections',
                        'update-request': '/media-request-service/v1/patients/{UNIQUE_ID}/requests/{MEDIA_REQUEST_ID}',
                    });
                }
            });

			fhirPatientMock.fetch.and.returnValue({
				then: function (callback) {
					callback({
						"id" : "patientId"
					});
				}
			});

			questionnaireServiceMock.getQuestionnaire.and.returnValue({
				then: function (callback) {
					callback({
						"resource": {
							"resourceType": "Questionnaire",
							"id": "questionnaireId",
							"group": {
								"text": "MYTELEDERM_QUESTIONNAIRE",
								"linkId": "IMAGE",
								"group": [
									{"linkId": "1", "text": "SKIN_STATUS"},
									{"linkId": "2", "text": "TOPICAL_MEDICATIONS"},
									{"linkId": "3", "text": "ORAL_MEDICATIONS"},
									{"linkId": "8", "text": "MEDIA"}
								]
							}
						}
					});
				}
			});

			fhirResourcesMock.resolveFhirResourceByReference.and.returnValue({
				then: function (callback) {
					callback({});
				}
			});

			fhirResourcesMock.createUpdateFhirResource.and.returnValue({
				then: function (callback) {
					callback("QuestionnaireResponse/id1");
				}
			});

			fhirResourcesMock.getSystem.and.returnValue('https://URL/display/PGDMS/Client+Provenance+Mapping');


			module(function ($provide) {
                $provide.value('localResourceDirectoryService', localResourceDirectoryServiceMock);
                $provide.value('fhirResources', fhirResourcesMock);
                $provide.value('fhirPatient', fhirPatientMock);
                $provide.value('questionnaireService', questionnaireServiceMock);
                $provide.value('medicationTypes', medTypes);
			});

            inject(function ($rootScope, _$q_, imageResponseService, $httpBackend) {
                httpMock = $httpBackend;
                $q = _$q_;
				$scope = $rootScope.$new();
                service = imageResponseService;

				var fhirResourcesDeferred = $q.defer();
				fhirResourcesDeferred.resolve();
				fhirResourcesMock.deleteFhirResourceByReference.and.returnValue(fhirResourcesDeferred.promise);
			});
        });

		describe('getEvaluation and setEvaluation', function () {

			it('should return a default evaluation if none is set', function () {
				expect(service.getEvaluation()).toEqual({});
			});

			it('should allow the user to set an evaluation and retrieve it', function () {
				service.setEvaluation({
					status: 'STARTED'
				});

				expect(service.getEvaluation()).toEqual({
					status: 'STARTED'
				});
			});

			it('should overwrite values', function () {
				service.setEvaluation({
					status: "STARTED",
					skinStatus: "BETTER",
					skinStatusComment: "Some comment"
				});

				expect(service.getEvaluation()).toEqual({
					status: "STARTED",
					skinStatus: "BETTER",
					skinStatusComment: "Some comment"
				});

				service.setEvaluation({
					status: "STARTED",
					skinStatus: "BETTER",
					skinStatusComment: "Some comment",
					medicationTypes: [
						'TOPICAL',
						'OTHER'
					],
					otherMedication: "Other medication comments",
					additionalComments: "Some comments",
					medications: [
						{
							type: "TOPICAL"
						}
					],
					images: [
						{
							imageType: "HEAD_FACE"
						}
					],
					pageNumber: 1
				});

				expect(service.getEvaluation()).toEqual({
					"status": "STARTED",
					"skinStatus": "BETTER",
					"skinStatusComment": "Some comment",
					"medicationTypes": [
						'TOPICAL',
						'OTHER'
					],
					"otherMedication": "Other medication comments",
					"additionalComments": "Some comments",
					"medications": [
						{
							type: "TOPICAL"
						}
					],
					"images": [
						{
							imageType: "HEAD_FACE"
						}
					],
					"pageNumber": 1
				});
			});
		});

		describe('getLinkId and getLinkText', function () {
			beforeEach(function () {
				service.currQuestionnaire = {
					"resource": {
						"group": {
							"text": "MYTELEDERM_QUESTIONNAIRE",
							"linkId": "IMAGE",
							"group": [{"linkId": "1", "text": "SKIN_STATUS"}]
						}
					}
				};
				service.initQuestionnaireMap("MyTelederm");
			});

			it('should return the linkId when link text is passed', function () {
				var linkId = service.getLinkId("SKIN_STATUS");
				expect(linkId).toEqual("1");
			});

			it('should return the link text when linkId is passed', function () {
				var linkText = service.getLinkText("1");
				expect(linkText).toEqual("SKIN_STATUS");
			});
		});

		describe('initQuestionnaireMap and parseQuestionnaire', function () {

			it('should return an array of key value pairs of linkId and text for MyTelederm', function () {
				service.currQuestionnaire = {
					"resource": {
						"group": {
							"text": "MYTELEDERM_QUESTIONNAIRE",
							"linkId": "IMAGE",
							"group": [
								{"linkId": "1", "text": "SKIN_STATUS"},
								{"linkId": "2", "text": "TOPICAL_MEDICATIONS"},
								{"linkId": "8", "text": "MEDIA"}
							]
						}
					}
				};
				service.initQuestionnaireMap("MyTelederm");
				var map = service.getQuestionnaireMap();
				expect(map).toEqual({"MYTELEDERM_QUESTIONNAIRE": ["IMAGE"], "SKIN_STATUS": ["1"], "TOPICAL_MEDICATIONS":["2"], "MEDIA":["8"]});
			});

			it('should return an array of key value pairs of linkId and text for MyVideo', function () {
				service.currQuestionnaire = {
					"resource": {
						"group": {
							"text": "MYVIDEO_QUESTIONNAIRE",
							"linkId": "VIDEO",
							"group": [
								{"linkId": "1", "text": "TAKE_VIDEO"}
							]
						}
					}
				};
				service.initQuestionnaireMap("MyVideo");
				var map = service.getQuestionnaireMap();
				expect(map).toEqual({"MYVIDEO_QUESTIONNAIRE": ["VIDEO"], "TAKE_VIDEO": ["1"]});
			});

			it('should not reload the questionnaire if questionnaire is already loaded', function () {
				spyOn(service, 'parseQuestionnaire');
				service.questionnaireMap = {"MYVIDEO_QUESTIONNAIRE": ["VIDEO"], "TAKE_VIDEO": ["1"]};
				service.initQuestionnaireMap("MyVideo");
				expect(service.parseQuestionnaire).not.toHaveBeenCalled();
			});

			it('should load the questionnaire if questionnaire is already loaded but is not the same type', function () {
				spyOn(service, 'parseQuestionnaire');
				service.currQuestionnaire = {
					"resource": {
						"group": {
							"text": "MYTELEDERM_QUESTIONNAIRE",
							"linkId": "IMAGE"
						}
					}
				};
				service.questionnaireMap = {"MYVIDEO_QUESTIONNAIRE": ["VIDEO"], "TAKE_VIDEO": ["1"]};
				service.initQuestionnaireMap("MyTelederm");
				expect(service.parseQuestionnaire).toHaveBeenCalled();
			});
		});

		describe('createResponse', function () {

			it('should create an empty response object if there is no existing questionnaire response', function () {
				service.initEvaluation("MyTelederm","mediaRequestId", null);
				$scope.$apply();
				expect(service.getCurrResponse()).toEqual({
					resourceType: 'QuestionnaireResponse',
					status: 'in-progress',
					identifier: {value: "mediaRequestId"},
					questionnaire: {reference: 'Questionnaire/questionnaireId'},
					subject: {reference: 'Patient/patientId'},
					group: {
						title:'MyTelederm Questionnaire Response',
						group: []
					}
				});
			});

			it('should create an empty response object for MyVideo if there is no existing questionnaire response', function () {
				service.initEvaluation("MyVideo","mediaRequestId", null);
				$scope.$apply();
				expect(service.getCurrResponse()).toEqual({
					resourceType: 'QuestionnaireResponse',
					status: 'in-progress',
					identifier: {value: "mediaRequestId"},
					questionnaire: {reference: 'Questionnaire/questionnaireId'},
					subject: {reference: 'Patient/patientId'},
					group: {
						title:'MyVideo Questionnaire Response',
						group: []
					}
				});
			});
		});

		describe('createAnswer', function () {

			it('should create an answer object based on the type passed', function () {
				var answer = service.createAnswer("1.1","string", "TEST STRING");
				expect(answer).toEqual({
					linkId : '1.1',
					answer : [{ valueString : "TEST STRING"}]
				});

				answer = service.createAnswer("1.1","bool", false);
				expect(answer).toEqual({
					linkId : '1.1',
					answer : [{ valueBoolean : false}]
				});
			});

			it('should create a coding  with fhirResources system name if a choice is passed', function () {
				var answer = service.createAnswer("1.1","choice", "TEST",1);
				expect(answer).toEqual({
					linkId : '1.1',
					answer : [{ valueCoding : {code: '1', display: 'TEST', system : 'https://URL/display/PGDMS/Client+Provenance+Mapping'}}]
				});
			});
		});

		describe('getValue and findElement', function () {
			beforeEach(function () {
				service.currResponse = {
					"resourceType": "QuestionnaireResponse",
					"group": {
						"text": "MyTelederm Questionnaire Response",
						"group": [
							{
								"linkId": "1",
								"question": [
									{
										"linkId": "1.1",
										"answer": [{
											"valueCoding": {
												"code": "0",
												"display": "BETTER",
												"system": "https://URL/display/PGDMS/Client+Provenance+Mapping"
											}
										}]
									},
									{"linkId": "1.2", "answer": [{"valueString": "comments on better"}]}
								]
							}
						]
					}
				}
			});

			it('getValue should return the value of the answer corresponding to the linkId passed', function () {
				var value = service.getValue(service.currResponse, "1.1");
				expect(value).toEqual('BETTER');

				value = service.getValue(service.currResponse, "1.2");
				expect(value).toEqual('comments on better');
			});

			it('findElement should return the object corresponding to the linkId passed', function () {
				var obj = service.findElement(service.currResponse, "1.1");
				expect(obj).toEqual({
					"linkId": "1.1",
					"answer": [{
						"valueCoding": {
							"code": "0",
							"display": "BETTER",
							"system": "https://URL/display/PGDMS/Client+Provenance+Mapping"
						}
					}]
				});
			});
		});

		describe('initEvaluation', function () {
            beforeEach(function () {
				spyOn(service, 'initUploadedImagesBySection');
             });

			it('should set service variables', function () {
				spyOn(service, 'createResponse');
				spyOn(service, 'transformResponse');
				service.initEvaluation("MyTelederm", "mediaRequestId");
				$scope.$apply();
				expect(service.resourceLinks).toEqual({
					'image-sections': '/media-request-service/v1/media/sections',
					'update-request': '/media-request-service/v1/patients/{UNIQUE_ID}/requests/{MEDIA_REQUEST_ID}',
				});
				expect(service.currQuestionnaire).toEqual({
					"resource": {
						"resourceType": "Questionnaire",
						"id": "questionnaireId",
						"group": {
							"text": "MYTELEDERM_QUESTIONNAIRE",
							"linkId": "IMAGE",
							"group": [
								{"linkId": "1", "text": "SKIN_STATUS"},
								{"linkId": "2", "text": "TOPICAL_MEDICATIONS"},
								{"linkId": "3", "text": "ORAL_MEDICATIONS"},
								{"linkId": "8", "text": "MEDIA"}
							]
						}
					}
				});
				expect(service.currPatient).toEqual({"id" : "patientId"});
				expect(service.questionnaireType).toEqual("MyTelederm");
				expect(service.createResponse).toHaveBeenCalledWith("MyTelederm", "mediaRequestId");
				expect(service.transformResponse).toHaveBeenCalled();
			});

			it('should set service variables based on questionnaire type', function () {
				spyOn(service, 'createResponse');
				spyOn(service, 'transformResponse');
				service.initEvaluation("MyVideo", "mediaRequestId");
				$scope.$apply();
				expect(service.questionnaireType).toEqual("MyVideo");
				expect(service.createResponse).toHaveBeenCalledWith("MyVideo", "mediaRequestId");
				expect(service.transformResponse).toHaveBeenCalled();
			});

        });
		
        describe('saveEvaluation', function () {
            beforeEach(function () {
                spyOn(service, 'updateResponse');
            });

            it('should call a POST on the current response if there is no id', function () {
				service.currResponse = {};
				//var data = { resourceType: 'QuestionnaireResponse', status: 'in-progress', group: { text: 'MyTelederm Questionnaire Response', group: [] }};

                service.saveEvaluation();

                expect(fhirResourcesMock.createUpdateFhirResource).toHaveBeenCalledWith({method: 'POST', resourceName: 'QuestionnaireResponse', data: { id: 'id1'}});
                expect(service.currResponse.id).toEqual('id1');
            });

			it('should call a PUT on the current response', function () {
				var data = '';
				service.setCurrResponse({
					id: 'id1'
				});

				service.saveEvaluation();

				expect(fhirResourcesMock.createUpdateFhirResource).toHaveBeenCalledWith({method: 'PUT', resourceName: 'QuestionnaireResponse/id1', data: { id: 'id1'}});
			});


			it('should update the evaluation status from "NOT_STARTED" to "STARTED" and response status to "in-progress"', function () {
                service.setEvaluation({
                    id: 'id1',
                    status: 'NOT_STARTED'
                });

                service.saveEvaluation().then(function (response) {});


                expect(service.getEvaluation().status).toEqual('STARTED');
                expect(service.getCurrResponse().status).toEqual('in-progress');
            });

			it('should update the status of the response from "COMPLETED" to "completed"', function () {
				service.setEvaluation({
					status: 'COMPLETED'
				});

				service.saveEvaluation().then(function (response) {});

				expect(service.getCurrResponse().status).toEqual('completed');
			});


			xit('should return an error if the call is unsuccessful', function () {
                var data;

				service.setCurrResponse({
					id: 'id1',
					status: 'STARTED'
				});

                service.saveEvaluation().then(function () {
                    //Empty success function
                }, function (error) {
                    data = "error!";
                });

                expect(data).toEqual('error!');
            });
        });
    });
});