define(['SubmitController'], function() {
    'use strict';

    describe("The Submit Controller", function () {
        var controller,
            scope,
            rootScope,
            fhirPatientMock,
            $stateMock,
            fhirPatientFetchCallback,
            fhirPatientUpdateCallback,
            fhirPatient;

        beforeEach(function () {
            module('angularTemplateApp');

            fhirPatient = {
                id: 444,
                name: {
                    family: [ 'userLast01' ],
                    given: [ 'userFirst01' ],
                    use: 'usual'
                }
            };

            $stateMock = jasmine.createSpyObj('$state', ['go']);
            fhirPatientMock = jasmine.createSpyObj('fhirPatient', ['fetch', 'update']);

            fhirPatientMock.fetch.and.returnValue({
                then: function (callback) {
                    callback(fhirPatient);
                }
            });

            fhirPatientMock.update.and.returnValue({
                then: function (callback) {
                    fhirPatientUpdateCallback = callback;
                }
            });

            module(function ($provide) {
                $provide.value('$state', $stateMock);
                $provide.value('fhirPatient', fhirPatientMock);
            });

            inject(function($controller, $rootScope) {
                scope = $rootScope.$new();
                controller = $controller;
                rootScope = $rootScope;

                spyOn(rootScope, '$broadcast');
            });
        });

        describe("initial state", function () {
            beforeEach(function () {
                fhirPatientMock.fetch.and.returnValue({
                    then: function (callback) {
                        fhirPatientFetchCallback = callback;
                    }
                });

                controller = controller('SubmitController', {$scope: scope});
            });

            it("should correctly set initial values", function () {
                expect(scope.contactType).toEqual('email');
                expect(scope.contactInfo).toEqual({
                    email: '',
                    phone: ''
                });

                expect(scope.errorHandling).toEqual({
                    'Valid-Email-Address' : {
                        message : 'Invalid email address',
                        priority : 1
                    },
                    'Phone-Number-Given' : {
                        message : 'You have chosen to be notified by phone but no phone number was provided.',
                        priority : 1
                    },
                    'Valid-Phone-Number' : {
                        message : 'You have entered an invalid phone number (formats accepted are: xxxxxxxxxx, xxx-xxx-xxxx, or (xxx) xxx-xxxx)',
                        priority : 1
                    }
                });
            });

            describe("fhirPatient fetch", function () {
                it("should be called on initial load", function () {
                    expect(fhirPatientMock.fetch).toHaveBeenCalled();
                });

                it("should update the fhirPatient on the scope after a successful fetch", function () {
                    fhirPatientFetchCallback(fhirPatient);
                    expect(scope.fhirPatient).toEqual(fhirPatient);
                });

                it("should call resetContactInfo after a successful fetch", function () {
                    spyOn(scope, 'resetContactInfo');

                    fhirPatientFetchCallback(fhirPatient);
                    expect(scope.resetContactInfo).toHaveBeenCalled();
                });
            });
        });

        describe("resetContactInfo function", function () {
            beforeEach(function () {
                controller = controller('SubmitController', {$scope: scope});
            });

            it("should set both email and phone to empty strings if there are no telecom entries", function () {
                scope.fhirPatient.telecom = undefined;

                scope.resetContactInfo();
                
                expect(scope.contactInfo.email).toEqual('');
                expect(scope.contactInfo.phone).toEqual('');
            });

            it("should set phone to an empty string if there is no mobile phone telecom entry", function () {
                scope.fhirPatient.telecom = [
                    {
                        system: 'phone',
                        value: '1111111111',
                        use: 'home'
                    },
                    {
                        system: 'email',
                        value: 'test@test.com',
                        use: 'home'
                    }
                ];

                scope.resetContactInfo();

                expect(scope.contactInfo.phone).toEqual('');
            });

            it("should set phone to the value of the mobile phone entry, if it exists", function () {
                scope.fhirPatient.telecom = [
                    {
                        system: 'phone',
                        value: '1111111111',
                        use: 'mobile'
                    },
                    {
                        system: 'email',
                        value: 'test@test.com',
                        use: 'home'
                    }
                ];

                scope.resetContactInfo();

                expect(scope.contactInfo.phone).toEqual('1111111111');
            });

            it("should set email to an empty string if there is no home email telecom entry", function () {
                scope.fhirPatient.telecom = [
                    {
                        system: 'phone',
                        value: '1111111111',
                        use: 'home'
                    },
                    {
                        system: 'email',
                        value: 'test@work.com',
                        use: 'work'
                    }
                ];

                scope.resetContactInfo();

                expect(scope.contactInfo.email).toEqual('');
            });

            it("should set email to the value of the email entry, if it exists", function () {
                scope.fhirPatient.telecom = [
                    {
                        system: 'phone',
                        value: '1111111111',
                        use: 'mobile'
                    },
                    {
                        system: 'email',
                        value: 'test@test.com',
                        use: 'home'
                    }
                ];

                scope.resetContactInfo();

                expect(scope.contactInfo.email).toEqual('test@test.com');
            });

            it("should set both phone and email if entries for both exist", function () {
                scope.fhirPatient.telecom = [
                    {
                        system: 'phone',
                        value: '1111111111',
                        use: 'mobile'
                    },
                    {
                        system: 'email',
                        value: 'test@test.com',
                        use: 'home'
                    }
                ];

                scope.resetContactInfo();

                expect(scope.contactInfo.email).toEqual('test@test.com');
                expect(scope.contactInfo.phone).toEqual('1111111111');
            });
        });

        describe("validateEmail function", function () {
            beforeEach(function () {
                scope.contactInfoForm = jasmine.createSpyObj('contactInfoForm', ['$setValidity']);

                controller = controller('SubmitController', {$scope: scope});
            });

            it("should validate to true if email is not set (required validation is done elsewhere)", function () {
                scope.contactInfo = {
                    email: ''
                };

                scope.validateEmail();

                expect(scope.contactInfoForm.$setValidity).toHaveBeenCalledWith('Valid-Email-Address', true);
            });

            it("should validate to false if email is invalid", function () {
                scope.contactInfo = {
                    email: 'test'
                };

                scope.validateEmail();

                expect(scope.contactInfoForm.$setValidity).toHaveBeenCalledWith('Valid-Email-Address', false);
            });

            it("should validate to false if email is valid", function () {
                scope.contactInfo = {
                    email: 'test@test.com'
                };

                scope.validateEmail();

                expect(scope.contactInfoForm.$setValidity).toHaveBeenCalledWith('Valid-Email-Address', true);
            });
        });

        describe("validatePhoneGiven function", function () {
            beforeEach(function () {
                scope.contactInfoForm = jasmine.createSpyObj('contactInfoForm', ['$setValidity']);

                controller = controller('SubmitController', {$scope: scope});
            });

            it("should validate to true if phone is not set and contactType is 'email'", function () {
                scope.contactType = 'email';
                
                scope.contactInfo = {
                    phone: ''
                };

                scope.validatePhoneGiven();

                expect(scope.contactInfoForm.$setValidity).toHaveBeenCalledWith('Phone-Number-Given', true);
            });

            it("should validate to false if phone is not set and contactType is 'phone'", function () {
                scope.contactType = 'phone';
                
                scope.contactInfo = {
                    phone: ''
                };

                scope.validatePhoneGiven();

                expect(scope.contactInfoForm.$setValidity).toHaveBeenCalledWith('Phone-Number-Given', false);
            });

            it("should validate to true if phone is set and contactType is 'phone'", function () {
                scope.contactType = 'phone';
                
                scope.contactInfo = {
                    phone: '1111111111'
                };

                scope.validatePhoneGiven();

                expect(scope.contactInfoForm.$setValidity).toHaveBeenCalledWith('Phone-Number-Given', true);
            });
        });

        describe("validateValidPhone function", function () {
            beforeEach(function () {
                scope.contactInfoForm = jasmine.createSpyObj('contactInfoForm', ['$setValidity']);

                controller = controller('SubmitController', {$scope: scope});
            });

            it("should validate to true if phone is not provided", function () {
                scope.contactInfo = {
                    phone: ''
                };

                scope.validateValidPhone();

                expect(scope.contactInfoForm.$setValidity).toHaveBeenCalledWith('Valid-Phone-Number', true);
            });

            it("should validate to false if phone is invalid", function () {
                scope.contactInfo = {
                    phone: 'asldkfjalsdjf'
                };

                scope.validateValidPhone();

                expect(scope.contactInfoForm.$setValidity).toHaveBeenCalledWith('Valid-Phone-Number', false);
            });

            it("should validate to true if phone has a format of ##########", function () {
                scope.contactInfo = {
                    phone: '1111111111'
                };

                scope.validateValidPhone();

                expect(scope.contactInfoForm.$setValidity).toHaveBeenCalledWith('Valid-Phone-Number', true);
            });

            it("should validate to true if phone has a format of ###-###-####", function () {
                scope.contactInfo = {
                    phone: '111-111-1111'
                };

                scope.validateValidPhone();

                expect(scope.contactInfoForm.$setValidity).toHaveBeenCalledWith('Valid-Phone-Number', true);
            });

            it("should validate to true if phone has a format of (###) ###-####", function () {
                scope.contactInfo = {
                    phone: '(111) 111-1111'
                };

                scope.validateValidPhone();

                expect(scope.contactInfoForm.$setValidity).toHaveBeenCalledWith('Valid-Phone-Number', true);
            });
        });

        describe("validate function", function () {
            beforeEach(function () {
                controller = controller('SubmitController', {$scope: scope});

                spyOn(scope, 'validateEmail');
                spyOn(scope, 'validatePhoneGiven');
                spyOn(scope, 'validateValidPhone');
            });

            it("should call both validateEmail and validatePhone", function () {
                scope.validate();

                expect(scope.validateEmail).toHaveBeenCalled();
                expect(scope.validatePhoneGiven).toHaveBeenCalled();
                expect(scope.validateValidPhone).toHaveBeenCalled();
            });
        });

        describe("emailEntryCheck function", function () {
            beforeEach(function () {
                controller = controller('SubmitController', {$scope: scope});
            });

            it("should return true if the entry has system 'email' and use 'home'", function () {
                expect(scope.emailEntryCheck({system: 'email', use: 'home'})).toEqual(true);
            });

            it("should return false if the entry has system 'email' but not use 'home'", function () {
                expect(scope.emailEntryCheck({system: 'email', use: 'work'})).toEqual(false);
            });

            it("should return false if the entry does not have system 'email' but has use 'home'", function () {
                expect(scope.emailEntryCheck({system: 'phone', use: 'home'})).toEqual(false);
            });

            it("should return false if the entry has neither system 'email' nor use 'home'", function () {
                expect(scope.emailEntryCheck({system: 'phone', use: 'mobile'})).toEqual(false);
            });
        });

        describe("phoneEntryCheck function", function () {
            beforeEach(function () {
                controller = controller('SubmitController', {$scope: scope});
            });

            it("should return true if the entry has system 'phone' and use 'mobile'", function () {
                expect(scope.phoneEntryCheck({system: 'phone', use: 'mobile'})).toEqual(true);
            });

            it("should return false if the entry has system 'phone' but not use 'moblie'", function () {
                expect(scope.phoneEntryCheck({system: 'phone', use: 'work'})).toEqual(false);
            });

            it("should return false if the entry does not have system 'phone' but has use 'mobile'", function () {
                expect(scope.phoneEntryCheck({system: 'email', use: 'mobile'})).toEqual(false);
            });

            it("should return false if the entry has neither system 'phone' nor use 'home'", function () {
                expect(scope.phoneEntryCheck({system: 'email', use: 'work'})).toEqual(false);
            });
        });

        describe("updateContactPreferenceRank", function () {
            beforeEach(function () {
                controller = controller('SubmitController', {$scope: scope});

                scope.fhirPatient.telecom = [
                    {
                        system: 'phone',
                        value: '111-111-1111',
                        use: 'mobile'
                    },
                    {
                        system: 'phone',
                        value: '222-222-2222',
                        use: 'work'
                    },
                    {
                        system: 'email',
                        value: 'test@test.com',
                        use: 'home'
                    },
                    {
                        system: 'email',
                        value: 'test@work.com',
                        use: 'work'
                    }
                ];
            });

            it("should set home email as rank 1 and everything else as rank 2 if contactType is 'email'", function () {
                scope.contactType = 'email';

                scope.updateContactPreferenceRank();

                expect(scope.fhirPatient.telecom).toEqual([
                    {
                        system: 'phone',
                        value: '111-111-1111',
                        use: 'mobile',
                        rank: 2
                    },
                    {
                        system: 'phone',
                        value: '222-222-2222',
                        use: 'work',
                        rank: 2
                    },
                    {
                        system: 'email',
                        value: 'test@test.com',
                        use: 'home',
                        rank: 1
                    },
                    {
                        system: 'email',
                        value: 'test@work.com',
                        use: 'work',
                        rank: 2
                    }
                ]);
            });

            it("should set mobile phone as rank 1 and everything else as rank 2 if contactType is 'phone'", function () {
                scope.contactType = 'phone';

                scope.updateContactPreferenceRank();

                expect(scope.fhirPatient.telecom).toEqual([
                    {
                        system: 'phone',
                        value: '111-111-1111',
                        use: 'mobile',
                        rank: 1
                    },
                    {
                        system: 'phone',
                        value: '222-222-2222',
                        use: 'work',
                        rank: 2
                    },
                    {
                        system: 'email',
                        value: 'test@test.com',
                        use: 'home',
                        rank: 2
                    },
                    {
                        system: 'email',
                        value: 'test@work.com',
                        use: 'work',
                        rank: 2
                    }
                ]);
            });
        });

        describe("updateEmailForFhirPatient function", function () {
            beforeEach(function () {
                controller = controller('SubmitController', {$scope: scope});
                scope.contactInfo = {
                    email: ''
                };
            });

            it("should leave the telecom entries on the fhirPatient unchanged if email is not set", function () {
                scope.fhirPatient.telecom = [
                    {
                        system: 'phone',
                        value: '222-222-2222',
                        use: 'home'
                    },
                    {
                        system: 'email',
                        value: 'test@test.info',
                        use: 'home'
                    }
                ];

                scope.updateEmailForFhirPatient();

                expect(scope.fhirPatient.telecom).toEqual([
                    {
                        system: 'phone',
                        value: '222-222-2222',
                        use: 'home'
                    },
                    {
                        system: 'email',
                        value: 'test@test.info',
                        use: 'home'
                    }
                ]);
            })

            it("should add the email entry when there are no telecom entries on the fhirPatient", function () {
                scope.contactInfo.email = 'test@test.com';

                scope.updateEmailForFhirPatient();

                expect(scope.fhirPatient.telecom).toEqual([
                    {
                        system: 'email',
                        value: 'test@test.com',
                        use: 'home'
                    }
                ]);
            });

            it("should add the email entry  if there are other telecom entries", function () {
                scope.contactInfo.email = 'test@test.com';

                scope.fhirPatient.telecom = [
                    {
                        system: 'phone',
                        value: '222-222-2222',
                        use: 'home'
                    },
                    {
                        system: 'email',
                        value: 'test@work.com',
                        use: 'work'
                    }
                ];

                scope.updateEmailForFhirPatient();

                expect(scope.fhirPatient.telecom).toEqual([
                    {
                        system: 'phone',
                        value: '222-222-2222',
                        use: 'home'
                    },
                    {
                        system: 'email',
                        value: 'test@work.com',
                        use: 'work'
                    },
                    {
                        system: 'email',
                        value: 'test@test.com',
                        use: 'home'
                    }
                ])
            });

            it("should replace the previous email entry", function () {
                scope.contactInfo.email = 'test@test.com';

                scope.fhirPatient.telecom = [
                    {
                        system: 'phone',
                        value: '222-222-2222',
                        use: 'mobile'
                    },
                    {
                        system: 'email',
                        value: 'test@test.info',
                        use: 'home'
                    }
                ];

                scope.updateEmailForFhirPatient();

                expect(scope.fhirPatient.telecom).toEqual([
                    {
                        system: 'phone',
                        value: '222-222-2222',
                        use: 'mobile'
                    },
                    {
                        system: 'email',
                        value: 'test@test.com',
                        use: 'home'
                    }
                ]);
            });
        });

        describe("updatePhoneForFhirPatient function", function () {
            beforeEach(function () {
                controller = controller('SubmitController', {$scope: scope});
                scope.contactInfo = {
                    phone: ''
                }
            });

            it("should remove the preexisting mobile phone entry when no phone is set", function () {
                scope.fhirPatient.telecom = [
                    {
                        system: 'phone',
                        value: '111-111-1111',
                        use: 'mobile',
                        rank: 1
                    }
                ];

                scope.updatePhoneForFhirPatient();

                expect(scope.fhirPatient.telecom).toEqual([]);
            });

            it("should leave the telecom entries unchanged if no phone is set", function () {
                scope.fhirPatient.telecom = [
                    {
                        system: 'phone',
                        value: '222-222-2222',
                        use: 'home'
                    },
                    {
                        system: 'email',
                        value: 'test@test.com',
                        use: 'home'
                    }
                ];

                scope.updatePhoneForFhirPatient();

                expect(scope.fhirPatient.telecom).toEqual([
                    {
                        system: 'phone',
                        value: '222-222-2222',
                        use: 'home'
                    },
                    {
                        system: 'email',
                        value: 'test@test.com',
                        use: 'home'
                    }
                ]);
            });

            it("should add the mobile phone entry when there are no telecom entries on the fhirPatient", function () {
                scope.contactInfo.phone = '111-111-1111';

                scope.updatePhoneForFhirPatient();

                expect(scope.fhirPatient.telecom).toEqual([
                    {
                        system: 'phone',
                        value: '111-111-1111',
                        use: 'mobile'
                    }
                ]);
            });

            it("should add the mobile phone entry if there are other telecom entries", function () {
                scope.contactInfo.phone = '111-111-1111';

                scope.fhirPatient.telecom = [
                    {
                        system: 'phone',
                        value: '222-222-2222',
                        use: 'home'
                    },
                    {
                        system: 'email',
                        value: 'test@test.com',
                        use: 'home'
                    }
                ];

                scope.updatePhoneForFhirPatient();

                expect(scope.fhirPatient.telecom).toEqual([
                    {
                        system: 'phone',
                        value: '222-222-2222',
                        use: 'home'
                    },
                    {
                        system: 'email',
                        value: 'test@test.com',
                        use: 'home'
                    },
                    {
                        system: 'phone',
                        value: '111-111-1111',
                        use: 'mobile'
                    }
                ])
            });

            it("should replace the previous mobile phone entry", function () {
                scope.contactInfo.phone = '111-111-1111';

                scope.fhirPatient.telecom = [
                    {
                        system: 'phone',
                        value: '222-222-2222',
                        use: 'mobile'
                    },
                    {
                        system: 'email',
                        value: 'test@test.com',
                        use: 'home'
                    }
                ];

                scope.updatePhoneForFhirPatient();

                expect(scope.fhirPatient.telecom).toEqual([
                    {
                        system: 'email',
                        value: 'test@test.com',
                        use: 'home'
                    },
                    {
                        system: 'phone',
                        value: '111-111-1111',
                        use: 'mobile'
                    }
                ]);
            });
        });

        describe("updateContactForFhirPatient function", function () {
            beforeEach(function () {
                controller = controller('SubmitController', {$scope: scope});

                spyOn(scope, 'updateEmailForFhirPatient');
                spyOn(scope, 'updatePhoneForFhirPatient');
                spyOn(scope, 'updateContactPreferenceRank');
            });

            it("should call both updateEmailForFhirPatient and updatePhoneForFhirPatient", function () {
                scope.updateContactForFhirPatient();

                expect(scope.updateEmailForFhirPatient).toHaveBeenCalled();
                expect(scope.updatePhoneForFhirPatient).toHaveBeenCalled();
                expect(scope.updateContactPreferenceRank).toHaveBeenCalled();
            });
        });

        describe("submit function", function () {
            var afterSubmitCallback;

            beforeEach(function () {
                controller = controller('SubmitController', {$scope: scope});

                scope.successRoute = 'success-confirmation';

                scope.contactInfoForm = {
                    validationSummary: {
                        validate: function () {}
                    }
                };

                scope.submitCallback = function () {};

                spyOn(scope, 'validate');
                spyOn(scope.contactInfoForm.validationSummary, 'validate').and.returnValue({
                    then: function (callback) {
                        callback();
                    }
                });
                spyOn(scope, 'updateContactForFhirPatient');
                spyOn(scope, 'submitCallback');
            });

            it("should call the validate function", function () {
                scope.submit();

                expect(scope.validate).toHaveBeenCalled();
            });
            
            it("should call the updateContactForFhirPatient function", function () {
                scope.submit();

                expect(scope.updateContactForFhirPatient).toHaveBeenCalled();
            });

            it("should update the fhirPatient with the current fhirPatient on the scope", function () {
                scope.submit();

                expect(fhirPatientMock.update).toHaveBeenCalledWith(scope.fhirPatient);
            });

            it("should call the submitCallback after updating the fhirPatient", function () {
                scope.submit();

                expect(fhirPatientMock.update).toHaveBeenCalled();
                expect(scope.submitCallback).not.toHaveBeenCalled();

                fhirPatientUpdateCallback();

                expect(scope.submitCallback).toHaveBeenCalled();
            });

            it("should not update the FHIR patient contact or call update if validate on the form is not successful", function () {
                scope.contactInfoForm.validationSummary.validate.and.returnValue({
                    then: function (callback) {
                        //empty success block
                    }
                });

                scope.submit();

                expect(scope.updateContactForFhirPatient).not.toHaveBeenCalled();
                expect(fhirPatientMock.update).not.toHaveBeenCalled();
            });
        });

        describe("previous function", function () {
            it("should call the previousCallback function", function () {
                scope.previousCallback = function () {};
                spyOn(scope, 'previousCallback');

                controller = controller('SubmitController', {$scope: scope});
                scope.previous();

                expect(scope.previousCallback).toHaveBeenCalled();
            });
        });

        describe("swipeLeft function", function () {
			it("should broadcast swipeLeft if showProgressBar is true", function () {
                scope.showProgressBar = true;
                controller = controller('SubmitController', {$scope: scope});
                scope.swipeLeft();
				expect(rootScope.$broadcast).toHaveBeenCalledWith('swipeLeft');
            });
            
            it("should call previous if showProgressBar is false", function () {
                scope.showProgressBar = false;
                scope.previousCallback = function () {};
                spyOn(scope, 'previousCallback');
                controller = controller('SubmitController', {$scope: scope});
                scope.swipeLeft();
				expect(scope.previousCallback).toHaveBeenCalled();
			});
        });
    });
});