define(['consentService'], function () {
    'use strict';

    describe("Consent Service", function () {
        var service,
            qMock,
            deferredMock = {
                resolve: function () {},
                promise: 'promise'
            },
            localStorageServiceMock,
            sessionStorageServiceMock,
            userSessionServiceMock,
            showModal,
            user = {
                id: '1234'
            },
            locStorage = {},
            sessStorage = {};

        beforeEach(function () {
            module('angularTemplateApp');

            qMock = jasmine.createSpyObj('$q', ['defer']);
            localStorageServiceMock = jasmine.createSpyObj('localStorageService', ['showModal']);
            sessionStorageServiceMock = jasmine.createSpyObj('sessionStorageService', ['showModal']);
            userSessionServiceMock = jasmine.createSpyObj('userSessionService', ['fetch']);

            spyOn(localStorage, 'getItem').and.callFake(function (itemName) {
                return locStorage[itemName];
            });

            spyOn(localStorage, 'setItem').and.callFake(function (itemName, item) {
                locStorage[itemName] = item; 
            });

            spyOn(sessionStorage, 'getItem').and.callFake(function (itemName) {
                return sessStorage[itemName];
            });

            spyOn(sessionStorage, 'setItem').and.callFake(function (itemName, item) {
                sessStorage[itemName] = item; 
            });

            spyOn(deferredMock, 'resolve');

            qMock.defer.and.returnValue(deferredMock);

            localStorageServiceMock.keys = [];

            localStorageServiceMock.showModal.and.callFake(function () {
                return showModal;
            });

            sessionStorageServiceMock.keys = [];
            
            sessionStorageServiceMock.showModal.and.callFake(function () {
                return showModal;
            });

            userSessionServiceMock.fetch.and.returnValue({
                then: function (callback) {
                    callback(user);
                }
            });

            module(function ($provide) {
                $provide.value('$q', qMock);
                $provide.value('localStorageService', localStorageServiceMock);
                $provide.value('sessionStorageService', sessionStorageServiceMock);
                $provide.value('userSessionService', userSessionServiceMock);
            });

            inject(function (consentService) {
                service = consentService;

                service.consentKey = 'consent';
            });
        });

        it("should set consentAccepted to false by default", function () {
            expect(service.consentAccepted).toEqual(false);
        });

        describe("showConsent function", function () {
            var showConsent;

            it("should not call sessionStorageService and localStorageService showModal with the user and the consentKey if consentAccepted is true", function () {
                service.consentAccepted = true;
                service.showConsent(user);

                expect(sessionStorageServiceMock.showModal).not.toHaveBeenCalledWith(user, 'consent');
                expect(localStorageServiceMock.showModal).not.toHaveBeenCalledWith(user, 'consent');
            });

            it("should call sessionStorageService and localStorageService showModal with the user and the consentKey if consentAccepted is false and sessionStorage is empty", function () {
                service.consentAccepted = false;
                showModal = true;
                service.showConsent(user);

                expect(sessionStorageServiceMock.showModal).toHaveBeenCalledWith(user, 'consent');
                expect(localStorageServiceMock.showModal).toHaveBeenCalledWith(user, 'consent');
            });

            it("should only call sessionStorageService showModal with the user and the consentKey if consentAccepted is false and sessionStorage has the consentKey", function () {
                service.consentAccepted = false;
                showModal = false;
                service.showConsent(user);

                expect(sessionStorageServiceMock.showModal).toHaveBeenCalledWith(user, 'consent');
                expect(localStorageServiceMock.showModal).not.toHaveBeenCalledWith(user, 'consent');
            });

            describe("consentAccepted is false, showModal returns false", function () {
                beforeEach(function () {
                    service.consentAccepted = false;
                    showModal = false;

                    showConsent = service.showConsent(user);
                });

                it("should setConsentAccepted to true", function () {
                    expect(service.consentAccepted).toEqual(true);
                });

                it("should return false", function () {
                    expect(showConsent).toEqual(false);
                });
            });

            describe("consentAccepted is false, showModal returns true", function () {
                beforeEach(function () {
                    service.consentAccepted = false;
                    showModal = true;

                    showConsent = service.showConsent(user);
                });

                it("should setConsentAccepted to true", function () {
                    expect(service.consentAccepted).toEqual(false);
                });

                it("should return false", function () {
                    expect(showConsent).toEqual(true);
                });
            });

            describe("consentAccepted is true, showModal returns false", function () {
                beforeEach(function () {
                    service.consentAccepted = true;
                    showModal = false;

                    showConsent = service.showConsent(user);
                });

                it("should setConsentAccepted to true", function () {
                    expect(service.consentAccepted).toEqual(true);
                });

                it("should return false", function () {
                    expect(showConsent).toEqual(false);
                });
            });

            describe("consentAccepted is true, showModal returns true", function () {
                beforeEach(function () {
                    service.consentAccepted = true;
                    showModal = true;

                    showConsent = service.showConsent(user);
                });

                it("should setConsentAccepted to true", function () {
                    expect(service.consentAccepted).toEqual(true);
                });

                it("should return false", function () {
                    expect(showConsent).toEqual(false);
                });
            });
        });

        describe("consent function", function () {
            it("should return a promise", function () {
                expect(service.consent()).toEqual('promise');
            });

            it("should set consentAccepted to true", function () {
                service.consentAccepted = false;

                service.consent();

                expect(service.consentAccepted).toEqual(true);
            });

            it("should only save the consent status to sessionStorage if noShowSelected is false", function () {
                service.consent(false);

                expect(sessionStorage.getItem(sessionStorageServiceMock.keys[service.consentKey])).toEqual(true);
                expect(localStorage.getItem(localStorageServiceMock.keys[service.consentKey])).not.toBeDefined();
            });

            it("should save the consent status to sessionStorage and localStorage if noShowSelected is true", function () {
                service.consent(true);

                expect(sessionStorage.getItem(sessionStorageServiceMock.keys[service.consentKey])).toEqual(true);
                expect(localStorage.getItem(localStorageServiceMock.keys[service.consentKey])).toEqual(true);
            });

            it("should only save the consent status to sessionStorage/localStorage and resolve the promise after the userSession is fetched", function () {
                var userSessionCallback;

                userSessionServiceMock.fetch.and.returnValue({
                    then: function (callback) {
                        userSessionCallback = callback;
                    }
                });
                
                service.consent(true);

                expect(userSessionServiceMock.fetch).toHaveBeenCalled();
                expect(sessionStorage.setItem).not.toHaveBeenCalled();
                expect(localStorage.setItem).not.toHaveBeenCalled();
                expect(deferredMock.resolve).not.toHaveBeenCalled();

                userSessionCallback(user);

                expect(sessionStorage.setItem).toHaveBeenCalled();
                expect(localStorage.setItem).toHaveBeenCalled();
                expect(deferredMock.resolve).toHaveBeenCalled();
            });

            it("should save to sessionStorage but not save the consent status to localStorage if noShowSelected is false", function () {
                locStorage = {};
                sessStorage = {};

                service.consent(false);

                expect(locStorage).toEqual({});
                expect(sessStorage).not.toEqual({});
            });

            it("should save consent status to sessionStorage and localStorage if noShowSelected is true", function () {
                locStorage = {};
                sessStorage = {};

                service.consent(true);

                expect(locStorage).not.toEqual({});
                expect(sessStorage).not.toEqual({});
            });
        });
    });
});