define(['ConsentController'], function () {
    'use strict';

    describe("Consent Controller", function () {
        var controller,
            scope,
            deferred,
            stateMock,
            stateParamsMock,
            modalMock,
            modalInstanceMock,
            focusServiceMock,
            consentServiceMock;

        beforeEach(function () {
            module('angularTemplateApp');

            stateMock = jasmine.createSpyObj('$state', ['go']);
            modalMock = jasmine.createSpyObj('$modal', ['open']);
            modalInstanceMock = jasmine.createSpyObj('modalInstance', ['dismiss']);
            focusServiceMock = jasmine.createSpyObj('focusService', ['focusElement']);
            consentServiceMock = jasmine.createSpyObj('consentServiceMock', ['consent']);

            modalMock.open.and.callFake(function (modalConfig) {
                modalInstanceMock.modalInfo = modalConfig.resolve.modalInfo();
                modalInstanceMock.params = modalConfig.resolve.params();

                return modalInstanceMock;
            });

            stateParamsMock = {routeBase: 'route-base'};

            module(function ($provide) {
                $provide.value('$state', stateMock);
                $provide.value('$stateParams', stateParamsMock);
                $provide.value('globalDelayTracker', 'globalDelayTracker');
                $provide.value('$modal', modalMock);
                $provide.value('focusService', focusServiceMock);
                $provide.value('consentService', consentServiceMock);

            });

            inject(function ($controller, $rootScope, $q) {
                scope = $rootScope.$new();
                controller = $controller;

                deferred = $q.defer();
                consentServiceMock.consent.and.returnValue(deferred.promise);
            });
        });

        describe("Initial State", function () {
            beforeEach(function () {
                controller = controller('ConsentController', {$scope: scope});
            });

            it("should set initial values", function () {
                expect(scope.globalDelayTracker).toEqual('globalDelayTracker');
                expect(scope.noShowConsent).toEqual(false);
            });

            it("should focus on the main header", function () {
                expect(focusServiceMock.focusElement).toHaveBeenCalledWith('.main-header h1');
            });
        });

        describe("cancel function", function () {
            beforeEach(function () {
                controller = controller('ConsentController', {$scope: scope});

                scope.cancel();
            });

            it("should send the user to the app select page", function () {
                expect(stateMock.go).toHaveBeenCalledWith('main.app-select');
            });
        });

        describe("updateNoShow", function () {
            beforeEach(function () {
                controller = controller('ConsentController', {$scope: scope});
            });
            
            it("should set noShowConsent to true if it was previously false", function () {
                scope.noShowConsent = false;
                scope.updateNoShow();

                expect(scope.noShowConsent).toEqual(true);
            });
            
            it("should set noShowConsent to false if it was previously true", function () {
                scope.noShowConsent = true;
                scope.updateNoShow();

                expect(scope.noShowConsent).toEqual(false);
            });
        });

        describe("consent function", function () {
            var consentCallback;

            beforeEach(function () {
                controller = controller('ConsentController', {$scope: scope});
                scope.noShowConsent = 'noShowConsent';

                scope.consent();
            });

            it("should call consent on the consentService", function () {
                expect(consentServiceMock.consent).toHaveBeenCalledWith('noShowConsent');
            });

            it("should redirect to the routeBase after calling consent on the consentService", function () {
                expect(consentServiceMock.consent).toHaveBeenCalled();
                expect(stateMock.go).not.toHaveBeenCalled();

                deferred.resolve();
                scope.$digest();

                expect(stateMock.go).toHaveBeenCalledWith('route-base');
            });
        });

        describe("dontConsent function", function () {
            beforeEach(function () {
                controller = controller('ConsentController', {$scope: scope});
            });

            it("should open the correct modal", function () {
                scope.dontConsent();

                var modalConfig = modalMock.open.calls.argsFor(0)[0];

                expect(modalConfig.windowTemplateUrl).toEqual('modules/ui-components/modals/helper/modal-window_template.html');
                expect(modalConfig.templateUrl).toEqual('modules/ui-components/modals/consent-declined/consent-declined_template.html');
                expect(modalConfig.controller).toEqual('ConsentDeclinedController');
                expect(modalConfig.backdrop).toEqual('static');
                expect(modalConfig.keyboard).toEqual(false);
                expect(modalConfig.resolve.modalInfo()).toEqual({value: null, dismissed: true});
            });

            it("should dismiss a previously opened modal", function () {
                scope.dontConsent();

                modalInstanceMock.modalInfo.dismissed = false;

                scope.dontConsent();

                expect(modalInstanceMock.modalInfo.dismissed).toEqual(true);
                expect(modalInstanceMock.dismiss).toHaveBeenCalled();
            });
        });

        describe("when dial 911 link is clicked", function () {
            beforeEach(function () {
                controller = controller('ConsentController', {$scope: scope});
            });

            it ("should open a modal with href 911", function() {
                var obj = {currentTarget : {id : 'btn911'}};
                scope.dial911(obj);

                var modalObject = modalMock.open.calls.argsFor(0)[0];

                expect(modalObject.windowTemplateUrl).toEqual('modules/ui-components/modals/helper/modal-window_template.html');
                expect(modalObject.templateUrl).toEqual('modules/ui-components/modals/confirm-dial/confirm-dial_template.html');
                expect(modalObject.controller).toEqual('ConfirmDialController');
                expect(modalObject.backdrop).toEqual('static');
                expect(modalObject.keyboard).toEqual(false);
                expect(modalObject.resolve.params()).toEqual({id:"btn911", href: "911" });

            });
        });
    });
});