define(['videoUploadService'], function () {
    'use strict';

    describe('Video Upload Service', function () {
        var service,
            qMock,
            deferredMock,
            fhirResourcesMock,
            videoServiceMock,
            evaluation = {
                'mediaRequestId': '1'
            };

        beforeEach(function () {
            module('angularTemplateApp');

            qMock = jasmine.createSpyObj('$q', ['defer']);
            deferredMock = jasmine.createSpyObj('deferred', ['resolve', 'reject', 'promise']);
            fhirResourcesMock = jasmine.createSpyObj('fhirResources', ['createUpdateFhirResource', 'getFhirBaseUrl', 'appendProvenanceMetadata']);
            videoServiceMock = jasmine.createSpyObj('videoService', ['getEvaluation', 'setEvaluation', 'saveEvaluation', 'getPatient', 'setVideo']);

            qMock.defer.and.returnValue(deferredMock);
			
			fhirResourcesMock.createUpdateFhirResource.and.returnValue({
				then: function (callback) {
					callback("Media/123");
				}
            });
            fhirResourcesMock.getFhirBaseUrl.and.returnValue('/pgd-fhir-services/v1/fhir');


			videoServiceMock.getEvaluation.and.callFake(function () {
                return evaluation;
            });

            videoServiceMock.setEvaluation.and.returnValue({
                then: function (callback) {
                    callback();
                }
            });

            videoServiceMock.saveEvaluation.and.returnValue({
                then: function (callback) {
                    callback();
                }
            });

			videoServiceMock.getPatient.and.callFake(function () {
				return {id: '1'};
			});
			
            module(function ($provide) {
                $provide.value('$q', qMock);
                $provide.value('videoService', videoServiceMock);
                $provide.value('fhirResources', fhirResourcesMock);
            });

            inject(function (videoUploadService) {
                service = videoUploadService;
            });
        });

        describe('uploadVideo function', function () {
			beforeEach(function () {
				spyOn(service, 'getBase64').and.returnValue({
					then: function (callback) {
						callback('mockBase64BinaryData');
					}
				});
			});

			it('should return a promise', function () {
                expect(service.uploadVideo({}, {})).toEqual(deferredMock.promise);
            });

            it('should make a post request if the id is not set on the data object', function () {
                service.uploadVideo({}, {});

				expect(fhirResourcesMock.createUpdateFhirResource.calls.allArgs()[0][0].method).toEqual('POST');
            });

            it('should make a put request if the id is set on the data object', function () {
                service.uploadVideo({}, {
                    id: 'someId'
                });

				expect(fhirResourcesMock.createUpdateFhirResource.calls.allArgs()[0][0].method).toEqual('PUT');
            });

            it('should make a request with base64 binary data', function () {
                var file = {
                    fileName: 'file.mov'
                };

                var data = {
                    id: 'someId'
                };
				
                service.uploadVideo(file, data);

				expect(fhirResourcesMock.createUpdateFhirResource.calls.allArgs()[0][0].data.type).toEqual('application/json');
            });

            it('should set the video on videoService if the request is successful', function () {
                service.uploadVideo({}, {});

                expect(videoServiceMock.setVideo).toHaveBeenCalledWith({ id: '123', imageUrl: '/pgd-fhir-services/v1/fhir/Media/123'});
            });

            it('should resolve the promise if the request is successful', function () {
                service.uploadVideo({}, {});

                expect(deferredMock.resolve).toHaveBeenCalled();
            });

            it('should call saveEvaluation if the saveEval parameter is undefined', function () {
                service.uploadVideo({}, {});

                expect(videoServiceMock.saveEvaluation).toHaveBeenCalled();
            });

            it('should call saveEvaluation if the saveEval parameter is set to true', function () {
                service.uploadVideo({}, {}, true);

                expect(videoServiceMock.saveEvaluation).toHaveBeenCalled();
            });

            it('should not call saveEvaluation if the saveEval parameter is set to false', function () {
                service.uploadVideo({}, {}, false);
				
                expect(videoServiceMock.saveEvaluation).not.toHaveBeenCalled();
            });
        });

		describe('createMedia function', function () {
			var data = {
				"id": "123"
			};
			
			var file = {
				'type': 'mov',
				'size': '50',
				'name': 'mockVideo.mov'
			};

			var mockResponse = {
				"resourceType":"Media",
				"type":"video",
				"identifier":[{
					"value": "1"
				}],
				"subject":{
					"reference":"Patient/1"
				},
				"content":{
					"contentType": "mov",
					"data": "[[base64-media]]",
					"size": '50',
					"title": "mockVideo.mov",
				},
				"id":"123"
			};

			it('should be defined', function () {
				expect(service.createMedia).toBeDefined();
			});

			it('should return the correct value when calling method', function () {
				expect(service.createMedia(data.id, evaluation.mediaRequestId, file)).toEqual(mockResponse);
			});
		});

		describe('getBase64 function', function () {
			it('should be defined', function () {
				expect(service.getBase64).toBeDefined();
			});

			it('should return a promise when called', function () {
				expect(service.getBase64()).toEqual(deferredMock.promise);
			});
		});


	});
});