define(['videoService'], function () {
    'use strict';

    describe('Video Service', function () {
        var service,
            httpMock,
            localResourceDirectoryServiceMock,
            fhirResourcesMock,
            fhirPatientMock,
            questionnaireServiceMock;

        beforeEach(function () {
            module('angularTemplateApp');

            localResourceDirectoryServiceMock = jasmine.createSpyObj('localResourceDirectoryServiceMock', ['fetch']);
            fhirResourcesMock = jasmine.createSpyObj('fhirResourcesMock', ['resolveFhirResourceByReference', 'getSystem']);
            fhirPatientMock = jasmine.createSpyObj('fhirPatient', ['fetch']);
            questionnaireServiceMock = jasmine.createSpyObj('questionnaireServiceMock', ['fetch', 'getQuestionnaire']);

            localResourceDirectoryServiceMock.fetch.and.returnValue({
                then: function (callback) {
                    callback({
                        'update-request': '/media-request-service/v1/patients/{UNIQUE_ID}/requests/{MEDIA_REQUEST_ID}'
                    });
                }
            });

			fhirResourcesMock.resolveFhirResourceByReference.and.returnValue({
				then: function (callback) {
					callback({
						 "resourceType": "QuestionnaireResponse",
						"group": {
							title: 'Image Questionnaire Response',
							group: []
						}
					});
				}
			});
            fhirResourcesMock.getSystem.and.returnValue('https://URL/display/PGDMS/Client+Provenance+Mapping');
            
            fhirPatientMock.fetch.and.returnValue({
                then: function () {}
            });

            questionnaireServiceMock.fetch.and.returnValue({
				then: function (callback) {
					callback({
						"MyVideo" : "5b99b48a-f8f6-4e82-b8de-99f99e476046"
					});
				}
			});

            module(function ($provide) {
                $provide.value('localResourceDirectoryService', localResourceDirectoryServiceMock);
                $provide.value('fhirResources', fhirResourcesMock);
                $provide.value('fhirPatient', fhirPatientMock);
                $provide.value('questionnaireService', questionnaireServiceMock);
            });

            inject(function ($q, videoService, $httpBackend) {
                httpMock = $httpBackend;
                
                service = videoService;
            });
        });

        describe('getMediaRequest and setMediaRequest', function () {
            it('should return a default mediaRequest if none is set', function () {
                expect(service.getMediaRequest()).toEqual({});
            });

            it('should allow the user to set an mediaRequest and retrieve it', function () {
                service.setMediaRequest({
                    requestType: 'VIDEO'
                });

                expect(service.getMediaRequest()).toEqual({
                    requestType: 'VIDEO'
                });
            });

            it('should initialize the evaluation object when the mediaRequest is set', function () {
                spyOn(service, 'initEvaluation').and.callThrough();
                
                service.setMediaRequest({});
                service.resourceLinks = {
                    'update-request': '/media-request-service/v1/patients/{UNIQUE_ID}/requests/{MEDIA_REQUEST_ID}'
                };

                expect(service.initEvaluation).toHaveBeenCalled();
            });
        });

        describe('saveMediaRequest', function () {
            it('should call a PUT on the current mediaRequest', function () {
                var data;

                service.setMediaRequest({
                    id: 'id1',
                    evaluation: {
                        status: 'STARTED'
                    }
                });

                httpMock.expectPATCH('/media-request-service/v1/patients/{UNIQUE_ID}/requests/id1').respond(200);

                service.resourceLinks = {
                    'update-request': '/media-request-service/v1/patients/{UNIQUE_ID}/requests/{MEDIA_REQUEST_ID}'
                };

                service.saveMediaRequest().then(function () {
                    data = 'success!';
                });

                httpMock.flush();

                expect(data).toEqual('success!');
            });

            it('should update the status from "NOT_STARTED" to "STARTED"', function () {
                service.setMediaRequest({
                    status: 'PENDING',
                    id: 'id1'
                });

                httpMock.expectPATCH('/media-request-service/v1/patients/{UNIQUE_ID}/requests/id1').respond(200);

                service.resourceLinks = {
                    'update-request': '/media-request-service/v1/patients/{UNIQUE_ID}/requests/{MEDIA_REQUEST_ID}'
                };

                service.saveMediaRequest().then(function () {});

                httpMock.flush();

                expect(service.getMediaRequest().status).toEqual('PENDING');
            });

            it('should return an error if the call is unsuccessful', function () {
                var data;

                service.setMediaRequest({
                    status: 'PENDING',
                    id: 'id1'
                });

                httpMock.expectPATCH('/media-request-service/v1/patients/{UNIQUE_ID}/requests/id1').respond(400, 'error!');

                service.resourceLinks = {
                    'update-request': '/media-request-service/v1/patients/{UNIQUE_ID}/requests/{MEDIA_REQUEST_ID}'
                };

                service.saveMediaRequest().then(function () {
                    //Empty success function
                }, function (error) {
                    data = error;
                });

                httpMock.flush();

                expect(data).toEqual('error!');
            });
        });
    });
});