define(['VideoUploadController'], function() {
    'use strict';

    describe("The Video Upload Controller", function () {
        var controller,
            scope,
            rootScope,
            stateMock,
            httpMock,
            timeoutMock,
            timeoutCallback,
            focusServiceMock,
            videoServiceMock,
            videoUploadServiceMock,
            fileValidationServiceMock,
            mediaRequest = {
                providerContacts: [],
                mediaTypes: []
            },
            evaluation = {},
            videoResponse = {
                data: {
                    content: {
                        'contentType': 'image',
                        'data': 'image-content'
                    }
                }
            },
            fileTypes = ['mov', 'mp4'],
            isfileSizeValid,
            isfileTypeValid,
            formatterMock,
            modalMock,
            fakeModal = {
                result: {
                    then: function(confirmCallback) {
                        this.confirmCallBack = confirmCallback;
                    }
                },
                close: function( item ) {
                    this.result.confirmCallBack( item );
                }
            },
            configServiceValues = {
                imageFileSizeLimit: 3,
                suggestedVideoSize: 10,
                videoFileSizeLimit: 20
            };

        beforeEach(function () {
            module('angularTemplateApp');
            
            stateMock = jasmine.createSpyObj('state', ['go']);
            httpMock = jasmine.createSpyObj('httpMock', ['$http']);
            timeoutMock = jasmine.createSpyObj('timeoutMock', ['$timeout']);
            focusServiceMock = jasmine.createSpyObj('focusService', ['focusElement']);
            videoServiceMock = jasmine.createSpyObj('videoService', ['getMediaRequest', 'getEvaluation', 'saveEvaluation']);
            videoUploadServiceMock = jasmine.createSpyObj('videoUploadServiceMock', ['uploadVideo']);
            fileValidationServiceMock = jasmine.createSpyObj('fileValidationServiceMock', ['validateFileSize', 'validateFileType']);
            formatterMock = jasmine.createSpyObj('formatter', ['base64toBlob']);
            modalMock = jasmine.createSpyObj('$modal', ['open', 'close', 'result']);

            httpMock.$http.and.returnValue({
                then: function (callback) {
                    callback(videoResponse);
                }
            });

            timeoutMock.$timeout.and.callFake(function (callback) {
                timeoutCallback = callback;
            });

            videoServiceMock.getMediaRequest.and.callFake(function () {
                return mediaRequest;
            });

            videoServiceMock.getEvaluation.and.callFake(function () {
                return evaluation;
            });

            fileValidationServiceMock.validateFileSize.and.callFake(function () {
                return isfileSizeValid;
            });

            fileValidationServiceMock.validateFileType.and.callFake(function () {
                return isfileTypeValid;
            });

            formatterMock.base64toBlob.and.callFake(function () {
                return 'image-content';
            })

            videoServiceMock.saveEvaluation.and.returnValue({
                then: function (callback) {
                    callback();
                }
            });

            videoUploadServiceMock.uploadVideo.and.returnValue({
                then: function (callback) {
                    callback();
                }
            });

            modalMock.open.and.callFake(function(){
                return fakeModal;
            });

            spyOn(URL, 'createObjectURL').and.callFake(function () {
                return 'someurl';
            });

            spyOn(angular, 'element').and.callFake(function (selector) {
                return [{'src':'','load': function(){}}];
            });

            module(function ($provide) {
                $provide.value('$state', stateMock);
                $provide.value('$timeout', timeoutMock.$timeout);
                $provide.value('$http', httpMock.$http);
                $provide.value('focusService', focusServiceMock);
                $provide.value('videoService', videoServiceMock);
                $provide.value('fileValidationService', fileValidationServiceMock);
                $provide.value('videoUploadService', videoUploadServiceMock);
                $provide.value('formatter', formatterMock);
                $provide.value('$modal', modalMock);
                $provide.value('configServiceValues', configServiceValues);
            });

            inject(function($controller, $rootScope) {
                scope = $rootScope.$new();
                controller = $controller;
                rootScope = $rootScope;
            });
        });

        afterEach(function () {
            angular.element.and.callThrough();
        });

		describe("initial state", function () {
            beforeEach(function () {
                evaluation = {
                    images: [
                        {
                            id: 'has id',
                            imageUrl: 'file.mov'
                        }
                    ]
                }
            });

            it("should set currVideo to a default value if video is not set", function () {
                controller = controller('VideoUploadController', {$scope: scope});
                scope.$apply();
                expect(scope.currVideo).toEqual({});
            });

            it("should have the correct errorHandling object set", function () {
                controller = controller('VideoUploadController', {$scope: scope});
                scope.$apply();
                expect(scope.errorHandling).toEqual({
                    'File-Size-Check' : {
                        message : 'The video you’re attempting to upload is too large. Videos are limited to 20 MB. To reduce the video size, try taking a shorter video, changing the camera settings or using compression software.',
                        priority : 1
                    },
                    'File-Type-Check' : {
                        message : 'Videos must be one of the following types: mov or mp4',
                        priority : 1
                    }
                });
            });
        });

        describe("load previously uploaded video", function () {
            beforeEach(function () {
                evaluation = {
                    images: [
                        {
                            id: 'has id',
                            imageUrl: 'file.mov'
                        }
                    ]
                };
                scope.videoSelected = true;

                controller = controller('VideoUploadController', {$scope: scope});
                scope.$apply();
            });
            
            it("should make a call to get the blob object if imageUrl is set on the video object", function () {
                expect(httpMock.$http).toHaveBeenCalledWith({
                    url: 'file.mov',
                    method: 'GET'
                });
            });

            it("should call createObjectURL with the blob object", function () {
                expect(URL.createObjectURL).toHaveBeenCalledWith('image-content');
            });
        });

        describe("retakeVideo function", function () {
            beforeEach(function () {
                controller = controller('VideoUploadController', {$scope: scope});

                spyOn(scope, 'takeVideo');

                scope.videoSelected = true;

                scope.retakeVideo();
            });

            it("should set videoAccepted to false", function () {
                expect(scope.videoSelected).toEqual(false);
            });

            it("should call takeVideo", function () {
                expect(scope.takeVideo).toHaveBeenCalled();
            });
        });

        describe("takeVideo function", function () {
            it("should set videoSelected to true", function () {
                controller = controller('VideoUploadController', {$scope: scope});

                scope.takeVideo();
                expect(scope.videoSelected).toEqual(true);
            });

            it("should wait 700ms before setting focus", function () {
                controller = controller('VideoUploadController', {$scope: scope});
                
                scope.takeVideo();
                expect(timeoutMock.$timeout.calls.argsFor(0)[1]).toEqual(700);

                timeoutCallback();

                expect(focusServiceMock.focusElement).toHaveBeenCalledWith('.media-container span');
            });
        });

        describe("validateFileSize function", function () {
            beforeEach(function () {
                controller = controller('VideoUploadController', {$scope: scope});

                scope.videoInputForm = {
                    $setValidity: function () {}
                };

                spyOn(scope.videoInputForm, '$setValidity');
            });

            it("should validate successfully file size validation passes", function () {
                isfileSizeValid = true;

                scope.validateFileSize();

                expect(scope.videoInputForm.$setValidity).toHaveBeenCalledWith('File-Size-Check', true);
            });

            it("should not validate successfully if the file size validation fails", function () {
                isfileSizeValid = false;

                scope.validateFileSize();

                expect(scope.videoInputForm.$setValidity).toHaveBeenCalledWith('File-Size-Check', false);
            });
        });

        describe("validateFileType function", function () {
            var fileExt = '';

            beforeEach(function () {
                controller = controller('VideoUploadController', {$scope: scope});

                scope.videoInputForm = {
                    $setValidity: function () {}
                };

                spyOn(scope.videoInputForm, '$setValidity');
            });

            it('should use the fileName from the image object, if present', function () {
                scope.video = {
                    fileName: 'file.mov'
                };

                scope.validateFileType();

                expect(fileValidationServiceMock.validateFileType).toHaveBeenCalledWith(scope.video.fileName, fileTypes);
            });

            it('should use the name from the fileObj, if fileObj is present and image is not', function () {
                scope.video = undefined;
                scope.currVideo.fileObj = {
                    name: 'fileObj.mov'
                };

                scope.validateFileType();

                expect(fileValidationServiceMock.validateFileType).toHaveBeenCalledWith(scope.currVideo.fileObj.name, fileTypes);
            });

            it('should use the name from the fileObj if image doesn\'t have a fileName', function () {
                scope.video = {};
                scope.currVideo.fileObj = {
                    name: 'fileObj.mov'
                };

                scope.validateFileType();

                expect(fileValidationServiceMock.validateFileType).toHaveBeenCalledWith(scope.currVideo.fileObj.name, fileTypes);
            });

            it('should use an empty string if neither fileObj nor image are present', function () {
                scope.video = undefined;
                scope.currVideo.fileObj = undefined;

                scope.validateFileType();

                expect(fileValidationServiceMock.validateFileType).toHaveBeenCalledWith('', fileTypes);
            });

            it('should validate successfully if the file extension matches one of the valid types', function () {
                isfileTypeValid = true;
                
                fileExt = 'mov';

                scope.validateFileType();

                fileExt = 'mp4';

                scope.validateFileType();

                var calls = scope.videoInputForm.$setValidity.calls;

                expect(calls.argsFor(0)).toEqual(['File-Type-Check', true]);
                expect(calls.argsFor(1)).toEqual(['File-Type-Check', true]);
                
                expect(scope.videoInputForm.$setValidity).not.toHaveBeenCalledWith('File-Type-Check', false);
            });

            it('should not validate successfully if the file extension does not match one of the valid types', function () {
                isfileTypeValid = false;
                fileExt = 'bmp';

                scope.validateFileType();

                expect(scope.videoInputForm.$setValidity).toHaveBeenCalledWith('File-Type-Check', false);
            });
        });

        describe("validate function", function () {
            beforeEach(function () {
                controller = controller('VideoUploadController', {$scope: scope});

                spyOn(scope, 'validateFileSize');
                spyOn(scope, 'validateFileType');

                scope.validate();
            });

            it('should call the validateFileSize function', function () {
                expect(scope.validateFileSize).toHaveBeenCalled();
            });

            it('should call the validateFileType function', function () {
                expect(scope.validateFileType).toHaveBeenCalled();
            });
        });

        describe("acceptVideo function", function () {
            beforeEach(function () {
                controller = controller('VideoUploadController', {$scope: scope});

                spyOn(scope, 'validate');

                scope.videoInputForm = {
                    validationSummary: {
                        validate: function () {}
                    }
                };
                spyOn(scope.videoInputForm.validationSummary, 'validate').and.returnValue({
                    then: function (callback) {
                        callback();
                    }
                });
            });

            it("should call the validate function", function () {
                scope.acceptVideo();

                expect(scope.validate).toHaveBeenCalled();
            });

            it("should call the videoUploadService.uploadVideo function when validation is successful and open the video-submit page if there is a fileObj", function () {
                scope.currVideo = {
                    fileObj: {
                        src: 'file.mov'
                    }
                };

                scope.acceptVideo();

                expect(videoUploadServiceMock.uploadVideo).toHaveBeenCalled();
                expect(stateMock.go).toHaveBeenCalledWith('main.myvideo.video-submit');
            });

            it("should call the videoService.saveEvaluation function when validation is successful and open the video-submit page if there is no fileObj", function () {
                scope.currVideo = {};

                scope.acceptVideo();

                expect(videoServiceMock.saveEvaluation).toHaveBeenCalled();
                expect(stateMock.go).toHaveBeenCalledWith('main.myvideo.video-submit');
            });

            it("should not call the videoUploadService.uploadVideo function when validation is unsuccessful", function () {
                scope.currVideo = {
                    fileObj: {
                        src: 'file.mov'
                    }
                };
                
                scope.videoInputForm.validationSummary.validate.and.returnValue({
                    then: function (callback) {
                        //empty success block
                    }
                });

                scope.acceptVideo();

                expect(videoUploadServiceMock.uploadVideo).not.toHaveBeenCalled();
            });
        });

        describe("back function", function () {
            it('should open modal if videoSelected and infoUpdated are true', function () {
                controller = controller('VideoUploadController', {$scope: scope});
                scope.videoSelected = true;
                scope.infoUpdated = true;

                scope.back();

                expect(modalMock.open).toHaveBeenCalled();
            });

            it('should not open modal if videoSelected is false and infoUpdated is true and go back to video instructions page', function () {
                controller = controller('VideoUploadController', {$scope: scope});
                scope.videoSelected = false;
                scope.infoUpdated = true;

                scope.back();

                expect(modalMock.open).not.toHaveBeenCalled();
                expect(stateMock.go).toHaveBeenCalledWith('main.myvideo.video-instructions');
            });

            it('should not open modal if videoSelected is true and infoUpdated is false and go back to video instructions page', function () {
                controller = controller('VideoUploadController', {$scope: scope});
                scope.videoSelected = true;
                scope.infoUpdated = false;

                scope.back();

                expect(modalMock.open).not.toHaveBeenCalled();
                expect(stateMock.go).toHaveBeenCalledWith('main.myvideo.video-instructions');
            });
        });
    });
});