define(['mediaRequestProgressService'], function () {
    'use strict';

    describe('Media Request Progress Service', function () {
        var service,
            imageTypesBySection = [],
            uploadedImagesBySection = [],
            imageResponseServiceMock,
            imageMediaRequestServiceMock,
            mediaRequest = {},
            evaluation = {},
            patientServiceMock,
            patientMock = {
				firstName: 'John',
				lastName: 'Doe',
				dateOfBirth: '01/01/1980',
				icn: '1234'
            };

        beforeEach(function () {
            module('angularTemplateApp');

            imageResponseServiceMock = jasmine.createSpyObj('imageResponseService', ['getEvaluation', 'getUploadedImagesBySection']);
            imageMediaRequestServiceMock = jasmine.createSpyObj('imageMediaRequestService', ['getMediaRequest', 'getImageTypesBySection']);
            patientServiceMock = jasmine.createSpyObj('patient', ['fetch']);

            imageMediaRequestServiceMock.getMediaRequest.and.callFake(function () {
                return mediaRequest;
            });

            imageMediaRequestServiceMock.getImageTypesBySection.and.callFake(function () {
                return imageTypesBySection;
            });

            imageResponseServiceMock.getEvaluation.and.callFake(function () {
                return evaluation;
            });

            imageResponseServiceMock.getUploadedImagesBySection.and.callFake(function () {
                return uploadedImagesBySection;
            });

            patientServiceMock.fetch.and.returnValue(function () {
                return patientMock;
            });

            module(function ($provide) {
                $provide.value('imageMediaRequestService', imageMediaRequestServiceMock);
                $provide.value('imageResponseService', imageResponseServiceMock);
                $provide.value('patient', patientServiceMock);
            });

            inject(function (mediaRequestProgressService) {
                service = mediaRequestProgressService;
            });
        });

        describe('getNumCompleted', function () {
            var isMedTypesComplete = false;
            var completedMedsSteps = 0;
            var completedPhotoSteps = 0;

            beforeEach(function () {
                spyOn(service, 'isMedTypesComplete').and.callFake(function () {
                    return isMedTypesComplete;
                });
                spyOn(service, 'getCompletedMedsSteps').and.callFake(function () {
                    return completedMedsSteps;
                });
                spyOn(service, 'getCompletedPhotoSteps').and.callFake(function () {
                    return completedPhotoSteps;
                });
            });

            it('should return 0 if none of the pages have been completed', function () {
                expect(service.getNumCompleted()).toEqual(0);
            });

            it('should calculate the number of completed pages based on the evaluation object', function () {
                evaluation = {
                    skinStatus: 'status',
                    additionalComments: 'comments',
                    isResponsesAccepted: true
                };

                expect(service.getNumCompleted()).toEqual(3);
            });

            it('should calculate the number of completed pages based on values from other functions', function () {
                isMedTypesComplete = true;
                completedMedsSteps = 3;
                completedPhotoSteps = 5;

                expect(service.getNumCompleted()).toEqual(12);
            });
        });

        describe('isMedTypesComplete', function () {
            beforeEach(function () {
                evaluation = {
                    'medicationTypes': []
                };
            });

            it('should return false for a new evaluation', function () {
                expect(service.isMedTypesComplete()).toBeFalsy();
            });

            it('should return true if medication types have been selected and OTHER is not selected', function () {
                evaluation = {
                    medicationTypes: [
                        'TOPICAL',
                        'ORAL'
                    ]
                };

                expect(service.isMedTypesComplete()).toBeTruthy();
            });

            it('should return false if OTHER is selected, but no comments were provided', function () {
                evaluation = {
                    medicationTypes: [
                        'TOPICAL',
                        'ORAL',
                        'OTHER'
                    ]
                };

                expect(service.isMedTypesComplete()).toBeFalsy();
            });

            it('should return true if OTHER is selected and comments are provided', function () {
                evaluation = {
                    medicationTypes: [
                        'TOPICAL',
                        'ORAL',
                        'OTHER'
                    ],
                    otherMedication: 'comments'
                };

                expect(service.isMedTypesComplete()).toBeTruthy();
            });

            it('should return true if no medication types are selected and the comments read "N/A"', function () {
                evaluation = {
                    medicationTypes: [],
                    otherMedication: 'N/A'
                };

                expect(service.isMedTypesComplete()).toBeTruthy();
            });
        });

        describe('getCompletedMedsSteps', function () {
            var isMedIncomplete = true;

            beforeEach(function () {
                evaluation = {
                    medicationTypes: []
                };

                spyOn(service, 'isMedIncomplete').and.callFake(function (med) {
                    return !med.name;
                });
            });

            it('should return 0 if no medications or medication types are specified', function () {
                expect(service.getCompletedMedsSteps()).toEqual(0);
            });

            it('should return 0 if there are medications, but no medication types', function () {
                evaluation = {
                    medicationTypes: [],
                    medications: [
                        {
                            "type": "TOPICAL"
                        }
                    ]
                };

                expect(service.getCompletedMedsSteps()).toEqual(0);
            });

            it('should return a completed page for each medication type with completed medications', function () {
                isMedIncomplete = false;

                evaluation = {
                    medicationTypes: [
                        "TOPICAL",
                        "ORAL",
                        "INJECTED"
                    ],
                    medications: [
                        {
                            "type": "TOPICAL",
                            "name": "Eucerin"
                        },
                        {
                            "type": "ORAL",
                            "name": "Advil"
                        },
                        {
                            "type": "ORAL"
                        }
                    ]
                };

                expect(service.getCompletedMedsSteps()).toEqual(1);
            });
        });

        describe('isMedIncomplete', function () {
            it('should return true if the name field is missing', function () {
                var med = {
                    numDosesPerTimeUnit: 3,
                    timeUnit: 'day',
                    missedDoses: 1
                };

                expect(service.isMedIncomplete(med)).toBeTruthy();
            });

            it('should return true if the numDosesPerTimeUnit field is missing', function () {
                var med = {
                    name: 'Eucerin',
                    timeUnit: 'day',
                    missedDoses: 1
                };

                expect(service.isMedIncomplete(med)).toBeTruthy();
            });

            it('should return true if the timeUnit field is missing', function () {
                var med = {
                    name: 'Eucerin',
                    numDosesPerTimeUnit: 3,
                    missedDoses: 1
                };

                expect(service.isMedIncomplete(med)).toBeTruthy();
            });

            it('should return true if the missedDoses field is missing', function () {
                var med = {
                    name: 'Eucerin',
                    numDosesPerTimeUnit: 3,
                    timeUnit: 'day'
                };

                expect(service.isMedIncomplete(med)).toBeTruthy();
            });

            it('should return false if the med has all required fields', function () {
                var med = {
                    name: 'Eucerin',
                    numDosesPerTimeUnit: 3,
                    timeUnit: 'day',
                    missedDoses: 1
                };

                expect(service.isMedIncomplete(med)).toBeFalsy();
            });

            it('should return false if the med has all required fields, even if the numbers are 0', function () {
                var med = {
                    name: 'Eucerin',
                    numDosesPerTimeUnit: 0,
                    timeUnit: 'day',
                    missedDoses: 0
                };

                expect(service.isMedIncomplete(med)).toBeFalsy();
            });
        });

        describe('getCompletedPhotoSteps', function () {
            beforeEach(function () {
                evaluation = {};
                imageTypesBySection = [];
                uploadedImagesBySection = [];
            });

            it('should return 0 by default', function () {
                expect(service.getCompletedPhotoSteps()).toEqual(0);
            });

            it('should not return completed if no required photos have been taken', function () {
                imageTypesBySection = {
                    'HEAD': [
                        {
                            name: 'HEAD_FACE',
                            description: 'Head and Face'
                        },
                        {
                            name: 'HEAD_LEFT_FACE',
                            description: 'Left Side of Face'
                        }
                    ]
                };

                expect(service.getCompletedPhotoSteps()).toEqual(0);
            });

            it('should return completed based on how many required photos are present for each section', function () {
                imageTypesBySection = {
                    'HEAD': [
                        {
                            name: 'HEAD_FACE',
                            description: 'Head and Face'
                        },
                        {
                            name: 'HEAD_LEFT_FACE',
                            description: 'Left Side of Face'
                        }
                    ],
                    'HAND': [
                        {
                            name: 'HAND_PALM_UP',
                            description: 'Hands with Palms Up'
                        },
                        {
                            name: 'HAND_PALM_DOWN',
                            description: 'Hands with Palms Down'
                        }
                    ],
                    'ARM': [
                        {
                            name: 'ARM_FOREARM_UP',
                            description: 'Forearms Facing Up'
                        },
                        {
                            name: 'ARM_FOREARM_DOWN',
                            description: 'Forearms Facing Down'
                        }
                    ],
                    'FEET': [
                        {
                            name: 'FEET_TOP',
                            description: 'Top Feet'
                        },
                        {
                            name: 'FEET_BOTTOM',
                            description: 'Bottom Feet'
                        }
                    ]
                };

                uploadedImagesBySection = {
                    'HEAD': {
                        'PRIMARY': [
                            'image',
                            'image1'
                        ] 
                    },
                    'HAND': {
                        'PRIMARY': [
                            'image'
                        ]
                    },
                    'ARM': {}
                };

                expect(service.getCompletedPhotoSteps()).toEqual(3);
            });

            it('should return completed based on if close ups are present for each section', function () {
                imageTypesBySection = {
                    'HEAD': [
                        {
                            name: 'HEAD_FACE',
                            description: 'Head and Face'
                        },
                        {
                            name: 'HEAD_LEFT_FACE',
                            description: 'Left Side of Face'
                        }
                    ],
                    'HAND': [
                        {
                            name: 'HAND_PALM_UP',
                            description: 'Hands with Palms Up'
                        },
                        {
                            name: 'HAND_PALM_DOWN',
                            description: 'Hands with Palms Down'
                        }
                    ],
                    'ARM': [
                        {
                            name: 'ARM_FOREARM_UP',
                            description: 'Forearms Facing Up'
                        },
                        {
                            name: 'ARM_FOREARM_DOWN',
                            description: 'Forearms Facing Down'
                        }
                    ],
                    'FEET': [
                        {
                            name: 'FEET_TOP',
                            description: 'Top Feet'
                        },
                        {
                            name: 'FEET_BOTTOM',
                            description: 'Bottom Feet'
                        }
                    ],
                    'LOWER_LEG': [
                        {
                            name: 'LOWER_LEG_LL_BACK',
                            description: 'Back Side of Lower Legs'
                        },
                        {
                            name: 'LOWER_LEG_LL_FRONT',
                            description: 'Front Side of Lower Legs'
                        }
                    ]
                };

                uploadedImagesBySection = {
                    'HEAD': {
                        'PRIMARY': [
                            'image',
                            'image1'
                        ],
                        'CLOSE_UP': [
                            {
                                id: 'someImage',
                                imageUrl: 'http://some-url/image-url'
                            },
                            {
                                imageClass: 'CLOSE_UP'  
                            },
                            {
                                imageClass: 'CLOSE_UP'
                            }
                        ] 
                    },
                    'HAND': {
                        'PRIMARY': [
                            'image',
                            'image1'
                        ],
                        'CLOSE_UP': [
                            {
                                id: 'someImage',
                                imageUrl: 'http://some-url/image-url'
                            }
                        ] 
                    },
                    'ARM': {
                        'PRIMARY': [
                            'image',
                            'image1'
                        ],
                        'CLOSE_UP': []
                    }
                };

                expect(service.getCompletedPhotoSteps()).toEqual(8);
            });

            it('should return completed if the user decided not to take optional photos', function () {
                evaluation = {
                    isOptionalImages: false
                };

                expect(service.getCompletedPhotoSteps()).toEqual(1);
            });

            it('should not return completed if the user decided to take optional photos but no optional photos are present', function () {
                evaluation = {
                    isOptionalImages: true
                };

                expect(service.getCompletedPhotoSteps()).toEqual(0);
            });

            it('should return completed if the user decided to take optional photos and optional photos are present', function () {
                evaluation = {
                    isOptionalImages: false
                };

                uploadedImagesBySection = {
                    'OPTIONAL': [
                        'image'
                    ]
                }

                expect(service.getCompletedPhotoSteps()).toEqual(1);
            });
        });

        describe('getTotalSteps', function () {
            beforeEach(function () {
                evaluation = {};
                imageTypesBySection = [];
            });

            it('should return 5 with no other options set', function () {
                expect(service.getTotalSteps()).toEqual(5);
            });

            it('should calculate the total pages based on the number of image types', function () {
                mediaRequest = {
                    mediaTypes: [
                        {
                            name: 'HEAD_FACE',
                            description: 'Head and Face'
                        },
                        {
                            name: 'HEAD_LEFT_FACE',
                            description: 'Left Side of Face'
                        }
                    ]
                };

                expect(service.getTotalSteps()).toEqual(7);
            });

            it('should calculate the total pages based on the number of image sections', function () {
                mediaRequest = {
                    mediaTypes: [
                        {
                            name: 'HEAD_FACE',
                            description: 'Head and Face'
                        },
                        {
                            name: 'HEAD_LEFT_FACE',
                            description: 'Left Side of Face'
                        }
                    ]
                };

                imageTypesBySection = {'HEAD': []};

                expect(service.getTotalSteps()).toEqual(8);
            });

            it('should calculate the total pages based on the number of medication types that aren\'t OTHER', function () {
                mediaRequest = {
                    mediaTypes: [
                        {
                            name: 'HEAD_FACE',
                            description: 'Head and Face'
                        },
                        {
                            name: 'HEAD_LEFT_FACE',
                            description: 'Left Side of Face'
                        }
                    ]
                };

                evaluation = {
                    medicationTypes: [
                        'TOPICAL',
                        'ORAL',
                        'OTHER'
                    ]
                };

                imageTypesBySection = {'HEAD': []};

                expect(service.getTotalSteps()).toEqual(10);
            });
        });
    });
});