define(['mediaRequestNavigationService'], function () {
    'use strict';

    describe('Media Request Navigation Service', function () {
        var service,
            q,
            stateMock,
            stateParamsMock = {},
            imageMediaRequestServiceMock,
            imageResponseServiceMock,
            medicationTypesMock = [
                'TOPICAL',
                'ORAL',
                'INJECTED'
            ],
            mediaRequest = {},
            evaluation = {},
            imageTypesBySection = {},
            uploadedImagesBySection = {},
            removeItemsFromArray = function (array) {
                var arrayLength = array.length;

                for (var i = 0; i < arrayLength; i++) {
                    array.pop();
                }
            };

        beforeEach(function () {
            module('angularTemplateApp');

            stateMock = jasmine.createSpyObj('$state', ['go']);
            imageMediaRequestServiceMock = jasmine.createSpyObj('imageMediaRequestService', ['getMediaRequest', 'saveMediaRequest', 'setQuestionnaireResponseId', 'getImageTypesBySection', 'getImageSectionsMap']);
            imageResponseServiceMock = jasmine.createSpyObj('imageResponseService', ['getEvaluation', 'saveEvaluation', 'setEvaluation' ,'getUploadedImagesBySection']);

            imageMediaRequestServiceMock.getMediaRequest.and.callFake(function () {
                return mediaRequest;
            });

            imageMediaRequestServiceMock.saveMediaRequest.and.returnValue({
                then: function (callback) {
                    callback();
                }
            });

            imageMediaRequestServiceMock.getImageTypesBySection.and.callFake(function () {
                return imageTypesBySection;
            });

            imageMediaRequestServiceMock.getImageSectionsMap.and.returnValue({
                'HEAD': {
                    name: 'HEAD',
                    description: 'Head'
                },
                'HAND': {
                    name: 'HAND',
                    description: 'Hand'
                },
                'FEET': {
                    name: 'FEET',
                    description: 'Feet'
                }
            });

            imageResponseServiceMock.getEvaluation.and.callFake(function () {
                return evaluation;
            });

            imageResponseServiceMock.saveEvaluation.and.returnValue({
                then: function (callback) {
                    callback();
                }
            });

            imageResponseServiceMock.getUploadedImagesBySection.and.callFake(function () {
                return uploadedImagesBySection;
            });

            module(function($provide) {
                $provide.value('$state', stateMock);
                $provide.value('$stateParams', stateParamsMock);
                $provide.value('imageMediaRequestService', imageMediaRequestServiceMock);
                $provide.value('imageResponseService', imageResponseServiceMock);
                $provide.value('medicationTypes', medicationTypesMock);
            });

            inject(function ($q, mediaRequestNavigationService) {
                q = $q;
                service = mediaRequestNavigationService;
            });
        });

        describe('buildArrayFromArgs function', function () {
            it('should return an empty array if nothing is passed in', function () {
                expect(service.buildArrayFromArgs()).toEqual([]);
            });

            it('should return an empty array if an empty array is passed in', function () {
                expect(service.buildArrayFromArgs([])).toEqual([]);
            });

            it('should return a new array that is separate from the original array', function () {
                var oldArray = ['one', 'two', 'three'];
                var newArray = service.buildArrayFromArgs(oldArray);
                
                newArray.push('four');

                expect(oldArray).toEqual(['one', 'two', 'three']);
                expect(newArray).toEqual(['one', 'two', 'three', 'four']);
            });

            it('should take an arguments array and return a regular array', function () {
                var sampleFunction = function () {
                    return service.buildArrayFromArgs(arguments);
                };

                var newArray = sampleFunction('one', 'two', 'three');

                expect(typeof newArray.forEach).toEqual('function');
            });
        });

        describe('previousPage function', function () {
            beforeEach(function () {
                spyOn(service, 'generateListOfPages');
                spyOn(service, 'goToPage');
                spyOn(service, 'resetPageNumber');
            });

            it('should regenerate the list of pages', function () {
                evaluation = { 
                    pageNumber: 1
                };

                service.previousPage();

                expect(service.generateListOfPages).toHaveBeenCalled();
            });

            it('should reset the page number', function () {
                evaluation = { 
                    pageNumber: 1
                };

                service.previousPage();

                expect(service.resetPageNumber).toHaveBeenCalled();
            });

            it('should not navigate the user to the previous page if on the first page', function () {
                evaluation = { 
                    pageNumber: 0
                };

                service.previousPage();

                expect(service.goToPage).not.toHaveBeenCalled();
            });

            it('should not save the evaluation if the user isn\'t navigated', function () {
                evaluation = { 
                    pageNumber: 0
                };

                service.previousPage();

                expect(imageResponseServiceMock.saveEvaluation).not.toHaveBeenCalled();
            });

            it('should navigate the user to the previous page if not on the first page', function () {
                evaluation = { 
                    pageNumber: 1
                };

                service.previousPage();

                expect(service.goToPage).toHaveBeenCalledWith(0);
            });

            it('should save the evaluation if the user is navigated', function () {
                evaluation = { 
                    pageNumber: 1
                };

                service.previousPage();

                expect(imageResponseServiceMock.saveEvaluation).toHaveBeenCalled();
            });
        });

        describe('nextPage function', function () {
            var pages;

            beforeEach(function () {
                spyOn(service, 'generateListOfPages');
                spyOn(service, 'goToPage');
                spyOn(service, 'resetPageNumber');

                pages = service.getPagesList();

                pages.push('page1');
                pages.push('page2');
                pages.push('page3');
            });

            afterEach(function () {
                pages = service.getPagesList();

                removeItemsFromArray(pages);
            });

            it('should regenerate the list of pages', function () {
                evaluation = { 
                    pageNumber: 1
                };

                service.nextPage();

                expect(service.generateListOfPages).toHaveBeenCalled();
            });

            it('should reset the page number', function () {
                evaluation = { 
                    pageNumber: 1
                };

                service.previousPage();

                expect(service.resetPageNumber).toHaveBeenCalled();
            });

            it('should not navigate the user to the next page if on the last page', function () {
                evaluation = { 
                    pageNumber: 2
                };

                service.nextPage();

                expect(service.goToPage).not.toHaveBeenCalled();
            });

            it('should not save the evaluation if the user isn\'t navigated', function () {
                evaluation = { 
                    pageNumber: 2
                };

                service.nextPage();

                expect(imageResponseServiceMock.saveEvaluation).not.toHaveBeenCalled();
            });

            it('should navigate the user to the next page if not on the last page', function () {
                evaluation = { 
                    pageNumber: 1
                };

                service.nextPage();

                expect(service.goToPage).toHaveBeenCalledWith(2);
            });

            it('should save the evaluation if the user is navigated and the saveEval param is undefined', function () {
                evaluation = { 
                    pageNumber: 1
                };

                service.nextPage();

                expect(imageResponseServiceMock.saveEvaluation).toHaveBeenCalled();
            });

            it('should save the evaluation if the user is navigated and the saveEval param is set to true', function () {
                evaluation = { 
                    pageNumber: 1
                };

                service.nextPage(true);

                expect(imageResponseServiceMock.saveEvaluation).toHaveBeenCalled();
            });

            it('should not save the evaluation if the user is navigated and the saveEval param is set to false', function () {
                evaluation = { 
                    pageNumber: 1
                };

                service.nextPage(false);

                expect(imageResponseServiceMock.saveEvaluation).not.toHaveBeenCalled();
            });
        });

        describe('skipToNextSection function', function () {
            beforeEach(function () {
                spyOn(service, 'skipToRoute');
            });

            it('should call skipToRoute with the possible routes to skip to', function () {
                service.skipToNextSection();

                expect(service.skipToRoute).toHaveBeenCalledWith('main.mytelederm.photo-input', 'main.mytelederm.take-another-picture', 'main.mytelederm.photo-input-optional');
            });
        });

        describe('skipToReviewPhotos function', function () {
            beforeEach(function () {
                spyOn(service, 'skipToRoute');
            });

            it('should call skipToRoute with the review-photos route', function () {
                service.skipToReviewPhotos();

                expect(service.skipToRoute).toHaveBeenCalledWith('main.mytelederm.review-photos');
            });
        });

        describe('skipToRoute function', function () {
            beforeEach(function () {
                spyOn(service, 'generateListOfPages');
                spyOn(service, 'goToPage');

                var pages = service.getPagesList();

                pages.push({
                    route: 'main.mytelederm.skin-status'
                });
                pages.push({
                    route: 'main.mytelederm.medications'
                });
                pages.push({
                    route: 'main.mytelederm.medications-entry'
                });
                pages.push({
                    route: 'main.mytelederm.additional-comments'
                });
                pages.push({
                    route: 'main.mytelederm.response-summary'
                });
                pages.push({
                    route: 'main.mytelederm.photo-instructions'
                });
                pages.push({
                    route: 'main.mytelederm.photo-input'
                });
                pages.push({
                    route: 'main.mytelederm.photo-input'
                });
                pages.push({
                    route: 'main.mytelederm.photo-input-close-up'
                });
                pages.push({
                    route: 'main.mytelederm.photo-input-close-up'
                });
                pages.push({
                    route: 'main.mytelederm.photo-input'
                });
                pages.push({
                    route: 'main.mytelederm.photo-input'
                });
                pages.push({
                    route: 'main.mytelederm.photo-input-close-up'
                });
                pages.push({
                    route: 'main.mytelederm.photo-input-close-up'
                });
                pages.push({
                    route: 'main.mytelederm.photo-input-optional'
                });
                pages.push({
                    route: 'main.mytelederm.photo-input-optional'
                });
                pages.push({
                    route: 'main.mytelederm.review-photos'
                });
                pages.push({
                    route: 'main.mytelederm.submit'
                });
            });

            afterEach(function () {
                var pages = service.getPagesList();

                removeItemsFromArray(pages);
            });

            it('should regenerate the list of pages', function () {
                evaluation = { 
                    pageNumber: 0
                };

                service.skipToRoute('main.mytelederm.review-photos');

                expect(service.generateListOfPages).toHaveBeenCalled();
            });

            it('should not change the page number if the pages list is empty', function () {
                evaluation = { 
                    pageNumber: 3
                };

                var pages = service.getPagesList();

                removeItemsFromArray(pages);

                service.skipToRoute('main.mytelederm.review-photos');

                expect(service.goToPage).toHaveBeenCalledWith(3);
                expect(evaluation.pageNumber).toEqual(3);
            });

            it('should not change the page number if the route is not found', function () {
                evaluation = { 
                    pageNumber: 3
                };

                service.skipToRoute('main.mytelederm.nonexistent');

                expect(service.goToPage).toHaveBeenCalledWith(3);
                expect(evaluation.pageNumber).toEqual(3);
            });

            it('should not change the page number if none of the routes are found', function () {
                evaluation = { 
                    pageNumber: 3
                };

                service.skipToRoute('main.mytelederm.nonexistent', 'main.mytelederm.fakeRoute');

                expect(service.goToPage).toHaveBeenCalledWith(3);
                expect(evaluation.pageNumber).toEqual(3);
            });

            it('should update the pageNumber to the given route is found', function () {
                evaluation = { 
                    pageNumber: 15
                };

                service.skipToRoute('main.mytelederm.review-photos');

                expect(service.goToPage).toHaveBeenCalledWith(16);
                expect(evaluation.pageNumber).toEqual(16);
            });

            it('should update the pageNumber to the next given route after the current pageNumber', function () {
                evaluation = { 
                    pageNumber: 8
                };

                service.skipToRoute('main.mytelederm.photo-input');

                expect(service.goToPage).toHaveBeenCalledWith(10);
                expect(evaluation.pageNumber).toEqual(10);
            });



            it('should update the pageNumber to the next given route that matches any of the given routes', function () {
                evaluation = { 
                    pageNumber: 8
                };

                service.skipToRoute('main.mytelederm.photo-input', 'main.mytelederm.take-another-picture', 'main.mytelederm.photo-input-optional');

                expect(service.goToPage).toHaveBeenCalledWith(10);
                expect(evaluation.pageNumber).toEqual(10);

                evaluation = { 
                    pageNumber: 12
                };

				service.skipToRoute('main.mytelederm.photo-input', 'main.mytelederm.take-another-picture');

				expect(service.goToPage).toHaveBeenCalledWith(12);
				expect(evaluation.pageNumber).toEqual(12);

				service.skipToRoute('main.mytelederm.photo-input', 'main.mytelederm.take-another-picture', 'main.mytelederm.photo-input-optional');

				expect(service.goToPage).toHaveBeenCalledWith(14);
				expect(evaluation.pageNumber).toEqual(14);
			});


			it('should not update the pageNumber if no matches of the given routes are found', function () {

				evaluation = { 
                    pageNumber: 12
                };

				service.skipToRoute('main.mytelederm.photo-input', 'main.mytelederm.take-another-picture');
				expect(service.goToPage).toHaveBeenCalledWith(12);
				expect(evaluation.pageNumber).toEqual(12);
			});

			it('should update the page number to the first matched route', function () {

				var pages = service.getPagesList();
				pages.splice(13 ,0,  { route: 'main.mytelederm.take-another-picture'});

				evaluation = { 
                    pageNumber: 12
                };

				service.skipToRoute('main.mytelederm.photo-input', 'main.mytelederm.take-another-picture', 'main.mytelederm.photo-input-optional');

				expect(service.goToPage).toHaveBeenCalledWith(13);
				expect(evaluation.pageNumber).toEqual(13);
			})
		});

        describe('resume function', function () {
            beforeEach(function () {
                spyOn(service, 'generateListOfPages');
                spyOn(service, 'goToPage');
            });

            it('should regenerate the list of pages and navigate the user', function () {
                evaluation = { 
                    pageNumber: 1
                };

                service.resume();

                expect(service.generateListOfPages).toHaveBeenCalled();
                expect(service.goToPage).toHaveBeenCalledWith(1);
            });
        });

        describe('generateListOfPages function', function () {
            var pages;
            var medicationEntryPages = [];
            var photoInputPages = [];
            var optionalPhotoPages = [];

            beforeEach(function () {
                spyOn(service, 'generateMedicationEntryPages').and.callFake(function () {
                    return medicationEntryPages;
                });
                spyOn(service, 'generatePhotoInputPages').and.callFake(function () {
                    return photoInputPages;
                });
                spyOn(service, 'generateOptionalPhotoPages').and.callFake(function () {
                    return optionalPhotoPages;
                });
            });

            afterEach(function () {
                pages = service.getPagesList();

                removeItemsFromArray(pages);
            });

            it('should return a default list of pages', function () {
                service.generateListOfPages();

                pages = service.getPagesList();

                expect(pages).toEqual([
                    {
                        route: 'main.mytelederm.skin-status'
                    },
                    {
                        route: 'main.mytelederm.medications'
                    },
                    {
                        route: 'main.mytelederm.additional-comments'
                    },
                    {
                        route: 'main.mytelederm.response-summary'
                    },
                    {
                        route: 'main.mytelederm.photo-instructions'
                    },
                    {
                        route: 'main.mytelederm.review-photos'
                    },
                    {
                        route: 'main.mytelederm.submit'
                    }
                ]);
            });

            it('should generate the list based on the results of other functions', function () {
                medicationEntryPages = [
                    {
                        route: 'main.mytelederm.medication-entry',
                        params: {
                            medType: 'TOPICAL'
                        }
                    }
                ];

                photoInputPages = [
                    {
                        route: 'main.mytelederm.photo-input',
                        params: {
                            imageSection: 'HEAD',
                            imageType: 'HEAD_FACE'
                        }
                    }
                ];

                optionalPhotoPages = [
                    {
                        route: 'main.mytelederm.photo-input-optional',
                        params: {
                            imageNum: 0
                        }
                    }
                ];

                service.generateListOfPages();

                pages = service.getPagesList();

                expect(pages).toEqual([
                    {
                        route: 'main.mytelederm.skin-status'
                    },
                    {
                        route: 'main.mytelederm.medications'
                    },
                    {
                        route: 'main.mytelederm.medication-entry',
                        params: {
                            medType: 'TOPICAL'
                        }
                    },
                    {
                        route: 'main.mytelederm.additional-comments'
                    },
                    {
                        route: 'main.mytelederm.response-summary'
                    },
                    {
                        route: 'main.mytelederm.photo-instructions'
                    },
                    {
                        route: 'main.mytelederm.photo-input',
                        params: {
                            imageSection: 'HEAD',
                            imageType: 'HEAD_FACE'
                        }
                    },
                    {
                        route: 'main.mytelederm.photo-input-optional',
                        params: {
                            imageNum: 0
                        }
                    },
                    {
                        route: 'main.mytelederm.review-photos'
                    },
                    {
                        route: 'main.mytelederm.submit'
                    }
                ]);
            });

            it('should generate a fresh list each time', function () {
                medicationEntryPages = [];

                photoInputPages = [];

                optionalPhotoPages = [];

                service.generateListOfPages();

                pages = service.getPagesList();

                expect(pages).toEqual([
                    {
                        route: 'main.mytelederm.skin-status'
                    },
                    {
                        route: 'main.mytelederm.medications'
                    },
                    {
                        route: 'main.mytelederm.additional-comments'
                    },
                    {
                        route: 'main.mytelederm.response-summary'
                    },
                    {
                        route: 'main.mytelederm.photo-instructions'
                    },
                    {
                        route: 'main.mytelederm.review-photos'
                    },
                    {
                        route: 'main.mytelederm.submit'
                    }
                ]);

                medicationEntryPages = [
                    {
                        route: 'main.mytelederm.medication-entry',
                        params: {
                            medType: 'TOPICAL'
                        }
                    }
                ];

                service.generateListOfPages();

                pages = service.getPagesList();

                expect(pages).toEqual([
                    {
                        route: 'main.mytelederm.skin-status'
                    },
                    {
                        route: 'main.mytelederm.medications'
                    },
                    {
                        route: 'main.mytelederm.medication-entry',
                        params: {
                            medType: 'TOPICAL'
                        }
                    },
                    {
                        route: 'main.mytelederm.additional-comments'
                    },
                    {
                        route: 'main.mytelederm.response-summary'
                    },
                    {
                        route: 'main.mytelederm.photo-instructions'
                    },
                    {
                        route: 'main.mytelederm.review-photos'
                    },
                    {
                        route: 'main.mytelederm.submit'
                    }
                ]);
            });
        });

        describe('generateMedicationEntryPages function', function () {
            it('should generate pages for each medication type selected, except OTHER', function () {
                evaluation = {
                    medicationTypes: [
                        'TOPICAL',
                        'ORAL',
                        'INJECTED',
                        'OTHER'
                    ]
                };

                expect(service.generateMedicationEntryPages()).toEqual([
                    {
                        route: 'main.mytelederm.medications-entry',
                        params: {
                            medType: 'TOPICAL'
                        }
                    },
                    {
                        route: 'main.mytelederm.medications-entry',
                        params: {
                            medType: 'ORAL'
                        }
                    },
                    {
                        route: 'main.mytelederm.medications-entry',
                        params: {
                            medType: 'INJECTED'
                        }
                    }
                ]);
            });

            it('should generate pages for each medication type selected in the correct order', function () {
                evaluation = {
                    medicationTypes: [
                        'ORAL',
                        'TOPICAL'
                    ]
                };

                expect(service.generateMedicationEntryPages()).toEqual([
                    {
                        route: 'main.mytelederm.medications-entry',
                        params: {
                            medType: 'TOPICAL'
                        }
                    },
                    {
                        route: 'main.mytelederm.medications-entry',
                        params: {
                            medType: 'ORAL'
                        }
                    }
                ]);
            });
        });

        describe('generatePhotoInputPages', function () {
            it('should generate the required photo and close up pages based on the imageTypes requested and the close ups already taken', function () {
                imageTypesBySection = {
                    'HEAD': [
                        {
                            name: 'HEAD_FACE',
                            description: 'Head and Face'
                        },
                        {
                            name: 'HEAD_LEFT_FACE',
                            description: 'Left Side of Face'
                        }
                    ],
                    'HAND': [
                        {
                            name: 'HAND_PALM_UP',
                            description: 'Hands with Palms Up'
                        },
                        {
                            name: 'HAND_PALM_DOWN',
                            description: 'Hands with Palms Down'
                        }
                    ],
                    'FEET': [
                        {
                            name: 'FEET_FEET_TOP',
                            description: 'Top Feet'
                        },
                        {
                            name: 'FEET_FEET_BOTTOM',
                            description: 'Bottom Feet'
                        }
                    ]
                };

                uploadedImagesBySection = {
                    'HEAD': {
                        'PRIMARY': [],
                        'CLOSE_UP': [
                            {
                                imageSection: {
                                    name: 'HEAD',
                                    description: 'Head'
                                },
                                imageType: {
                                    name: 'OTHER',
                                    description: 'Other'
                                },
                                imageClass: 'CLOSE_UP'
                            }
                        ]
                    },
                    'HAND': {
                        'PRIMARY': [],
                        'CLOSE_UP': [
                            {
                                imageSection: {
                                    name: 'HAND',
                                    description: 'Hand'
                                },
                                imageType: {
                                    name: 'OTHER',
                                    description: 'Other'
                                },
                                imageClass: 'CLOSE_UP'
                            },
                            {
                                imageSection: {
                                    name: 'HAND',
                                    description: 'Hand'
                                },
                                imageType: {
                                    name: 'OTHER',
                                    description: 'Other'
                                },
                                imageClass: 'CLOSE_UP'
                            },
                            {
                                imageSection: {
                                    name: 'HAND',
                                    description: 'Hand'
                                },
                                imageType: {
                                    name: 'OTHER',
                                    description: 'Other'
                                },
                                imageClass: 'CLOSE_UP'
                            }
                        ]
                    },
                    'FEET': {
                        'PRIMARY': [],
                        'CLOSE_UP': []
                    }
                };

                expect(service.generatePhotoInputPages()).toEqual([
                    {
                        route: 'main.mytelederm.photo-input',
                        params: {
                            imageSection: {
                                name: 'HEAD',
                                description: 'Head'
                            },
                            imageType: {
                                name: 'HEAD_FACE',
                                description: 'Head and Face'
                            },
                            imageNum: 1,
                            totalImagesInSection: 2
                        }
                    },
                    {
                        route: 'main.mytelederm.photo-input',
                        params: {
                            imageSection: {
                                name: 'HEAD',
                                description: 'Head'
                            },
                            imageType: {
                                name: 'HEAD_LEFT_FACE',
                                description: 'Left Side of Face'
                            },
                            imageNum: 2,
                            totalImagesInSection: 2
                        }
                    },
                    {
                        route: 'main.mytelederm.photo-input-close-up',
                        params: {
                            imageSection: {
                                name: 'HEAD',
                                description: 'Head'
                            },
                            imageNum: 1
                        }
                    },
                    {
                        route: 'main.mytelederm.photo-input',
                        params: {
                            imageSection: {
                                name: 'HAND',
                                description: 'Hand'
                            },
                            imageType: {
                                name: 'HAND_PALM_UP',
                                description: 'Hands with Palms Up'
                            },
                            imageNum: 1,
                            totalImagesInSection: 2
                        }
                    },
                    {
                        route: 'main.mytelederm.photo-input',
                        params: {
                            imageSection: {
                                name: 'HAND',
                                description: 'Hand'
                            },
                            imageType: {
                                name: 'HAND_PALM_DOWN',
                                description: 'Hands with Palms Down'
                            },
                            imageNum: 2,
                            totalImagesInSection: 2
                        }
                    },
                    {
                        route: 'main.mytelederm.photo-input-close-up',
                        params: {
                            imageSection: {
                                name: 'HAND',
                                description: 'Hand'
                            },
                            imageNum: 1
                        }
                    },
                    {
                        route: 'main.mytelederm.photo-input-close-up',
                        params: {
                            imageSection: {
                                name: 'HAND',
                                description: 'Hand'
                            },
                            imageNum: 2
                        }
                    },
                    {
                        route: 'main.mytelederm.photo-input-close-up',
                        params: {
                            imageSection: {
                                name: 'HAND',
                                description: 'Hand'
                            },
                            imageNum: 3
                        }
                    },
                    {
                        route: 'main.mytelederm.photo-input',
                        params: {
                            imageSection: {
                                name: 'FEET',
                                description: 'Feet'
                            },
                            imageType: {
                                name: 'FEET_FEET_TOP',
                                description: 'Top Feet',
                            },
                            imageNum: 1,
                            totalImagesInSection: 2
                        }
                    },
                    {
                        route: 'main.mytelederm.photo-input',
                        params: {
                            imageSection: {
                                name: 'FEET',
                                description: 'Feet'
                            },
                            imageType:{
                                name: 'FEET_FEET_BOTTOM',
                                description: 'Bottom Feet',
                            },
                            imageNum: 2,
                            totalImagesInSection: 2
                        }
                    },
                    {
                        route: 'main.mytelederm.photo-input-close-up',
                        params: {
                            imageSection: {
                                name: 'FEET',
                                description: 'Feet'
                            },
                            imageNum: 1
                        }
                    }
                ]);
            });

            it('should return the photo-input pages even if no images have been uploaded', function () {
                imageTypesBySection = {
                    'HEAD': [
                        {
                            name: 'HEAD_FACE',
                            description: 'Head and Face'
                        },
                        {
                            name: 'HEAD_LEFT_FACE',
                            description: 'Left Side of Face'
                        }
                    ]
                };

                uploadedImagesBySection = {
                    'HEAD': {
                        'PRIMARY': [],
                        'CLOSE_UP': []
                    }
                };

                expect(service.generatePhotoInputPages()).toEqual([
                    {
                        route: 'main.mytelederm.photo-input',
                        params: {
                            imageSection: {
                                name: 'HEAD',
                                description: 'Head'
                            },
                            imageType: {
                                name: 'HEAD_FACE',
                                description: 'Head and Face'
                            },
                            imageNum: 1,
                            totalImagesInSection: 2
                        }
                    },
                    {
                        route: 'main.mytelederm.photo-input',
                        params: {
                            imageSection: {
                                name: 'HEAD',
                                description: 'Head'
                            },
                            imageType: {
                                name: 'HEAD_LEFT_FACE',
                                description: 'Left Side of Face'
                            },
                            imageNum: 2,
                            totalImagesInSection: 2
                        }
                    },
                    {
                        route: 'main.mytelederm.photo-input-close-up',
                        params: {
                            imageSection: {
                                name: 'HEAD',
                                description: 'Head'
                            },
                            imageNum: 1
                        }
                    }
                ]);
            });

            it('should return the photo-input pages if no close-up images have been uploaded', function () {

                imageTypesBySection = {
                    'HEAD': [
                        {
                            name: 'HEAD_FACE',
                            description: 'Head and Face'
                        },
                        {
                            name: 'HEAD_LEFT_FACE',
                            description: 'Left Side of Face'
                        }
                    ]
                };

                uploadedImagesBySection = {
                    'HEAD': {
                        'PRIMARY': [
                            {
                                imageSection: {
                                    name: 'HEAD',
                                    description: 'Head'
                                },
                                imageType: {
                                    name: 'HEAD_FACE',
                                    description: 'Head and Face'
                                },
                                imageClass: 'PRIMARY'
                            }
                        ],
                        'CLOSE_UP': []
                    }
                };

                expect(service.generatePhotoInputPages()).toEqual([
                    {
                        route: 'main.mytelederm.photo-input',
                        params: {
                            imageSection: {
                                name: 'HEAD',
                                description: 'Head'
                            },
                            imageType: {
                                name: 'HEAD_FACE',
                                description: 'Head and Face'
                            },
                            imageNum: 1,
                            totalImagesInSection: 2
                        }
                    },
                    {
                        route: 'main.mytelederm.photo-input',
                        params: {
                            imageSection: {
                                name: 'HEAD',
                                description: 'Head'
                            },
                            imageType: {
                                name: 'HEAD_LEFT_FACE',
                                description: 'Left Side of Face'
                            },
                            imageNum: 2,
                            totalImagesInSection: 2
                        }
                    },
                    {
                        route: 'main.mytelederm.photo-input-close-up',
                        params: {
                            imageSection: {
                                name: 'HEAD',
                                description: 'Head'
                            },
                            imageNum: 1
                        }
                    }
                ]);
            });
        });

        describe('generateOptionalPhotoPages', function () {
            it('should return only take another page if isOptionalImages is false', function () {
                uploadedImagesBySection = {};

                evaluation = {
                    isOptionalImages: false
                };

                expect(service.generateOptionalPhotoPages()).toEqual([{ route: 'main.mytelederm.take-another-picture' }]);
            });


            it('should return two pages if isOptionalImages is true and no images have been uploaded', function () {
                uploadedImagesBySection = {};

                evaluation = {
                    isOptionalImages: true
                };

                expect(service.generateOptionalPhotoPages()).toEqual([
					{
						route: 'main.mytelederm.take-another-picture'
					},
					{
                        route: 'main.mytelederm.photo-input-optional',
                        params: {
                            imageNum: 1
                        }
                    }
                ]);
            });

            it('should return the number of pages equal to the number of uploaded images if isOptionalImages is true', function () {
                uploadedImagesBySection = {
                    'OPTIONAL': [
                        {
                            imageSection: {
                                name: 'OTHER',
                                description: 'Other'
                            },
                            imageType: {
                                name: 'OTHER',
                                description: 'Other'
                            },
                            imageClass: 'OPTIONAL'
                        },
                        {
                            imageSection: {
                                name: 'OTHER',
                                description: 'Other'
                            },
                            imageType: {
                                name: 'OTHER',
                                description: 'Other'
                            },
                            imageClass: 'OPTIONAL'
                        }
                    ]
                };

                evaluation = {
                    isOptionalImages: true
                };

                expect(service.generateOptionalPhotoPages()).toEqual([
                    {
                        route: 'main.mytelederm.photo-input-optional',
                        params: {
                            imageNum: 1
                        }
                    },
                    {
                        route: 'main.mytelederm.photo-input-optional',
                        params: {
                            imageNum: 2
                        }
                    }
                ]);
            });
        });

        describe('goToPage function', function () {
            var pages;

            beforeEach(function () {
                pages = service.getPagesList();

                pages.push({
                    route: 'main.page1'
                });

                pages.push({
                    route: 'main.page2',
                    params: {
                        type: 'param page'
                    }
                });
            });

            afterEach(function () {
                pages = service.getPagesList();
                
                removeItemsFromArray(pages);
            });

            it('should navigate the user to the given page number', function () {
                service.goToPage(0);

                expect(stateMock.go).toHaveBeenCalledWith('main.page1', undefined);
            });

            it('should pass the correct parameters for the given page number', function () {
                service.goToPage(1);

                expect(stateMock.go).toHaveBeenCalledWith('main.page2', {
                    type: 'param page'
                });
            });
        });

        describe('resetPageNumber function', function () {
            var pages;

            beforeEach(function () {
                pages = service.getPagesList();

                pages.push({
                    route: 'main.page1'
                });

                pages.push({
                    route: 'main.page2',
                    params: {
                        type: 'param page'
                    }
                });

                pages.push({
                    route: 'main.page3'
                });
            });

            afterEach(function () {
                pages = service.getPagesList();

                removeItemsFromArray(pages);
            });

            it('should reset the page number to the position of the current route', function () {
                stateMock.current = {
                    name: 'main.page3'
                };

                evaluation = { 
                    pageNumber: 3
                };

                service.resetPageNumber();

                expect(evaluation.pageNumber).toEqual(2);
            });

            it('should reset the page number to the position of the current route, taking params into account', function () {
                stateMock.current = {
                    name: 'main.page2'
                };

                evaluation = { 
                    pageNumber: 3
                };

                stateParamsMock.type = 'param page';

                service.resetPageNumber();

                expect(evaluation.pageNumber).toEqual(1);

                delete stateParamsMock.type;
            });

            it('should preserve the current page number the current route isn\'t found in the list', function () {
                stateMock.current = {
                    name: 'main.page4'
                };

                evaluation = { 
                    pageNumber: 2
                };

                service.resetPageNumber();

                expect(evaluation.pageNumber).toEqual(2);
            });
        });
    });
});