define(['imageUploadService'], function () {
    'use strict';

    describe('Image Upload Service', function () {
        var service,
            qMock,
            deferredMock,
            fhirResourcesMock,
            imageResponseServiceMock,
            evaluation = {
                'mediaRequestId': '1'
            },
            file = {
                'type': 'png',
                'size': '50',
                'name': 'mockImage.png'
            },
            response = {
                "resourceType": "Media",
                "type": "photo",
                "identifier": [{
                    "value": "mockValue"
                }],
                "subject": {
                    "reference":"Patient/1"
                },
                "content": {
                    "contentType": "image/png",
                    "data": "",
                    "size": 1,
                    "title": "mockTitle",
                },
                "id": "mockId"
            },
            params = {},
            route = 'photo-input';

        beforeEach(function () {
            module('angularTemplateApp');

            qMock = jasmine.createSpyObj('$q', ['defer']);
            deferredMock = jasmine.createSpyObj('deferred', ['resolve', 'reject', 'promise']);
            imageResponseServiceMock = jasmine.createSpyObj('imageResponseService', ['getEvaluation', 'saveEvaluation', 'getPatient', 'setMedia']);
            fhirResourcesMock = jasmine.createSpyObj('fhirResourcesMock', ['createUpdateFhirResource', 'getFhirBaseUrl']);

            qMock.defer.and.returnValue(deferredMock);

            fhirResourcesMock.createUpdateFhirResource.and.returnValue({
                then: function (callback) {
                    callback(
                        'Media/123'
                    );
                }
            });

            imageResponseServiceMock.getEvaluation.and.callFake(function () {
                return evaluation;
            });

            imageResponseServiceMock.saveEvaluation.and.returnValue({
                then: function (callback) {
                    callback();
                }
            });

            imageResponseServiceMock.getPatient.and.callFake(function () {
                return {id: '1'};
            });
            
            module(function ($provide) {
                $provide.value('$q', qMock);
                $provide.value('imageResponseService', imageResponseServiceMock);
                $provide.value('fhirResources', fhirResourcesMock);
            });

            inject(function (imageUploadService) {
                service = imageUploadService;
            });
        });

        describe('uploadImage function', function () {
            beforeEach(function () {
                spyOn(service, 'getBase64').and.returnValue({
                    then: function (callback) {
                        callback('image-content');
                    }
                });

                spyOn(service, 'createMedia').and.returnValue(response);
                spyOn(service, 'getRoute').and.returnValue(route);
                spyOn(service, 'getParams').and.returnValue(params);
            });

            it('should be defined', function () {
                expect(service.uploadImage).toBeDefined();
            });

            it('should return a promise', function () {
                expect(service.uploadImage({}, {})).toEqual(deferredMock.promise);
            });

            it('should call setMedia with correct argument', function () {
                var data = {
                    id: '123'
                };

                service.uploadImage(file, data);

                expect(imageResponseServiceMock.setMedia).toHaveBeenCalledWith(data);
            });

            it('should set a PUT method in resourceConfig if the id is set on the data object', function () {
                var data = {
                    id: '123'
                };

                var resourceConfig = {
                    "data": {
                        "content": {
                            "contentType": "image/png",
                            "data": "image-content",
                            "size": 1,
                            "title": "mockTitle",
                        },
                        "id": "mockId",
                        "identifier": [{
                            value: "mockValue"
                        }],
                        "resourceType": "Media",
                        "subject": {
                            reference: "Patient/1"
                        },
                        "type": "photo",
                        },
                    "method": "PUT",
                    "resourceName": "Media/123"
                };

                service.uploadImage(file, data);

                expect(fhirResourcesMock.createUpdateFhirResource).toHaveBeenCalledWith(resourceConfig);
                expect(resourceConfig.method).toEqual("PUT");
            });

            it('should set a POST method in resourceConfig if the id is not set on the data object', function () {
                var data = {};

                var resourceConfig = {
                    "data": {
                        "content": {
                            "contentType": "image/png",
                            "data": "image-content",
                            "size": 1,
                            "title": "mockTitle",
                        },
                        "id": "mockId",
                        "identifier": [{
                            value: "mockValue"
                        }],
                        "resourceType": "Media",
                        "subject": {
                            reference: "Patient/1"
                        },
                        "type": "photo",
                        },
                    "method": "POST",
                    "resourceName": "Media"
                };

                service.uploadImage(file, data);

                expect(fhirResourcesMock.createUpdateFhirResource).toHaveBeenCalledWith(resourceConfig);
                expect(resourceConfig.method).toEqual("POST");
            });


            it('should call getRoute method with correct argument', function () {
                var data = {
                    "imageClass": "mockImageClass"
                };

                service.uploadImage(file, data);

                expect(service.getRoute).toHaveBeenCalledWith(data.imageClass);
            });

            it('should call getParams method with correct argument', function () {
                var data = {
                    "id": "123"
                };

                service.uploadImage(file, data);

                expect(service.getParams).toHaveBeenCalledWith(data);
            });

            it('should call createMedia method with correct argument', function () {
                var data = {
                    "id": "123"
                };

                service.uploadImage(file, data);

                expect(service.createMedia).toHaveBeenCalledWith(data.id, evaluation.mediaRequestId, file);
            });

            it('should call getBase64 method with correct argument', function () {
                var data = {
                    "id": "123"
                };

                service.uploadImage(file, data);

                expect(service.getBase64).toHaveBeenCalledWith(file);
            });

            it('should call saveEvaluation if the saveEval parameter is undefined', function () {
                service.uploadImage(file, {});

                expect(imageResponseServiceMock.saveEvaluation).toHaveBeenCalled();
            });

            it('should call saveEvaluation if the saveEval parameter is set to true', function () {
                service.uploadImage(file, {}, true);

                expect(imageResponseServiceMock.saveEvaluation).toHaveBeenCalled();
            });

            it('should not call saveEvaluation if the saveEval parameter is set to false', function () {
                service.uploadImage(file, {}, false);

                expect(imageResponseServiceMock.saveEvaluation).not.toHaveBeenCalled();
            });
        });

        describe('getRoute function', function () {
            var imageClass1 = "OPTIONAL",
                route1 = "photo-input-optional",
                imageClass2 = "PRIMARY",
                route2 = "photo-input",
                imageClass3 = "CLOSE_UP",
                route3 = "photo-input-close-up";


            it('should be defined', function () {
                expect(service.getRoute).toBeDefined();
            });

            it('should return the correct value when calling method', function () {
                expect(service.getRoute(imageClass1)).toEqual(route1);
            }); 

            it('should return the correct value when calling method', function () {
                expect(service.getRoute(imageClass2)).toEqual(route2);
            }); 

            it('should return the correct value when calling method', function () {
                expect(service.getRoute(imageClass3)).toEqual(route3);
            }); 
        });

        describe('getBase64 function', function () {
            // https://github.com/ariya/phantomjs/issues/14247
            // https://stackoverflow.com/questions/27159179/how-to-convert-blob-to-file-in-javascript

            var myBlob = new Blob();
                myBlob.lastModifiedDate = new Date();
                myBlob.name = "mockText";

            it('should be defined', function () {
                expect(service.getBase64).toBeDefined();
            });

            it('should return the correct value when calling method', function () {
                expect(service.getBase64(myBlob)).toEqual(deferredMock.promise);
            });
        });

        describe('createMedia function', function () {
            var data = {
                "id": "123"
            };

            var mockResponse = {
                "resourceType":"Media",
                "type":"photo",
                "identifier":[{
                    "value": "1"
                }],
                "subject":{
                    "reference":"Patient/1"
                },
                "content":{
                    "contentType": "png",
                    "data": "",
                    "size": '50',
                    "title": "mockImage.png",
                },
                "id":"123"
            };
            
            it('should be defined', function () {
                expect(service.createMedia).toBeDefined();
            });

            it('should return the correct value when calling method', function () {
                expect(service.createMedia(data.id, evaluation.mediaRequestId, file)).toEqual(mockResponse);
            });
        });

        describe('getParams function', function () {
            var imageData = {
                "fileName": "mockName",
                "imageClass": "CLOSE_UP",
                "imageDescription": "mockDescription",
                "imageNumber": 1,
                "imageSection": {
                    "description": "HEAD",
                    "name": "HEAD"
                },
                "imageType": {
                    "description": "mockDescription",
                    "name": "mockName"
                }
            };

            var params = {
                "imageNum": 1,
                "imageSection": {
                "description": "HEAD",
                "name": "HEAD"
                },
                "imageType": {
                "description": "mockDescription",
                "name": "mockName"
                }
            };

            it('should be defined', function () {
                expect(service.getParams).toBeDefined();
            });

            it('should return the correct value when calling method', function () {
                expect(service.getParams(imageData)).toEqual(params);
            });
        });

    });
});