define(['imageResponseService'], function () {
	'use strict';

	describe('image Response Service', function () {
		var service,
			httpMock,
			localResourceDirectoryServiceMock,
			fhirResourcesMock,
			fhirPatientMock,
			questionnaireServiceMock,
			medTypes = ['TOPICAL', 'ORAL', 'INJECTED'],
			$q,
			$scope,
			questionnaireMap = {"MYTELEDERM_QUESTIONNAIRE": ["IMAGE"], "SKIN_STATUS": ["1"], "TOPICAL_MEDICATIONS":["2"], "ORAL_MEDICATIONS":["3"], "OTHER_MEDICATIONS":["5"],
				"MEDIA": "8", "HEAD": ["8.1"],"HEAD_FACE": ["8.1.1"],"HEAD_LEFT_FACE": ["8.1.2"],"HEAD_CLOSEUP1": ["8.1.6"], "PRIMARY": ["8.1.1.1"], "CLOSE_UP": ["8.1.6.1"],
				"MEDIA_OPTIONAL": "9", "OPTIONAL1": ["9.1"],"OPTIONAL2": ["9.2"]},
			imageTypesBySection = {"HEAD": [{"name":"HEAD_FACE","description":"Head and Face","category":"IMAGE"},{"name":"HEAD_LEFT_FACE","description":"Left Side of Face","category":"IMAGE"}]},
			imageSectionsMap = {"HEAD" : {"name":"HEAD","description":"Head"}};


		beforeEach(function () {
			module('angularTemplateApp');

			localResourceDirectoryServiceMock = jasmine.createSpyObj('localResourceDirectoryServiceMock', ['fetch']);
			fhirResourcesMock = jasmine.createSpyObj('fhirResourcesMock', ['resolveFhirResourceByReference','resolveFhirResourceByParams', 'deleteFhirResourceByReference','getSystem']);
			fhirPatientMock = jasmine.createSpyObj('fhirPatientMock', ['fetch']);
			questionnaireServiceMock = jasmine.createSpyObj('questionnaireServiceMock', ['fetch','getQuestionnaire']);

			localResourceDirectoryServiceMock.fetch.and.returnValue({
				then: function (callback) {
					callback({
						'image-sections': '/media-request-service/v1/media/sections',
						'update-request': '/media-request-service/v1/patients/{UNIQUE_ID}/requests/{MEDIA_REQUEST_ID}',
						'questionnaireResponse': '/pgd-fhir-services/v1/fhir/QuestionnaireResponse/{QUESTIONNAIRE_RESPONSE_ID}'
					});
				}
			});

			fhirPatientMock.fetch.and.returnValue({
				then: function (callback) {
					callback({
						"id" : "399649c0-ea7e-4ff4-ac9c-8eee722a4d9e"
					});
				}
			});

			questionnaireServiceMock.getQuestionnaire.and.returnValue({
				then: function (callback) {
					callback({
						"resource": {
							resourceType: "Questionnaire",
							"id": "669936f6-43b8-49dc-aab3-0f962f5a88d6",
							group: {
								text: "MYTELEDERM_QUESTIONNAIRE",
								linkId: "IMAGE",
								group: [
									{linkId: "1", text: "SKIN_STATUS"},
									{linkId: "2", text: "TOPICAL_MEDICATIONS"},
									{linkId: "5", text: "OTHER_MEDICATIONS"},
									{linkId: "8", text: "MEDIA"}
								]
							}
						}
					});
				}
			});

			fhirResourcesMock.getSystem.and.returnValue('https://URL/display/PGDMS/Client+Provenance+Mapping');

			module(function ($provide) {
				$provide.value('localResourceDirectoryService', localResourceDirectoryServiceMock);
				$provide.value('fhirResources', fhirResourcesMock);
				$provide.value('fhirPatient', fhirPatientMock);
				$provide.value('questionnaireService', questionnaireServiceMock);
				$provide.value('medicationTypes', medTypes);
			});

			inject(function ($rootScope, _$q_, imageResponseService, questionnaireResponseService, $httpBackend) {
				httpMock = $httpBackend;
				$q = _$q_;
				$scope = $rootScope.$new();
				service = imageResponseService;

				var fhirResourcesDeferred = $q.defer();
				fhirResourcesDeferred.resolve();
				fhirResourcesMock.deleteFhirResourceByReference.and.returnValue(fhirResourcesDeferred.promise);
			});
		});

		describe('initEvaluation', function () {
			it('should initialize a new MyTelederm evaluation if none is set and the questionnaire response is blank', function () {
				fhirResourcesMock.resolveFhirResourceByReference.and.returnValue({
					then: function (callback) {
						callback({});
					}
				});

				service.initEvaluation("MyTelederm");
				$scope.$apply();
				expect(service.getEvaluation()).toEqual({
					"mediaRequestId": "",
					"status": "STARTED",
					"skinStatus": "",
					"skinStatusComment": "",
					"medicationTypes": [],
					"otherMedication": "",
					"additionalComments": "",
					"isResponsesAccepted": false,
					"medications": [],
					"images": [],
					"isOptionalImages": null,
					"pageNumber": 0
				});
			});
		});

		describe('transformResponse', function () {

			it('should transform the data from the retrieved questionnaire response to a MyTelederm evaluation', function () {
				fhirResourcesMock.resolveFhirResourceByReference.and.returnValue({
					then: function (callback) {
						callback({
							resourceType: "QuestionnaireResponse",
							group: {
								text: 'MyTelederm Questionnaire Response',
								group: [
									{
										linkId: "1",
										question: [{linkId: "1.2", answer: [{valueString: "comments on better"}]}]
									},
									{
										linkId: "5",
										question: [{linkId: "5.1", answer: [{valueString: "other medications details"}]}]
									}
								]
							}
						});
					}
				});
				service.initEvaluation("MyTelederm");
				$scope.$apply();


				expect(service.getEvaluation()).toEqual({
					"mediaRequestId": "",
					"status": "NOT_STARTED",
					"skinStatus": "",
					"skinStatusComment": "comments on better",
					"medicationTypes": ['OTHER'],
					"otherMedication": "other medications details",
					"additionalComments": "",
					"isResponsesAccepted": false,
					"medications": [],
					"images": [],
					"isOptionalImages": null,
					"pageNumber": 0
				});
			});
		});

		describe('updateResponse', function () {

			it('should update only data related to the page being saved and not replace any data related to other pages', function () {
				service.questionnaireMap = questionnaireMap;
				service.currResponse = {
					resourceType: "QuestionnaireResponse",
					group: {
						text: "MyTelederm Questionnaire Response",
						group: [ {linkId: "1", question: [{linkId: "1.2", answer: [{valueString: "old comments on better"}]}]},
							{linkId: "5", question: [{linkId: "5.1", answer: [{valueString: "this medication comment should appear in the response since it was already there"}]}]}]
					}
				};
				service.setEvaluation({
					"skinStatus": "BETTER",
					"skinStatusComment": "new comments on better should appear in the response",
					"otherMedication": "this other medication should appear in the response since it was already there",
					"additionalComments": "this comment should not appear in the response"
				});

				service.updateResponse({route: 'skin-status'});

				expect(service.getCurrResponse()).toEqual({
					resourceType: "QuestionnaireResponse",
					group: {
						text: "MyTelederm Questionnaire Response",
						group: [
							{
								linkId: "1",
								question: [
									{linkId: "1.1",answer: [{valueCoding: {code: "0",display: "BETTER",system: "https://URL/display/PGDMS/Client+Provenance+Mapping"}}]},
									{linkId: "1.2", answer: [{valueString: "new comments on better should appear in the response"}]}
								]
							},
							{linkId: "5", question: [{linkId: "5.1", answer: [{valueString: "this medication comment should appear in the response since it was already there"}]}]}
						]
					}
				});
			});

			describe('updateResponse for medications', function () {
				beforeEach(function () {
					service.questionnaireMap = questionnaireMap;
					service.currResponse = {
						resourceType: "QuestionnaireResponse",
						group: {
							text: "MyTelederm Questionnaire Response",
							group: []
						}
					};
					service.setEvaluation({
						"medicationTypes": ['TOPICAL'],
					});

					service.updateResponse({route: 'medications'});

				});

				it('should create other medications group with N/A if page is medications or medications entry irrespective of whether other medications was selected or not', function () {

					expect(service.getCurrResponse()).toEqual({
						resourceType: "QuestionnaireResponse",
						group: {
							text: "MyTelederm Questionnaire Response",
							group: [
								{linkId: "2", group: []},
								{linkId: "5", question: [{linkId: "5.1", answer: [{valueString: "N/A"}]}]}
							]
						}
					});
				});

				it('should delete medications group if page is medications and medication type was unselected and is not OTHER', function () {
					service.currResponse.group.group = [{linkId: "2", group: []}];
					service.currEvaluation.medicationTypes = [];

					service.updateResponse({route: 'medications'});

					expect(service.getCurrResponse()).toEqual({
						resourceType: "QuestionnaireResponse",
						group: {
							text: "MyTelederm Questionnaire Response",
							group: [
								{linkId: "5", question: [{linkId: "5.1", answer: [{valueString: "N/A"}]}]}
							]
						}
					});
				});

				it('should update only one medications group if page params medType is passed', function () {
					spyOn(service, 'updatePage');
					service.currEvaluation.medicationTypes.push('ORAL');

					service.updateResponse({route: 'medications', params : {medType: 'TOPICAL'}});
					expect(service.updatePage).toHaveBeenCalledWith('TOPICAL_MEDICATIONS');
					expect(service.updatePage).not.toHaveBeenCalledWith('ORAL_MEDICATIONS');
				});

				it('should update all selected medications group if no page params medType is passed', function () {
					spyOn(service, 'updatePage');
					service.currEvaluation.medicationTypes.push('ORAL');

					service.updateResponse({route: 'medications'});
					expect(service.updatePage).toHaveBeenCalledWith('TOPICAL_MEDICATIONS');
					expect(service.updatePage).toHaveBeenCalledWith('ORAL_MEDICATIONS');
					expect(service.updatePage).toHaveBeenCalledWith('OTHER_MEDICATIONS');
					expect(service.updatePage).not.toHaveBeenCalledWith('INJECTED_MEDICATIONS');
				});

			});

			describe('updateResponse for photo-inputs', function () {
				beforeEach(function () {
					service.questionnaireMap = questionnaireMap;

					service.setEvaluation({
						"images": [
							{
								imageSection: {name: 'HEAD', description: 'Head'},
								imageType: {name: 'HEAD_FACE', description: 'Head and Face'},
								imageClass: 'PRIMARY',
								fileName: 'required1.png'
							},
							{
								imageSection: {name: 'HEAD', description: 'Head'},
								imageType: {name: 'HEAD_LEFT_FACE', description: 'Left Side of Face'},
								imageClass: 'PRIMARY',
								fileName: 'required2.png'
							},
							{
								imageSection: {name: 'HEAD', description: 'Head'},
								imageType: {name: 'CLOSE_UP', description: 'Close up'},
								imageClass: 'CLOSE_UP',
								imageNumber: 1,
								fileName: 'closeup1.png'
							}]
					});
				});

				it('should create the Media group, one image section and one type group based on the page params passed', function () {
					service.currResponse = {
						resourceType: "QuestionnaireResponse",
						group: {
							text: "MyTelederm Questionnaire Response",
							group: []
						}
					};

					service.updateResponse({route: 'photo-input', params : {imageSection: {name: 'HEAD',description: 'Head'}, imageType: {name: 'HEAD_FACE',description: 'Head and Face'}}});
					expect(service.getCurrResponse()).toEqual({
						resourceType: "QuestionnaireResponse",
						group: {
							text: "MyTelederm Questionnaire Response",
							group: [
								{
									linkId: "8",
									group: [{
										linkId: '8.1', group: [{
											linkId: '8.1.1', question: [
												{linkId: '8.1.1.1', text: 'PRIMARY', answer: [{valueString: ''}]},
												{linkId: '8.1.1.2', answer: [{valueBoolean: false}]},
												{linkId: '8.1.1.3', answer: [{valueBoolean: false}]}
											]
										}]
									}]
								}
							]
						}
					});
				});

				it('should create a closeup group if page param image Type is CLOSE_UP', function () {
					service.currResponse = {
						resourceType: "QuestionnaireResponse",
						group: {
							text: "MyTelederm Questionnaire Response",
							group: [{
								linkId: "8",
								group: [{
									linkId: '8.1',
									group: []
								}]
							}]
						}
					};

					service.updateResponse({route: 'photo-input', params : {imageSection: {name: 'HEAD',description: 'Head'}, imageType: {name: 'CLOSE_UP',description: 'Close up'}, imageNum: 1}});
					expect(service.getCurrResponse()).toEqual({
						resourceType: "QuestionnaireResponse",
						group: {
							text: "MyTelederm Questionnaire Response",
							group: [
								{
									linkId: "8",
									group: [{
										linkId: '8.1',
										group: [
											{	linkId: '8.1.6', question: [
												{linkId: '8.1.6.1', text: 'CLOSE_UP', answer: [{valueString: ''}]},
												{linkId: '8.1.6.2', answer: [{valueBoolean: false}]},
												{linkId: '8.1.6.3', answer: [{valueBoolean: false}]},
												{linkId: '8.1.6.4', answer: [{valueInteger: 1}]},
												{linkId: '8.1.6.5', answer: [{valueString: ''}]}
											]
											}]
									}]
								}
							]
						}
					});
				});

				it('should create the image section group if it does not exist', function () {
					spyOn(service, 'createImageSectionGroup');
					spyOn(service, 'updateImageTypeGroup');
					service.currResponse = {
						resourceType: "QuestionnaireResponse",
						group: {
							text: "MyTelederm Questionnaire Response",
							group: [{
								linkId: "8",
								group: []
							}]
						}
					};

					service.updateResponse({route: 'photo-input', params : {imageSection: {name: 'HEAD',description: 'Head'}, imageType: {name: 'HEAD_FACE',description: 'Head and Face'}}});
					expect(service.createImageSectionGroup).toHaveBeenCalled();
					expect(service.updateImageTypeGroup).not.toHaveBeenCalled();
				});

				it('should not create the image section group if it exists instead create/update the image type group', function () {
					spyOn(service, 'createImageSectionGroup');
					spyOn(service, 'updateImageTypeGroup');

					service.currResponse = {
						resourceType: "QuestionnaireResponse",
						group: {
							text: "MyTelederm Questionnaire Response",
							group: [{
								linkId: "8",
								group: [{
									linkId: '8.1',
									group: []
								}]
							}]
						}
					};

					service.updateResponse({route: 'photo-input', params : {imageSection: {name: 'HEAD',description: 'Head'}, imageType: {name: 'HEAD_FACE',description: 'Head and Face'}}});
					expect(service.createImageSectionGroup).not.toHaveBeenCalled();
					expect(service.updateImageTypeGroup).toHaveBeenCalled();
				});

			});

			describe('updateResponse for review photos', function () {
				beforeEach(function () {
					service.questionnaireMap = questionnaireMap;

					service.setEvaluation({
						"images": [
							{
								id: "1",
								imageSection: {name: 'HEAD', description: 'Head'},
								imageType: {name: 'HEAD_FACE', description: 'Head and Face'},
								imageClass: 'PRIMARY',
								fileName: 'required1.png'
							},
							{
								id: "3",
								imageSection: {name: 'HEAD', description: 'Head'},
								imageType: {name: 'CLOSE_UP', description: 'Close up'},
								imageClass: 'CLOSE_UP',
								imageNumber: 1,
								fileName: 'closeup1.png'
							},
							{
								id: "4",
								imageSection: {name: 'HEAD', description: 'Head'},
								imageType: {name: 'CLOSE_UP', description: 'Close up'},
								imageClass: 'CLOSE_UP',
								imageNumber: 2,
								fileName: 'closeup2.png'
							}]
					});
				});
				it('should recreate the images section group and assign the closeups to the new linkId if closeups were deleted', function () {

					service.setImageTypesBySection(imageTypesBySection);
					service.currResponse = {
						resourceType: "QuestionnaireResponse",
						group: {
							text: "MyTelederm Questionnaire Response",
							group: [
								{
									linkId: "8",
									group: [
										{linkId: '8.1', group: [{linkId: '8.1.1'}, {linkId: '8.1.6'},{linkId: '8.1.7'}]}
									]
								}
							]
						}
					};

					service.deletePhotos(['3']);
					service.updateResponse({route: 'review-photos'});
					expect(service.currResponse).toEqual({
						resourceType: "QuestionnaireResponse",
						group: {
							text: "MyTelederm Questionnaire Response",
							group: [
								{
									linkId: "8",
									group: [
										{linkId: '8.1', group: [
											{linkId: '8.1.1', question: [
												{linkId: '8.1.1.1', text: 'PRIMARY', answer: [{valueString: '1'}]},
												{linkId: '8.1.1.2', answer: [{valueBoolean: false}]},
												{linkId: '8.1.1.3', answer: [{valueBoolean: false}]}]},
											{linkId: '8.1.6', question: [
												{linkId: '8.1.6.1', text: 'CLOSE_UP', answer: [{valueString: '4'}]},
												{linkId: '8.1.6.2', answer: [{valueBoolean: false}]},
												{linkId: '8.1.6.3', answer: [{valueBoolean: false}]},
												{linkId: '8.1.6.4', answer: [{valueInteger: 1}]},
												{linkId: '8.1.6.5', answer: [{valueString: ''}]}]}
										]
										}
									]
								}]}});
				});
			});

		});

		describe('createSkinStatusGroup', function () {
			it('should create a JSON object with linkId 1 and answer of valueCoding', function () {
				service.currEvaluation.skinStatus = "WORSE";
				service.currEvaluation.skinStatusComment = "";

				var group = service.createSkinStatusGroup("1");
				expect(group).toEqual({linkId:"1",question:[{linkId:"1.1",answer:[{valueCoding:{code:"2",display:"WORSE",system:"https://URL/display/PGDMS/Client+Provenance+Mapping"}}]}]});
			});
		});

		describe('createMedicationGroup and createMedicationSubGroup', function () {
			it('should create a JSON object with linkId corresponding to the med type and a subgroup of medications', function () {
				service.currEvaluation.medicationTypes = ['ORAL'];

				var group = service.createMedicationGroup("ORAL", '3');
				expect(group).toEqual({linkId:"3",group:[]});
			});

			it('should create a subgroup containing upto 10 medication groups per type based on the medications entered', function () {
				spyOn(service, 'createMedicationSubGroup');
				service.currEvaluation.medicationTypes = ['ORAL'];
				service.currEvaluation.medications = [ {type: 'ORAL',name: 'Advil',dose: '2',frequency: '1 time', timeUnit: 'a day',missedDoses: 'None'},
					{type: 'ORAL',name: 'Tylenol',dose: '2', frequency: '1 time', timeUnit: 'a day',missedDoses: 'None'}];

				var group = service.createMedicationGroup("ORAL", '3');
				expect(service.createMedicationSubGroup.calls.allArgs()).toEqual([["3",1, {type: 'ORAL',name: 'Advil',dose: '2', frequency: '1 time', timeUnit: 'a day',missedDoses: 'None'}], ["3",2, {type: 'ORAL',name: 'Tylenol',dose: '2', frequency: '1 time', timeUnit: 'a day',missedDoses: 'None'}]]);
			});

			it('createMedicationSubGroup should create a group with a set of 5 answers per medication', function () {
				service.currEvaluation.medicationTypes = ['ORAL'];

				service.currEvaluation.medications = [{type: 'ORAL',name: 'Advil',dose: '2', frequency: '1 time', timeUnit: 'a day',missedDoses: 'None'}];
				var group = service.createMedicationGroup("ORAL", '3');
				expect(group).toEqual({linkId:"3",
					group:[
						{  linkId:"3.1",
							question:[
								{linkId:"3.1.1",answer:[{valueString:"Advil"}]},
								{linkId:"3.1.2",answer:[{valueString:"2"}]},
								{linkId:"3.1.3",answer:[{valueCoding:{code:"0",display:"1 time",system:"https://URL/display/PGDMS/Client+Provenance+Mapping"}}]},
								{linkId:"3.1.5",answer:[{valueCoding:{code:"0",display:"a day",system:"https://URL/display/PGDMS/Client+Provenance+Mapping"}}]},
								{linkId:"3.1.7",answer:[{valueCoding:{code:"0",display:"None",system:"https://URL/display/PGDMS/Client+Provenance+Mapping"}}]}

							]}
					]});
			});

			it('createMedicationSubGroup should create a group with a set of 7 answers per medication if other is selected for frequency and timeUnit', function () {
				service.currEvaluation.medicationTypes = ['ORAL'];

				service.currEvaluation.medications = [{type: 'ORAL',name: 'Advil',dose: '2', frequency: 'other', frequencyOther: '100 times', timeUnit: 'other', timeUnitOther: '100 times', missedDoses: 'None'}];
				var group = service.createMedicationGroup("ORAL", '3');
				expect(group).toEqual({linkId:"3",
					group:[
						{  linkId:"3.1",
							question:[
								{linkId:"3.1.1",answer:[{valueString:"Advil"}]},
								{linkId:"3.1.2",answer:[{valueString:"2"}]},
								{linkId:"3.1.3",answer:[{valueCoding:{code:"6",display:"other",system:"https://URL/display/PGDMS/Client+Provenance+Mapping"}}]},
								{linkId:"3.1.4",answer:[{valueString:"100 times"}]},
								{linkId:"3.1.5",answer:[{valueCoding:{code:"5",display:"other",system:"https://URL/display/PGDMS/Client+Provenance+Mapping"}}]},
								{linkId:"3.1.6",answer:[{valueString:"100 times"}]},
								{linkId:"3.1.7",answer:[{valueCoding:{code:"0",display:"None",system:"https://URL/display/PGDMS/Client+Provenance+Mapping"}}]}

							]}
					]});
			});
		});

		describe('createImageGroup and createImageSectionGroup and createImageTypeSubGroup', function () {
			beforeEach(function () {
				service.questionnaireMap = questionnaireMap;
			});

			it('should not create a image section group if there are no images', function () {
				service.currEvaluation.images = [];

				var group = service.createImageGroup("HEAD", "HEAD_FACE", "8.1", "8.1.1");
				expect(group).toEqual({linkId:"8",group:[]});
			});

			it('should create a JSON object with linkId corresponding to the image section type and a subgroup corresponding to the image type', function () {
				service.currEvaluation.images = [{
					imageSection: {name: 'HEAD', description: 'Head'},
					imageType: {name: 'HEAD_FACE', description: 'Head and Face'},
					imageClass: 'PRIMARY',
					fileName: 'required1.png'
				}];
				var group = service.createImageGroup("HEAD", "HEAD_FACE", "8.1", "8.1.1");
				expect(group).toEqual({
					linkId: "8",
					group: [{
						linkId: "8.1",
						group: [{
							linkId: "8.1.1",
							question: [
								{linkId: '8.1.1.1', text: 'PRIMARY', answer: [{valueString: ''}]},
								{linkId: '8.1.1.2', answer: [{valueBoolean: false}]},
								{linkId: '8.1.1.3', answer: [{valueBoolean: false}]}]
						}]
					}]
				});
			});

			it('createImageTypeSubGroup should create a group with a set of 3 answers for PRIMARY and 5 answers for CLOSEUPs and OPTIONAL images', function () {
				service.currEvaluation.images = [{
					imageSection: {name: 'HEAD', description: 'Head'},
					imageType: {name: 'HEAD_FACE', description: 'Head and Face'},
					imageClass: 'PRIMARY',
					fileName: 'required1.png'
				}];
				var group = service.createImageTypeSubGroup("8.1.1", service.currEvaluation.images[0]);
				expect(group).toEqual({
					linkId: "8.1.1",
					question: [
						{linkId: '8.1.1.1', text: 'PRIMARY', answer: [{valueString: ''}]},
						{linkId: '8.1.1.2', answer: [{valueBoolean: false}]},
						{linkId: '8.1.1.3', answer: [{valueBoolean: false}]}]
				});

				service.currEvaluation.images = [{
					imageSection: {name: 'HEAD', description: 'Head'},
					imageType: {name: 'CLOSE_UP', description: 'Close up'},
					imageClass: 'CLOSE_UP',
					fileName: 'closeup1.png',
					imageNumber: 1
				}];
				var group = service.createImageTypeSubGroup("8.1.6", service.currEvaluation.images[0]);
				expect(group).toEqual({
					linkId: "8.1.6",
					question: [
						{linkId: '8.1.6.1', text: 'CLOSE_UP', answer: [{valueString: ''}]},
						{linkId: '8.1.6.2', answer: [{valueBoolean: false}]},
						{linkId: '8.1.6.3', answer: [{valueBoolean: false}]},
						{linkId: '8.1.6.4', answer: [{valueInteger: 1}]},
						{linkId: '8.1.6.5', answer: [{valueString: ''}]}]
				});
			});
		});

		describe('refreshImageSectionGroup and refreshOptionalImagesGroup', function () {
			it('refreshImageSectionGroup should recreate an image section group excluding deleted closeups and reassigning linkIds to the remaining ones ', function () {
				service.questionnaireMap = questionnaireMap;
				service.setImageTypesBySection(imageTypesBySection);
				service.setEvaluation({
					"images": [
						{
							id: "deleteId",
							imageSection: {name: 'HEAD', description: 'Head'},
							imageType: {name: 'CLOSE_UP', description: 'Close up'},
							imageClass: 'CLOSE_UP',
							imageNumber: 1,
							fileName: 'closeup1.png'
						},
						{
							id: "idPreviouslyLinkedto8.1.7",
							imageSection: {name: 'HEAD', description: 'Head'},
							imageType: {name: 'CLOSE_UP', description: 'Close up'},
							imageClass: 'CLOSE_UP',
							imageNumber: 2,
							fileName: 'closeup2.png'
						}]
				});
				service.deletePhotos("deleteId");
				var group = service.refreshImageSectionGroup("HEAD", "8.1");
				expect(group).toEqual(
					{linkId: '8.1', group: [
						{linkId: '8.1.6', question: [
							{linkId: '8.1.6.1', text: 'CLOSE_UP', answer: [{valueString: 'idPreviouslyLinkedto8.1.7'}]},
							{linkId: '8.1.6.2', answer: [{valueBoolean: false}]},
							{linkId: '8.1.6.3', answer: [{valueBoolean: false}]},
							{linkId: '8.1.6.4', answer: [{valueInteger: 1}]},
							{linkId: '8.1.6.5', answer: [{valueString: ''}]}]}
					]
					});
			});

			it('refreshOptionalImagesGroup should recreate the optional images group excluding deleted images and reassigning linkIds to the remaining ones ', function () {
				service.questionnaireMap = questionnaireMap;
				service.setEvaluation({
					"images": [
						{
							id: "deleteId",
							imageSection: {name: 'HEAD', description: 'Head'},
							imageType: {name: 'OPTIONAL', description: 'Optional'},
							imageClass: 'OPTIONAL',
							imageNumber: 10,
							fileName: 'optional1.png'
						},
						{
							id: "idPreviouslyLinkedto9.2",
							imageSection: {name: 'HEAD', description: 'Head'},
							imageType: {name: 'OPTIONAL', description: 'Optional'},
							imageClass: 'OPTIONAL',
							imageNumber: 1,
							fileName: 'optional2.png'
						}]
				});

				var group = service.refreshOptionalImagesGroup();
				expect(group).toEqual([
					{linkId: '9.1', question: [
						{linkId: '9.1.1', text: 'OPTIONAL', answer: [{valueString: 'idPreviouslyLinkedto9.2'}]},
						{linkId: '9.1.2', answer: [{valueBoolean: false}]},
						{linkId: '9.1.3', answer: [{valueBoolean: false}]},
						{linkId: '9.1.4', answer: [{valueInteger: 1}]},
						{linkId: '9.1.5', answer: [{valueString: ''}]}]
					}]);
			});

		});

		describe('getMedications and getMedication', function () {
			var response;
			beforeEach(function () {
				response = {
					group: {
						group: [
							{
								linkId:"2",
								group:[
									{  linkId:"2.1",
										question:[
											{linkId:"2.1.1",answer:[{valueString:"Benadryl Cream"}]},
											{linkId:"2.1.2",answer:[{valueString:"2"}]},
											{linkId:"2.1.3",answer:[{valueCoding:{code:"0",display:"1 time",system:"https://URL/display/PGDMS/Client+Provenance+Mapping"}}]},
											{linkId:"2.1.5",answer:[{valueCoding:{code:"0",display:"a day",system:"https://URL/display/PGDMS/Client+Provenance+Mapping"}}]},
											{linkId:"2.1.7",answer:[{valueCoding:{code:"0",display:"None",system:"https://URL/display/PGDMS/Client+Provenance+Mapping"}}]}
										]
									}]
							},
							{
								linkId:"3",
								group:[
									{  linkId:"3.1",
										question:[
											{linkId:"3.1.1",answer:[{valueString:"Advil"}]},
											{linkId:"3.1.2",answer:[{valueString:"2"}]},
											{linkId:"3.1.3",answer:[{valueCoding:{code:"6",display:"other",system:"https://URL/display/PGDMS/Client+Provenance+Mapping"}}]},
											{linkId:"3.1.4",answer:[{valueString:"100 times"}]},
											{linkId:"3.1.5",answer:[{valueCoding:{code:"0",display:"a day",system:"https://URL/display/PGDMS/Client+Provenance+Mapping"}}]},
											{linkId:"3.1.7",answer:[{valueCoding:{code:"0",display:"None",system:"https://URL/display/PGDMS/Client+Provenance+Mapping"}}]}

										]
									}
								]
							}]
					}
				};
				service.questionnaireMap = questionnaireMap;
			});

			it('getMedications should return an empty array if there are no medications', function () {
				var response = { group: { group: [] } };
				var meds = service.getMedications(response);
				expect(meds).toEqual([]);
			});

			it('getMedications should return an array of all the medications entered', function () {
				var meds = service.getMedications(response);
				expect(meds).toEqual([{
					type: 'TOPICAL',
					name: 'Benadryl Cream',
					frequency: '1 time',
					frequencyOther: '',
					timeUnit: 'a day',
					timeUnitOther: '',
					dose: '2',
					missedDoses: 'None'
				},
				{
					type: 'ORAL',
					name: 'Advil',
					frequency: 'other',
					frequencyOther: '100 times',
					timeUnit: 'a day',
					timeUnitOther: '',
					dose: '2',
					missedDoses: 'None'
				}]);
			});

			it ('getMedication should return the medications of the given type', function () {
				var meds = service.getMedication(response, "2", "TOPICAL");
				expect(meds).toEqual([{
					type: 'TOPICAL',
					name: 'Benadryl Cream',
					frequency: '1 time',
					frequencyOther: '',
					timeUnit: 'a day',
					timeUnitOther: '',
					dose: '2',
					missedDoses: 'None'
				}]);
			});
		});

		describe('getImages and getImage', function () {
			var response;
			beforeEach(function () {
				response = {
					group: {
						group: [
							{
								linkId: "8",
								group: [
									{linkId: '8.1', group: [
										{linkId: '8.1.1', question: [
											{linkId: '8.1.1.1', text: 'PRIMARY', answer: [{valueString: '1'}]},
											{linkId: '8.1.1.2', answer: [{valueBoolean: false}]},
											{linkId: '8.1.1.3', answer: [{valueBoolean: false}]}]},
										{linkId: '8.1.6', question: [
											{linkId: '8.1.6.1', text: 'CLOSE_UP', answer: [{valueString: '2'}]},
											{linkId: '8.1.6.2', answer: [{valueBoolean: false}]},
											{linkId: '8.1.6.3', answer: [{valueBoolean: false}]},
											{linkId: '8.1.6.4', answer: [{valueInteger: 1}]},
											{linkId: '8.1.6.5', answer: [{valueString: ''}]}]}
									]
									}
								]
							}]}
				};
				service.questionnaireMap = questionnaireMap;
				service.resourceLinks = {'media':'{MEDIA_ID}'};
				service.setImageSectionsMap(imageSectionsMap);
				service.setImageTypesBySection(imageTypesBySection);
			});

			it('getImages should return an empty array if there are no images', function () {
				var response = { group: { group: [] } };
				var imgs = service.getImages(response);
				expect(imgs).toEqual([]);
			});

			it('getImages should return an array of all the images uploaded', function () {
				var imgs = service.getImages(response);
				expect(imgs).toEqual([
					{
						id: "1",
						imageUrl: '1',
						imageSection: {name: 'HEAD', description: 'Head'},
						imageType: {name: 'HEAD_FACE', description: 'Head and Face'},
						imageClass: 'PRIMARY',
						imageDescription : '',
						imageNumber: 0
					},
					{
						id: "2",
						imageUrl: '2',
						imageSection: {name: 'HEAD', description: 'Head'},
						imageType: {name: 'CLOSE_UP', description: 'Other'},
						imageClass: 'CLOSE_UP',
						imageDescription : '',
						imageNumber: 1
					}
				]);
			});

			it ('getImage should return the image of the given image section and type', function () {
				var img = service.getImage(response.group.group[0].group[0].group[0], "HEAD", "HEAD_FACE");
				expect(img).toEqual({
					id: "1",
					imageUrl: '1',
					imageSection: {name: 'HEAD', description: 'Head'},
					imageType: {name: 'HEAD_FACE', description: 'Head and Face'},
					imageClass: 'PRIMARY',
					imageDescription : '',
					imageNumber: 0
				});
			});
		});

		describe('getUploadedImagesBySection and initUploadedImagesBySection', function () {
			beforeEach(function () {
				var imageTypesBySection = [];

				imageTypesBySection['HEAD'] = [
					{
						'name': 'HEAD_FACE',
						'description': 'Head and Face'
					},
					{
						'name': 'HEAD_LEFT_FACE',
						'description': 'Left Side of Face'
					}
				];

				imageTypesBySection['LOWER_LEG'] = [
					{
						'name': 'LOWER_LEG_LL_LEFT',
						'description': 'Left Side of Lower Legs'
					},
					{
						'name': 'LOWER_LEG_LL_RIGHT',
						'description': 'Right Side of Lower Legs'
					}
				];

				service.setImageTypesBySection(imageTypesBySection);
			});

			afterEach(function () {
				var imageTypesBySection = [];

				delete imageTypesBySection['HEAD'];

				delete imageTypesBySection['LOWER_LEG'];
			});

			it('should return a default object if there are no uploaded images', function () {
				service.setEvaluation({
					status: 'NOT_STARTED',
					images: []
				});
				service.initUploadedImagesBySection();

				expect(service.getUploadedImagesBySection()).toEqual({
					'HEAD': {
						'PRIMARY': [],
						'CLOSE_UP': []
					},
					'LOWER_LEG': {
						'PRIMARY': [],
						'CLOSE_UP': []
					},
					'OPTIONAL': []
				});
			});

			it('should correctly return the object based on the uploaded images', function () {
				service.setEvaluation({
					images: [
						{
							imageSection: {
								name: 'HEAD',
								description: 'Head'
							},
							imageType: {
								'name': 'HEAD_FACE',
								'description': 'Head and Face'
							},
							imageClass: 'PRIMARY'
						},
						{
							imageSection: {
								name: 'HEAD',
								description: 'Head'
							},
							imageType: {
								'name': 'HEAD_LEFT_FACE',
								'description': 'Left Side of Face'
							},
							imageClass: 'PRIMARY'
						},
						{
							imageSection: {
								name: 'HEAD',
								description: 'Head'
							},
							imageType: {
								name: 'CLOSE_UP',
								description: 'Close up'
							},
							imageClass: 'CLOSE_UP'
						},
						{
							imageSection: {
								name: 'LOWER_LEG',
								description: 'Lower Leg'
							},
							imageType: {
								'name': 'LOWER_LEG_LL_LEFT',
								'description': 'Left Side of Lower Legs'
							},
							imageClass: 'PRIMARY'
						},
						{
							imageSection: {
								name: 'LOWER_LEG',
								description: 'Lower Leg'
							},
							imageType: {
								'name': 'LOWER_LEG_LL_RIGHT',
								'description': 'Right Side of Lower Legs'
							},
							imageClass: 'PRIMARY'
						},
						{
							imageSection: {
								name: 'LOWER_LEG',
								description: 'Lower Leg'
							},
							imageType: {
								name: 'CLOSE_UP',
								description: 'Close up'
							},
							imageClass: 'CLOSE_UP'
						},
						{
							imageSection: {
								name: 'OPTIONAL',
								description: 'Optional'
							},
							imageType: {
								name: 'OPTIONAL',
								description: 'Optional'
							},
							imageClass: 'OPTIONAL'
						}
					]
				});
				service.initUploadedImagesBySection();

				expect(service.getUploadedImagesBySection()).toEqual({
					'HEAD': {
						'PRIMARY': [
							{
								imageSection: {
									name: 'HEAD',
									description: 'Head'
								},
								imageType: {
									'name': 'HEAD_FACE',
									'description': 'Head and Face'
								},
								imageClass: 'PRIMARY'
							},
							{
								imageSection: {
									name: 'HEAD',
									description: 'Head'
								},
								imageType: {
									'name': 'HEAD_LEFT_FACE',
									'description': 'Left Side of Face'
								},
								imageClass: 'PRIMARY'
							}
						],
						'CLOSE_UP': [
							{
								imageSection: {
									name: 'HEAD',
									description: 'Head'
								},
								imageType: {
									name: 'CLOSE_UP',
									description: 'Close up'
								},
								imageClass: 'CLOSE_UP'
							}
						]
					},
					'LOWER_LEG': {
						'PRIMARY': [
							{
								imageSection: {
									name: 'LOWER_LEG',
									description: 'Lower Leg'
								},
								imageType: {
									'name': 'LOWER_LEG_LL_LEFT',
									'description': 'Left Side of Lower Legs'
								},
								imageClass: 'PRIMARY'
							},
							{
								imageSection: {
									name: 'LOWER_LEG',
									description: 'Lower Leg'
								},
								imageType: {
									'name': 'LOWER_LEG_LL_RIGHT',
									'description': 'Right Side of Lower Legs'
								},
								imageClass: 'PRIMARY'
							}
						],
						'CLOSE_UP': [
							{
								imageSection: {
									name: 'LOWER_LEG',
									description: 'Lower Leg'
								},
								imageType: {
									name: 'CLOSE_UP',
									description: 'Close up'
								},
								imageClass: 'CLOSE_UP'
							}
						]
					},
					'OPTIONAL': [
						{
							imageSection: {
								name: 'OPTIONAL',
								description: 'Optional'
							},
							imageType: {
								name: 'OPTIONAL',
								description: 'Optional'
							},
							imageClass: 'OPTIONAL'
						}
					]
				});
			});
		});

		describe('getMedsByType', function () {
			beforeEach(function () {
				spyOn(service, 'initUploadedImagesBySection');
			});

			it('should return an empty array if there are no medications', function () {
				service.setEvaluation({ medications: []});

				expect(service.getMedsByType('TOPICAL')).toEqual([]);
			});

			it('should return an empty array if there are no medications of the given type', function () {
				service.setEvaluation({
					medications: [
						{
							type: 'ORAL',
							name: 'Advil',
							dose: '2',
							frequency: '1 time',
							timeUnit: 'a day',
							missedDoses: 'None'
						},
						{
							type: 'ORAL',
							name: 'Tylenol',
							dose: '2',
							frequency: '1 time',
							timeUnit: 'a day',
							missedDoses: 'None'
						}
					]
				});

				expect(service.getMedsByType('TOPICAL')).toEqual([]);
			});

			it ('should return the medications of the given type', function () {
				service.setEvaluation({
					medications: [
						{
							type: 'ORAL',
							name: 'Advil',
							dose: '2',
							frequency: '1 time',
							timeUnit: 'a day',
							missedDoses: 'None'
						},
						{
							type: 'ORAL',
							name: 'Tylenol',
							dose: '2',
							frequency: '1 time',
							timeUnit: 'a day',
							missedDoses: 'None'
						},
						{
							type: 'TOPICAL',
							name: 'Eucerin',
							dose: '2',
							frequency: '1 time',
							timeUnit: 'a day',
							missedDoses: 'None'
						},
						{
							type: 'TOPICAL',
							name: 'Neosporin',
							dose: '1',
							frequency: '1 time',
							timeUnit: 'a day',
							missedDoses: 'None'
						}
					]
				});

				expect(service.getMedsByType('TOPICAL')).toEqual([
					{
						type: 'TOPICAL',
						name: 'Eucerin',
						dose: '2',
						frequency: '1 time',
						timeUnit: 'a day',
						missedDoses: 'None'
					},
					{
						type: 'TOPICAL',
						name: 'Neosporin',
						dose: '1',
						frequency: '1 time',
						timeUnit: 'a day',
						missedDoses: 'None'
					}
				]);
			});
		});

		describe('updateMedsByType', function () {
			beforeEach(function () {
				spyOn(service, 'initUploadedImagesBySection');
			});

			it('should not update the list if no meds are passed in', function () {
				service.setEvaluation({
					medications: [
						{
							type: 'ORAL',
							name: 'Advil',
							dose: '2',
							frequency: '1 time',
							timeUnit: 'a day',
							missedDoses: 'None'
						},
						{
							type: 'ORAL',
							name: 'Tylenol',
							dose: '2',
							frequency: '1 time',
							timeUnit: 'a day',
							missedDoses: 'None'
						}
					]
				});

				service.updateMedsByType([], 'TOPICAL');

				expect(service.getEvaluation().medications).toEqual([
					{
						type: 'ORAL',
						name: 'Advil',
						dose: '2',
						frequency: '1 time',
						timeUnit: 'a day',
						missedDoses: 'None'
					},
					{
						type: 'ORAL',
						name: 'Tylenol',
						dose: '2',
						frequency: '1 time',
						timeUnit: 'a day',
						missedDoses: 'None'
					}
				]);
			});

			it('should add the given meds to the list if there are no preexisting meds of the given type', function () {
				service.setEvaluation({
					medications: [
						{
							type: 'ORAL',
							name: 'Advil',
							dose: '2',
							frequency: '1 time',
							timeUnit: 'a day',
							missedDoses: 'None'
						},
						{
							type: 'ORAL',
							name: 'Tylenol',
							dose: '2',
							frequency: '1 time',
							timeUnit: 'a day',
							missedDoses: 'None'
						}
					]
				});

				service.updateMedsByType([{
					type: 'TOPICAL',
					name: 'Eucerin',
					dose: '2',
					frequency: '1 time',
					timeUnit: 'a day',
					missedDoses: 'None'
				}], 'TOPICAL');

				expect(service.getEvaluation().medications).toEqual([
					{
						type: 'ORAL',
						name: 'Advil',
						dose: '2',
						frequency: '1 time',
						timeUnit: 'a day',
						missedDoses: 'None'
					},
					{
						type: 'ORAL',
						name: 'Tylenol',
						dose: '2',
						frequency: '1 time',
						timeUnit: 'a day',
						missedDoses: 'None'
					},
					{
						type: 'TOPICAL',
						name: 'Eucerin',
						dose: '2',
						frequency: '1 time',
						timeUnit: 'a day',
						missedDoses: 'None'
					}
				]);
			});

			it('should replace the existing meds of that type with the given meds', function () {
				service.setEvaluation({
					medications: [
						{
							type: 'ORAL',
							name: 'Advil',
							dose: '2',
							frequency: '1 time',
							timeUnit: 'a day',
							missedDoses: 'None'
						},
						{
							type: 'ORAL',
							name: 'Tylenol',
							dose: '2',
							frequency: '1 time',
							timeUnit: 'a day',
							missedDoses: 'None'
						},
						{
							type: 'TOPICAL',
							name: 'Eucerin',
							dose: '2',
							frequency: '1 time',
							timeUnit: 'a day',
							missedDoses: 'None'
						}
					]
				});

				service.updateMedsByType([
					{
						type: 'TOPICAL',
						name: 'Neosporin',
						dose: '1',
						frequency: '1 time',
						timeUnit: 'a day',
						missedDoses: '7 times'
					},
					{
						type: 'TOPICAL',
						name: 'Lotion',
						dose: '3',
						frequency: '1 time',
						timeUnit: 'week',
						missedDoses: '5 times'
					}
				], 'TOPICAL');

				expect(service.getEvaluation().medications).toEqual([
					{
						type: 'ORAL',
						name: 'Advil',
						dose: '2',
						frequency: '1 time',
						timeUnit: 'a day',
						missedDoses: 'None'
					},
					{
						type: 'ORAL',
						name: 'Tylenol',
						dose: '2',
						frequency: '1 time',
						timeUnit: 'a day',
						missedDoses: 'None'
					},
					{
						type: 'TOPICAL',
						name: 'Neosporin',
						dose: '1',
						frequency: '1 time',
						timeUnit: 'a day',
						missedDoses: '7 times'
					},
					{
						type: 'TOPICAL',
						name: 'Lotion',
						dose: '3',
						frequency: '1 time',
						timeUnit: 'week',
						missedDoses: '5 times'
					}
				]);
			});
		});

		describe('getPrimaryImage', function () {
			beforeEach(function () {
				var uploadedImagesBySection = service.getUploadedImagesBySection();

				uploadedImagesBySection['HEAD'] = {
					'PRIMARY': [
						{
							imageSection: {
								name: 'HEAD',
								description: 'Head'
							},
							imageType: {
								name: 'HEAD_FACE',
								description: 'Head and Face'
							}
						}
					]
				};
			});

			afterEach(function () {
				var uploadedImagesBySection = service.getUploadedImagesBySection();

				delete uploadedImagesBySection['HEAD'];
			});

			it('should return undefined if the imageType isn\'t found', function () {
				expect(service.getPrimaryImage('HEAD_LEFT_FACE', 'HEAD')).toEqual(undefined);
			});

			it('should return the image object if it is found', function () {
				expect(service.getPrimaryImage('HEAD_FACE', 'HEAD')).toEqual({
					imageSection: {
						name: 'HEAD',
						description: 'Head'
					},
					imageType: {
						name: 'HEAD_FACE',
						description: 'Head and Face'
					}
				});
			});
		});

		describe('getCloseUpImage', function () {
			beforeEach(function () {
				var uploadedImagesBySection = service.getUploadedImagesBySection();

				uploadedImagesBySection['HEAD'] = {
					'CLOSE_UP': [
						{
							imageSection: {
								name: 'HEAD',
								description: 'Head'
							},
							imageType: {
								name: 'CLOSE_UP',
								description: 'Close up'
							},
							imageClass: 'CLOSE_UP',
							fileName: 'CLOSE_UP1.png'
						},
						{
							imageSection: {
								name: 'HEAD',
								description: 'Head'
							},
							imageType: {
								name: 'CLOSE_UP',
								description: 'Close up'
							},
							imageClass: 'CLOSE_UP',
							fileName: 'CLOSE_UP2.png'
						}
					]
				};
			});

			afterEach(function () {
				var uploadedImagesBySection = service.getUploadedImagesBySection();

				delete uploadedImagesBySection['HEAD'];
			});

			it('should return the close up image if found', function () {
				expect(service.getCloseUpImage('HEAD', 1)).toEqual(
					{
						imageSection: {
							name: 'HEAD',
							description: 'Head'
						},
						imageType: {
							name: 'CLOSE_UP',
							description: 'Close up'
						},
						imageClass: 'CLOSE_UP',
						fileName: 'CLOSE_UP2.png'
					}
				);
			});

			it('should return undefined if the image is not found', function () {
				expect(service.getCloseUpImage('HEAD', 2)).toEqual(undefined);
			});
		});

		describe('getOptionalImage', function () {
			beforeEach(function () {
				var uploadedImagesBySection = service.getUploadedImagesBySection();

				uploadedImagesBySection['OPTIONAL'] = [
					{
						imageSection: {
							name: 'OPTIONAL',
							description: 'Optional'
						},
						imageType: {
							name: 'OPTIONAL',
							description: 'Optional'
						},
						imageClass: 'OPTIONAL',
						fileName: 'OPTIONAL1.png'
					},
					{
						imageSection: {
							name: 'OPTIONAL',
							description: 'Optional'
						},
						imageType: {
							name: 'OPTIONAL',
							description: 'Optional'
						},
						imageClass: 'OPTIONAL',
						fileName: 'OPTIONAL2.png'
					}
				];
			});

			afterEach(function () {
				var uploadedImagesBySection = service.getUploadedImagesBySection();

				delete uploadedImagesBySection['HEAD'];
			});

			it('should return the optional image if found', function () {
				expect(service.getOptionalImage(1)).toEqual(
					{
						imageSection: {
							name: 'OPTIONAL',
							description: 'Optional'
						},
						imageType: {
							name: 'OPTIONAL',
							description: 'Optional'
						},
						imageClass: 'OPTIONAL',
						fileName: 'OPTIONAL2.png'
					}
				);
			});

			it('should return undefined if the image is not found', function () {
				expect(service.getOptionalImage(2)).toEqual(undefined);
			});
		});

		describe('createImagePlaceholder', function () {
			beforeEach(function () {
				spyOn(service, 'initUploadedImagesBySection');

				service.setEvaluation({
					images: []
				});

				service.createImagePlaceholder({
					imageSection: {
						name: 'OPTIONAL',
						description: 'Optional'
					},
					imageType: {
						name: 'OPTIONAL',
						description: 'Optional'
					},
					imageClass: 'OPTIONAL'
				});
			});

			it('should append the given image to the images array in the current evaluation', function () {
				expect(service.getEvaluation().images).toEqual([{
					imageSection: {
						name: 'OPTIONAL',
						description: 'Optional'
					},
					imageType: {
						name: 'OPTIONAL',
						description: 'Optional'
					},
					imageClass: 'OPTIONAL'
				}]);
			});

			it('should call initUploadedImagesBySection', function () {
				expect(service.initUploadedImagesBySection.calls.count()).toEqual(1);
			});
		});

		describe('deletePhotos', function () {
			var uploadedImagesBySection;
			beforeEach(function () {
				service.initEvaluation('MyTelederm');
				$scope.$apply();
				spyOn(service, 'initUploadedImagesBySection');

				service.setEvaluation({
					images: [
						{
							id: 'id1',
							imageSection: {
								name: 'HEAD',
								description: 'Head'
							},
							imageType: {
								name: 'CLOSE_UP',
								description: 'Close up'
							},
							imageClass: 'CLOSE_UP',
							imageNumber: 1
						},
						{
							id: 'id2',
							imageSection: {
								name: 'HEAD',
								description: 'Head'
							},
							imageType: {
								name: 'CLOSE_UP',
								description: 'Close up'
							},
							imageClass: 'CLOSE_UP',
							imageNumber: 2
						}
					],
					pageNumber: 10
				});

				uploadedImagesBySection = service.getUploadedImagesBySection();

				uploadedImagesBySection['HEAD'] = {
					'CLOSE_UP': [
						{
							id: 'id1',
							imageSection: {
								name: 'HEAD',
								description: 'Head'
							},
							imageType: {
								name: 'CLOSE_UP',
								description: 'Close up'
							},
							imageClass: 'CLOSE_UP',
							imageNumber: 1
						},
						{
							id: 'id2',
							imageSection: {
								name: 'HEAD',
								description: 'Head'
							},
							imageType: {
								name: 'CLOSE_UP',
								description: 'Close up'
							},
							imageClass: 'CLOSE_UP',
							imageNumber: 2
						}
					]
				};
			});

			afterEach(function () {
				var uploadedImagesBySection = service.getUploadedImagesBySection();

				delete uploadedImagesBySection['HEAD'];
			});

			it('should set deleteImage to true and imageNumber to greater than 3 for the images being deleted', function () {
				service.deletePhotos(['id1','id2']);
				expect(service.getEvaluation().images).toEqual(
					[{
						id: 'id1',
						imageSection: {
							name: 'HEAD',
							description: 'Head'
						},
						imageType: {
							name: 'CLOSE_UP',
							description: 'Close up'
						},
						imageClass: 'CLOSE_UP',
						imageNumber: 10,
						deleteImage: true
					},
						{
							id: 'id2',
							imageSection: {
								name: 'HEAD',
								description: 'Head'
							},
							imageType: {
								name: 'CLOSE_UP',
								description: 'Close up'
							},
							imageClass: 'CLOSE_UP',
							imageNumber: 10,
							deleteImage: true
						}]);
			});


			it('should set the imageNumber of the non-deleted images to less the no. of images being deleted', function () {
				service.deletePhotos(['id1']);
				expect(service.getEvaluation().images).toEqual(
					[{
						id: 'id1',
						imageSection: {
							name: 'HEAD',
							description: 'Head'
						},
						imageType: {
							name: 'CLOSE_UP',
							description: 'Close up'
						},
						imageClass: 'CLOSE_UP',
						imageNumber: 10,
						deleteImage: true
					},
						{
							id: 'id2',
							imageSection: {
								name: 'HEAD',
								description: 'Head'
							},
							imageType: {
								name: 'CLOSE_UP',
								description: 'Close up'
							},
							imageClass: 'CLOSE_UP',
							imageNumber: 1
						}]);
			});

			it('should set the pageNumber to less the no. of images being deleted', function () {
				service.deletePhotos(['id1','id2']);
				expect(service.getEvaluation().pageNumber).toEqual(8);
			});

			it('should reduce pageNumber one more than the total images deleted and set isOptionalImages to null if all optional photos have been deleted', function () {
				service.getEvaluation().images.push(
					{
						id: 'optionalId1',
						imageSection: {
							name: 'OPTIONAL',
							description: 'Optional'
						},
						imageType: {
							name: 'OPTIONAL',
							description: 'Optional'
						},
						imageClass: 'OPTIONAL',
						imageNumber: 1
					});
				service.getEvaluation().images.push({
					id: 'optionalId2',
					imageSection: {
						name: 'OPTIONAL',
						description: 'Optional'
					},
					imageType: {
						name: 'OPTIONAL',
						description: 'Optional'
					},
					imageClass: 'OPTIONAL',
					imageNumber: 2
				});

				uploadedImagesBySection['OPTIONAL'] =  [
					{
						id: 'optionalId1',
						imageSection: {
							name: 'OPTIONAL',
							description: 'Optional'
						},
						imageType: {
							name: 'OPTIONAL',
							description: 'Optional'
						},
						imageClass: 'OPTIONAL',
						imageNumber: 1
					},
					{
						id: 'optionalId2',
						imageSection: {
							name: 'OPTIONAL',
							description: 'Optional'
						},
						imageType: {
							name: 'OPTIONAL',
							description: 'Optional'
						},
						imageClass: 'OPTIONAL',
						imageNumber: 2
					}
				];

				service.deletePhotos(['optionalId1','optionalId2']);
				expect(service.getEvaluation().pageNumber).toEqual(7);
				expect(service.getEvaluation().isOptionalImages).toBeFalsy();
			});

		});
	});
});