define(['imageMediaRequestService'], function () {
    'use strict';

    describe('Image Media Request Service', function () {
        var service,
            httpMock,
            localResourceDirectoryServiceMock,
            imageResponseServiceMock;

        beforeEach(function () {
            module('angularTemplateApp');

            localResourceDirectoryServiceMock = jasmine.createSpyObj('localResourceDirectoryServiceMock', ['fetch']);
            imageResponseServiceMock = jasmine.createSpyObj('imageResponseService', ['initEvaluation', 'setImageTypesBySection', 'initUploadedImagesBySection', 'setImageSectionsMap']);

            localResourceDirectoryServiceMock.fetch.and.returnValue({
                then: function (callback) {
                    callback({
                        'image-sections': '/media-request-service/v1/media/sections',
                        'update-request': '/media-request-service/v1/patients/{UNIQUE_ID}/requests/{MEDIA_REQUEST_ID}'
                    });
                }
            });

            imageResponseServiceMock.initEvaluation.and.returnValue({
                then: function (callback) {
                    callback();
                }
            });

            module(function ($provide) {
                $provide.value('localResourceDirectoryService', localResourceDirectoryServiceMock);
                $provide.value('imageResponseService', imageResponseServiceMock);
            });

            inject(function ($q, imageMediaRequestService, $httpBackend) {
                httpMock = $httpBackend;
                
                service = imageMediaRequestService;
            });
        });

        describe('getMediaRequest and setMediaRequest', function () {
            beforeEach(function () {
                spyOn(service, 'fetchImageSections').and.returnValue({
                    then: function (callback) {
                        callback();
                    }
                });
                spyOn(service, 'initImageTypesBySection');
            });

            it('should return a default MediaRequest if none is set', function () {
                expect(service.getMediaRequest()).toEqual({});
            });

            it('should allow the user to set an MediaRequest and retrieve it', function () {
                service.setMediaRequest({
                    requestType: ''
                });

                expect(service.getMediaRequest()).toEqual({
                    requestType: ''
                });
            });

            it('should fetch the image sections from the BackEnd when the MediaRequest is set', function () {
                service.setMediaRequest({});

                expect(service.fetchImageSections).toHaveBeenCalled();
            });

            it('should initialize the evaluation object when the MediaRequest is set', function () {
                service.setMediaRequest({});

                expect(imageResponseServiceMock.initEvaluation).toHaveBeenCalled();
                expect(service.initImageTypesBySection).toHaveBeenCalled();
                expect(imageResponseServiceMock.initUploadedImagesBySection).toHaveBeenCalled();
            });
        });

        describe('fetchImageSections', function () {
            var success,
                error;

            beforeEach(function () {
                spyOn(service, 'initImageSectionsMap');
            });

            it('should resolve without making a call if the imageSectionsMap is already populated', function () {
                var imageSectionsMap = service.getImageSectionsMap();

                imageSectionsMap['HEAD'] = {
                    name: 'HEAD',
                    description: 'Head'
                };

                service.fetchImageSections();

                expect(localResourceDirectoryServiceMock.fetch).not.toHaveBeenCalled();
            });

            it('should make a call to get the image sections from the BackEnd if the imageSectionsMap doesn\'t already exist', function () {
                var imageSectionsMap = service.getImageSectionsMap();

                delete imageSectionsMap['HEAD'];

                httpMock.when('GET', '/media-request-service/v1/media/sections').respond([
                    {
                        name: 'HEAD',
                        description: 'Head'
                    },
                    {
                        name: 'HAND',
                        description: 'Hand'
                    }
                ]);

                service.fetchImageSections();

                httpMock.flush();

                expect(localResourceDirectoryServiceMock.fetch).toHaveBeenCalled();

                expect(service.initImageSectionsMap).toHaveBeenCalledWith([
                    {
                        name: 'HEAD',
                        description: 'Head'
                    },
                    {
                        name: 'HAND',
                        description: 'Hand'
                    }
                ]);
            });

            it('should throw an error if the call is unsuccessful', function () {
                httpMock.expectGET('/media-request-service/v1/media/sections').respond(400, 'error');

                var data;

                service.fetchImageSections().then(function () {
                    //Empty success function
                }, function (response) {
                    data = response;
                });

                httpMock.flush();

                expect(service.initImageSectionsMap).not.toHaveBeenCalled();

                expect(data).toEqual('error');
            });
        });

        describe('initImageSectionsMap and getImageSectionsMap', function () {
            it('should return an empty object by default', function () {
                expect(service.getImageSectionsMap()).toEqual({});
            });

            it('should return an empty object if no sections are passed in', function () {
                service.initImageSectionsMap();

                expect(service.getImageSectionsMap()).toEqual({});
            });

            it('should build the imageSectionsMap based on the given sections', function () {
                var sections = [
                    {
                        name: 'HEAD',
                        description: 'Head'
                    },
                    {
                        name: 'LOWER_LEG',
                        description: 'Lower Leg'
                    }
                ];

                service.initImageSectionsMap(sections);

                expect(service.getImageSectionsMap()).toEqual({
                    'HEAD': {
                        name: 'HEAD',
                        description: 'Head'
                    },
                    'LOWER': {
                        name: 'LOWER_LEG',
                        description: 'Lower Leg'
                    }
                });
            });
        });

        describe('getImageTypesBySection and initImageTypesBySection', function () {
            beforeEach(function () {
                var imageSectionsMap = service.getImageSectionsMap();

                imageSectionsMap['HEAD'] = {
                    name: 'HEAD',
                    description: 'Head'
                };

                imageSectionsMap['LOWER'] = {
                    name: 'LOWER_LEG',
                    description: 'Lower Leg'
                };

                spyOn(service, 'fetchImageSections').and.returnValue({
                    then: function (callback) {
                        callback();
                    }
                });
            });

            afterEach(function () {
                var imageSectionsMap = service.getImageSectionsMap();

                delete imageSectionsMap['HEAD'];

                delete imageSectionsMap['LOWER'];
            });

            it('should return an empty object if there are no image types', function () {
                service.initImageTypesBySection();

                expect(service.getImageTypesBySection()).toEqual({});
            });

            it('should correctly return the object based on the specified image types', function () {
                service.setMediaRequest({
                    mediaTypes: [
                        {
                            name: 'HEAD_FACE',
                            description: 'Head and Face'
                        },
                        {
                            name: 'HEAD_LEFT_FACE',
                            description: 'Left Side of Face'
                        },
                        {
                            name: 'LOWER_LEG_LL_LEFT',
                            description: 'Left Side of Lower Legs'
                        },
                        {
                            name: 'LOWER_LEG_LL_RIGHT',
                            description: 'Right Side of Lower Legs',
                        },
                        {
                            name: 'OTHER_TYPE',
                            description: 'Other'
                        }
                    ]
                });

                expect(service.getImageTypesBySection()).toEqual({
                    'HEAD': [
                        {
                            name: 'HEAD_FACE',
                            description: 'Head and Face'
                        },
                        {
                            name: 'HEAD_LEFT_FACE',
                            description: 'Left Side of Face'
                        }
                    ],
                    'LOWER_LEG': [
                        {
                            name: 'LOWER_LEG_LL_LEFT',
                            description: 'Left Side of Lower Legs'
                        },
                        {
                            name: 'LOWER_LEG_LL_RIGHT',
                            description: 'Right Side of Lower Legs',
                        }
                    ]
                })
            });
        });

        describe('saveMediaRequest', function () {
            beforeEach(function () {
                spyOn(service, 'fetchImageSections').and.returnValue({
                    then: function (callback) {
                        callback();
                    }
                });
                spyOn(service, 'initImageTypesBySection');
            });

            it('should call a PUT on the current MediaRequest', function () {
                var data;

                service.setMediaRequest({
                    id: 'id1'
                });

                httpMock.expectPATCH('/media-request-service/v1/patients/{UNIQUE_ID}/requests/id1').respond(200);

                service.saveMediaRequest().then(function () {
                    data = 'success!';
                });

                httpMock.flush();

                expect(localResourceDirectoryServiceMock.fetch).toHaveBeenCalled();
                expect(data).toEqual('success!');
            });

            it('should update the status to "STARTED"', function () {
                service.setMediaRequest({
                    id: 'id1',
                    status: 'STARTED'
                });

                httpMock.expectPATCH('/media-request-service/v1/patients/{UNIQUE_ID}/requests/id1').respond(200);

                service.saveMediaRequest().then(function () {});

                httpMock.flush();

                expect(service.getMediaRequest().status).toEqual('STARTED');
            });

            it('should return an error if the call is unsuccessful', function () {
                var data;

                service.setMediaRequest({
                    id: 'id1'
                });

                httpMock.expectPATCH('/media-request-service/v1/patients/{UNIQUE_ID}/requests/id1').respond(400, 'error!');

                service.saveMediaRequest().then(function () {
                    //Empty success function
                }, function (error) {
                    data = error;
                });

                httpMock.flush();

                expect(data).toEqual('error!');
            });
        });
    });
});