define(['ReviewPhotosController'], function() {
	'use strict';

	describe("The Review Photos Controller", function () {
		var controller,
			scope,
			rootScope,
			stateMock,
			modalMock,
			modalServiceMock,
			focusServiceMock,
			mediaRequestNavigationServiceMock,
			localResourceDirectoryServiceMock,
			imageMediaRequestServiceMock,
			imageResponseServiceMock,
			stateParams = {},
			imageSections = ['HEAD','HAND'],
			imageSectionsMap = { "HEAD": { description : 'Head' },"HAND" : {description : 'Hand'}},
			images = {
				"HEAD": {
					"PRIMARY": [{
						id: 'primaryId1',
						imageSection: {"name": "HEAD", "description": "Head"}
					}],
					"CLOSE_UP": [{
						id: 'closeupId1',
						imageSection: {"name": "HEAD", "description": "Head"},
						imageNumber: 1
					}]
				},
				"HAND": {
					"PRIMARY": [{
						id: 'primaryId2',
						imageSection: {"name": "HAND", "description": "Hand"},
						imageClass: "PRIMARY"
					}],
					"CLOSE_UP": [{
						id: 'closeupId2',
						imageSection: {"name": "HAND", "description": "Hand"},
						imageClass: "CLOSE_UP",
						imageNumber: 1
					}]
				},
				"OPTIONAL": [
					{
						id: 'optionalId1',
						imageClass: "OPTIONAL",
						imageNumber: 1
					},
					{
						id: 'optionalId2',
						imageClass: "OPTIONAL",
						imageNumber: 2
					}
				]
			},
			optionalImages = [
				{
					id: 'optionalId1',
					imageClass: "OPTIONAL",
					imageNumber: 1
				},
				{
					id: 'optionalId2',
					imageClass: "OPTIONAL",
					imageNumber: 2
				}
			];

		beforeEach(function () {
			module('angularTemplateApp');

			modalMock = jasmine.createSpyObj('modalMock', ['open']);
			mediaRequestNavigationServiceMock = jasmine.createSpyObj('mediaRequestNavigationService', ['previousPage', 'nextPage']);
			stateMock = jasmine.createSpyObj('$state', ['go']);
			modalServiceMock = jasmine.createSpyObj('modalService', ['showModal']);
			focusServiceMock = jasmine.createSpyObj('focusService', ['focusElement']);
			localResourceDirectoryServiceMock = jasmine.createSpyObj('localResourceDirectoryService', ['fetch']);
			imageMediaRequestServiceMock = jasmine.createSpyObj('imageMediaRequestService', ['getUploadedImagesBySection','getImageSectionsMap']);
			imageResponseServiceMock = jasmine.createSpyObj('imageResponseService', ['getUploadedImagesBySection','getEvaluation','deletePhotos']);

			imageResponseServiceMock.getUploadedImagesBySection.and.callFake(function () {
				return images;
			});

			imageResponseServiceMock.deletePhotos.and.returnValue({
				then: function (callback) {
					callback();
				}
			});

			imageMediaRequestServiceMock.getImageSectionsMap.and.callFake(function () {
				return imageSectionsMap;
			});

			imageResponseServiceMock.getEvaluation.and.callFake(function () {
				return  {
					images: [
						{
							id: 'closeupId1',
							imageSection: {"name": "HEAD", "description": "Head"},
							imageClass: "CLOSE_UP",
							imageNumber: 1
						},
						{
							id: 'closeupId2',
							imageSection: {"name": "HAND", "description": "Hand"},
							imageClass: "CLOSE_UP",
							imageNumber: 1
						},
						{
							id: 'optionalId1',
							imageClass: "OPTIONAL",
							imageNumber: 1
						}
					]
				}
			});

			module(function ($provide) {
				$provide.value('$state', stateMock);
				$provide.value('$stateParams', stateParams);
				$provide.value('$modal', modalMock);
				$provide.value('modalService', modalServiceMock);
				$provide.value('focusService', focusServiceMock);
				$provide.value('mediaRequestNavigationService', mediaRequestNavigationServiceMock);
				$provide.value('imageMediaRequestService', imageMediaRequestServiceMock);
				$provide.value('imageResponseService', imageResponseServiceMock);
				$provide.value('imageRotationService');
			});

			inject(function ($controller, $rootScope) {
				scope = $rootScope.$new();
				controller = $controller;
				rootScope = $rootScope;

				scope.reviewPhotosForm = {
					$setValidity: function () {
					},
					validationSummary: {
						validate: function () {
						}
					}
				};

				spyOn(rootScope, '$broadcast');
				spyOn(scope.reviewPhotosForm, '$setValidity');
				spyOn(scope.reviewPhotosForm.validationSummary, 'validate').and.returnValue({
					then: function (callback) {
						callback();
					}
				});
			});
		});

		describe("initial state", function () {
			it("should correctly set initial values", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.$apply();

				expect(scope.images).toEqual(images);
				expect(scope.optionalPhotos).toEqual(optionalImages);
				expect(scope.imageSections).toEqual(imageSections);
				expect(scope.deletedPhotos).toEqual([]);
			});

			it("should set submitted to true if stateParams.submitted is set to true", function () {
				stateParams.submitted = true;
				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.$apply();

				expect(scope.submitted).toEqual(true);
			});

			it("should set submitted to false if stateParams.submitted is not set", function () {
				stateParams.submitted = null;
				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.$apply();

				expect(scope.submitted).toEqual(false);
			});
		});
		describe("isIE11", function () {
			it("should set isIE11 to true if MSInputMethodContent and documentMode are defined", function () {
				window.MSInputMethodContext = true;
				document.documentMode = true;

				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.$apply();

				expect(scope.isIE11).toEqual(true);
			});

			it("should set isIE11 to false if MSInputMethodContext is defined, but documentMode is not", function () {
				window.MSInputMethodContext = true;
				document.documentMode = undefined;

				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.$apply();

				expect(scope.isIE11).toEqual(false);
			});

			it("should set isIE11 to false if documentMode is defined, but MSInputMethodContext is not", function () {
				window.MSInputMethodContext = undefined;
				document.documentMode = true;

				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.$apply();

				expect(scope.isIE11).toEqual(false);
			});

			it("should set isIE11 to false if both MSInputMethodContext and documentMode are undefined", function () {
				window.MSInputMethodContext = undefined;
				document.documentMode = undefined;

				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.$apply();

				expect(scope.isIE11).toEqual(false);
			});
		});

		describe("setDeleted and isDeleted functions", function () {
			it("should add the photoId into deletedPhotos if the photo was not already deleted, and set selected to false", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.setDeleted('closeupId1');

				expect(scope.deletedPhotos).toEqual(['closeupId1']);
				expect(scope.isSelected('closeupId1')).toBeFalsy();
			});

			it("should remove the photoId from deletedPhotos if the photo was already deleted, and set selected to true", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.deletedPhotos = ['closeupId1'];
				scope.setDeleted('closeupId1');

				expect(scope.deletedPhotos).toEqual([]);
				expect(scope.isSelected('closeupId1')).toBeTruthy();
			});
		});

		describe("getImagesByClassAndSection function", function () {
			it("should return filtered list of images based on the section", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});

				expect(scope.getImagesByClassAndSection('PRIMARY','HEAD')).toEqual([{
					id: 'primaryId1',
					imageSection: {"name": "HEAD", "description": "Head"}
				}]);
			});
		});

		describe("validateCloseups function", function () {
			it("should setValidity to false for each section if no closeups are selected for a section", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.deletedPhotos = ['closeupId1', 'closeupId2'];
				scope.validateCloseups();

				expect(scope.reviewPhotosForm.$error.length === 2);
			});

			it("should setValidity to false for only the section that has no closeups", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.deletedPhotos = ['closeupId1'];
				scope.validateCloseups();

				expect(scope.reviewPhotosForm.$error.length === 1);
			});

			it("should setValidity to true if each section has at least one closeup", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.validateCloseups();

				expect(scope.reviewPhotosForm.$error.length === 0);
			});

			it("should have error messages specifying the section that has no closeups", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.deletedPhotos = ['closeupId1'];
				scope.validateCloseups();

				expect(scope.errorHandling.CloseUp_Required_1.message).toEqual('You must select at least one close up photo for Head.');
			});
		});

		describe("showDeleteConfirmation function", function () {
			it("should add a confirmation message for each photo being deleted", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.deletedPhotos = ['closeupId1', 'optionalId1'];
				scope.showDeleteConfirmation();

				var modalOptions = {
					closeButtonText: 'No',
					actionButtonText: 'Yes',
					headerText: 'Confirmation',
					bodyContentUrl: 'modules/ui-components/modals/confirm-delete/confirm-delete_template.html',
					data: {
						messages: ['Head Close up 1', 'Optional Photo 1']
					}};
				expect(modalServiceMock.showModal).toHaveBeenCalledWith({}, modalOptions);
			});
		});


		describe("previous function", function () {
			it("should call validate first then the previousPage function on the mediaRequestNavigationService", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				spyOn(scope, 'validateCloseups');
				scope.previous();

				expect(scope.validateCloseups).toHaveBeenCalled();
				expect(mediaRequestNavigationServiceMock.previousPage).toHaveBeenCalled();
			});

			it("should call the showDeleteConfirmation function if there are photos selected for delete", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				spyOn(scope, 'showDeleteConfirmation').and.returnValue({
					then: function () {
						return true;
					}
				});
				scope.deletedPhotos = ['closeupId1'];
				scope.previous();

				expect(scope.showDeleteConfirmation).toHaveBeenCalled();
			});

			it("should call previousPage if there are photos selected for delete and user confirms Yes", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				spyOn(scope, 'showDeleteConfirmation').and.returnValue({
					then: function (callback) {
						callback();
					}
				});
				scope.deletedPhotos = ['closeupId1'];
				scope.previous();

				expect(scope.showDeleteConfirmation).toHaveBeenCalled();
				expect(imageResponseServiceMock.deletePhotos).toHaveBeenCalled();
				expect(mediaRequestNavigationServiceMock.previousPage).toHaveBeenCalled();
			});

			it("should call not call previousPage if there are photos selected for delete and user cancels confirmation", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				spyOn(scope, 'showDeleteConfirmation').and.returnValue({
					then: function (callback) {
						return false;
					}
				});
				scope.deletedPhotos = ['closeupId1'];
				scope.previous();

				expect(scope.showDeleteConfirmation).toHaveBeenCalled();
				expect(imageResponseServiceMock.deletePhotos).not.toHaveBeenCalled();
				expect(mediaRequestNavigationServiceMock.previousPage).not.toHaveBeenCalled();
			});
		});

		describe("next function", function () {
			it("should call validate first then the nextPage function on the mediaRequestNavigationService", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				spyOn(scope, 'validateCloseups');
				scope.next();

				expect(scope.validateCloseups).toHaveBeenCalled();
				expect(mediaRequestNavigationServiceMock.nextPage).toHaveBeenCalled();
			});
		});

        describe("swipeLeft function", function () {
			it("should broadcast swipeLeft if submitted is false", function () {
                controller = controller('ReviewPhotosController', {$scope: scope});
				scope.submitted = false;
				scope.swipeLeft();
				
				expect(rootScope.$broadcast).toHaveBeenCalledWith('swipeLeft');
			});

			it("should not broadcast swipeLeft if submitted is true", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.submitted = true;
				scope.swipeLeft();
				
				expect(rootScope.$broadcast).not.toHaveBeenCalled();
			});
        });
        
        describe("swipeRight function", function () {
			it("should broadcast swipeRight if submitted is false", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.submitted = false;
				scope.swipeRight();
				
				expect(rootScope.$broadcast).toHaveBeenCalledWith('swipeRight');
			});

			it("should not broadcast swipeRight if submitted is true", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.submitted = true;
				scope.swipeRight();
				
				expect(rootScope.$broadcast).not.toHaveBeenCalled();
			});
		});
		
		describe("back function", function () {
			it("should redirect the user to the media request info page", function () {
				controller = controller('ReviewPhotosController', {$scope: scope});
				scope.back();

				expect(stateMock.go).toHaveBeenCalledWith('main.mytelederm.media-request-info');
			});
		});
	});
});