define(['ResponseSummaryController'], function() {
    'use strict';

    describe("The Response Summary Controller", function () {
        var controller,
            rootScope,
            scope,
            formatterObj,
            imageResponseServiceMock,
            mediaRequestNavigationServiceMock,
            isMock,
            evaluation,
            medicationTypes = [
                'TOPICAL',
                'ORAL',
                'INJECTED'
            ];

        beforeEach(function () {
            module('angularTemplateApp');

            evaluation = {
                medicationTypes: []
            };
            
            imageResponseServiceMock = jasmine.createSpyObj('imageResponseService', ['getEvaluation']);
            mediaRequestNavigationServiceMock = jasmine.createSpyObj('mediaRequestNavigationService', ['previousPage', 'nextPage']);

            imageResponseServiceMock.getEvaluation.and.callFake(function () {
                return evaluation;
            });

            module(function ($provide) {
                $provide.value('medicationTypes', medicationTypes);
                $provide.value('imageResponseService', imageResponseServiceMock);
                $provide.value('mediaRequestNavigationService', mediaRequestNavigationServiceMock);
            });

            inject(function($controller, $rootScope, formatter) {
                scope = $rootScope.$new();
                controller = $controller;
                formatterObj = formatter;
                rootScope = $rootScope;

                spyOn(rootScope, '$broadcast');
            });
        });

		describe("initial state", function () {
            it("should correctly set initial values", function() {
                controller = controller('ResponseSummaryController', {$scope: scope});
                scope.$apply();

                expect(scope.evaluation).toEqual(evaluation);
                expect(scope.toTitleCase).toEqual(formatterObj.toTitleCase);
            });
        });

        describe("getMedsByType function", function () {
            it("should return only specified meds by type", function () {
                evaluation = {
                    medications: [
                        {
                            type: 'TOPICAL',
                            name: 'testMed1'
                        },
                        {
                            type: 'ORAL',
                            name: 'testMed2'
                        }
                    ],
                    medicationTypes: ['TOPICAL', 'ORAL']
                };

                controller = controller('ResponseSummaryController', {$scope: scope});
                scope.$apply();

                var meds = scope.getMedsByType('TOPICAL');

                expect(meds).toEqual([
                    {
                        type: 'TOPICAL',
                        name: 'testMed1'
                    }
                ]);
            });
        });

        describe("hasComments and comments", function () {
            it("should return No if there are no comments", function () {
                evaluation = {
                    medicationTypes: [],
                    additionalComments: 'N/A'
                };

                controller = controller('ResponseSummaryController', {$scope: scope});
                scope.$apply();

                expect(scope.hasComments + scope.comments).toEqual('No');
            });

            it("should return Yes and the comments if there are comments", function () {
                evaluation = {
                    medicationTypes: [],
                    additionalComments: 'Some Comments'
                };

                controller = controller('ResponseSummaryController', {$scope: scope});
                scope.$apply();

                expect(scope.hasComments + scope.comments).toEqual('Yes - Some Comments');
            });
        });

        describe("hasMeds", function () {
            it("should be false if there are no med types", function () {
                evaluation = {
                    medicationTypes: []
                };

                controller = controller('ResponseSummaryController', {$scope: scope});
                scope.$apply();

                expect(scope.hasMeds).toBeFalsy();
            });

            it("should be true if there are med types", function () {
                evaluation = {
                    medicationTypes: [
                        'TOPICAL',
                        'ORAL',
                        'OTHER'
                    ]
                };

                controller = controller('ResponseSummaryController', {$scope: scope});
                scope.$apply();

                expect(scope.hasMeds).toBeTruthy();
            });
        });

        describe("medTypes", function () {
            it("should retrieve the med types without 'OTHER'", function () {
                evaluation = {
                    medicationTypes: [
                        'TOPICAL',
                        'ORAL',
                        'OTHER'
                    ]
                };

                controller = controller('ResponseSummaryController', {$scope: scope});
                scope.$apply();

                expect(scope.medTypes).toEqual([
                    'TOPICAL',
                    'ORAL'
                ]);
            });
        });

        describe("hasOtherMeds", function () {
            it("should return true if 'OTHER' was selected", function () {
                evaluation = {
                    medicationTypes: [
                        'OTHER'
                    ],
                    otherMedication: 'Some other medication'
                };

                controller = controller('ResponseSummaryController', {$scope: scope});
                scope.$apply();

                expect(scope.hasOtherMeds).toBeTruthy();
            });

            it("should return false if 'OTHER' was not selected", function () {
                evaluation = {
                    medicationTypes: []
                };

                controller = controller('ResponseSummaryController', {$scope: scope});
                scope.$apply();

                expect(scope.hasOtherMeds).toBeFalsy();
            });
        });

        describe("previous function", function () {
            it("should call the previousPage function on the mediaRequestNavigationService", function () {
                controller = controller('ResponseSummaryController', {$scope: scope});
                scope.previous();

                expect(mediaRequestNavigationServiceMock.previousPage).toHaveBeenCalled();
            });
        });

        describe("accept function", function () {
            it("should set isResponsesAccepted on the evaluation to true", function () {
                controller = controller('ResponseSummaryController', {$scope: scope});
                scope.accept();

                expect(evaluation.isResponsesAccepted).toEqual(true);
            });

            it("should call the nextPage function on the mediaRequestNavigationService", function () {
                controller = controller('ResponseSummaryController', {$scope: scope});
                scope.accept();

                expect(mediaRequestNavigationServiceMock.nextPage).toHaveBeenCalled();
            });
        });

        describe("swipeLeft function", function () {
			it("should broadcast swipeLeft", function () {
                controller = controller('ResponseSummaryController', {$scope: scope});
                scope.swipeLeft();
				expect(rootScope.$broadcast).toHaveBeenCalledWith('swipeLeft');
			});
        });
    });
});