define(['RequiredPhotoController'], function() {
    'use strict';

    describe("The Required Photo Controller", function () {
        var controller,
            scope,
            modalMock,
            imageUploadServiceMock,
            focusServiceMock,
            imageResponseServiceMock,
            mediaRequestNavigationServiceMock,
            image,
            stateParams = {
                imageSection: {
                    name: 'HEAD',
                    description: 'Head'
                },
                imageType: {
                    name: 'HEAD_FACE',
                    description: 'Head and Face'
                },
                imageNum: 1,
                totalImagesInSection: 3
            },
            configServiceValues = {
                imageFileSizeLimit: 3,
                suggestedVideoSize: 10,
                videoFileSizeLimit: 20
            };

        beforeEach(function () {
            module('angularTemplateApp');

            modalMock = jasmine.createSpyObj('$modal', ['open']);
            imageUploadServiceMock = jasmine.createSpyObj('imageUploadService', ['uploadImage']);
            focusServiceMock = jasmine.createSpyObj('focusService', ['focusElement']);
            imageResponseServiceMock = jasmine.createSpyObj('imageMediaRequestService', ['getPrimaryImage', 'getEvaluation']);
            mediaRequestNavigationServiceMock = jasmine.createSpyObj('mediaRequestNavigationService', ['nextPage']);

            imageUploadServiceMock.uploadImage.and.returnValue({
                then: function (callback) {
                    callback();
                }
            });

            imageResponseServiceMock.getPrimaryImage.and.callFake(function () {
                return image;
            });

            imageResponseServiceMock.getEvaluation.and.returnValue({});

            module(function ($provide) {
                $provide.value('$stateParams', stateParams);
                $provide.value('configServiceValues', configServiceValues);
                $provide.value('$modal', modalMock);
                $provide.value('imageUploadService', imageUploadServiceMock);
                $provide.value('imageRotationService', {});
                $provide.value('focusService', focusServiceMock);
                $provide.value('imageResponseService', imageResponseServiceMock);
                $provide.value('mediaRequestNavigationService', mediaRequestNavigationServiceMock);
            });

            inject(function($controller, $rootScope) {
                scope = $rootScope.$new();
                controller = $controller;
            });
        });

		describe("initial state", function () {
            it("should correctly set initial values", function() {
                controller = controller('RequiredPhotoController', {$scope: scope});
                scope.$apply();

                expect(scope.imageSection).toEqual({
                    name: 'HEAD',
                    description: 'Head'
                });
                expect(scope.imageType).toEqual({
                    name: 'HEAD_FACE',
                    description: 'Head and Face'
                }); 
                expect(scope.imageNum).toEqual(1);
                expect(scope.totalImagesInSection).toEqual(3);
                expect(scope.imageTypeTemplate).toEqual('modules/applets/mytelederm/pages/photo-input/partials/_required-photo.html');
            });

            it("should call getPrimaryImage to set the image value in the scope", function () {
                image = {
                    imageSection: {
                        name: 'HEAD',
                        description: 'Head'
                    },
                    imageType: {
                        name: 'HEAD_FACE',
                        description: 'Head and Face'
                    }
                };

                controller = controller('RequiredPhotoController', {$scope: scope});
                scope.$apply();

                expect(imageResponseServiceMock.getPrimaryImage).toHaveBeenCalledWith('HEAD_FACE', 'HEAD');
                expect(scope.image).toEqual(image);
            });

            describe("imageSelected", function () {
                it("should set imageSelected to false if the image is undefined", function () {
                    image = undefined;

                    controller = controller('RequiredPhotoController', {$scope: scope});
                    scope.$apply();

                    expect(scope.imageSelected).toBeFalsy();
                });

                it("should set imageSelected to true if the image is defined", function () {
                    image = {
                        imageSection: {
                            name: 'HEAD',
                            description: 'Head'
                        },
                        imageType: {
                            name: 'HEAD_FACE',
                            description: 'Head and Face'
                        }
                    };

                    controller = controller('RequiredPhotoController', {$scope: scope});
                    scope.$apply();

                    expect(scope.imageSelected).toBeTruthy();
                });
            });

            describe("hideNext", function () {
                it("should set hideNext to true if image and currPhoto are undefined", function () {
                    image = undefined;
                    scope.currPhoto = undefined;

                    controller = controller('RequiredPhotoController', {$scope: scope});
                    scope.$apply();

                    expect(scope.hideNext).toBeTruthy();
                });

                it("should set hideNext to false if image is defined and currPhoto is undefined", function () {
                    image = {
                        imageSection: {
                            name: 'HEAD',
                            description: 'Head'
                        },
                        imageType: {
                            name: 'HEAD_FACE',
                            description: 'Head and Face'
                        }
                    };
                    scope.currPhoto = undefined;

                    controller = controller('RequiredPhotoController', {$scope: scope});
                    scope.$apply();

                    expect(scope.hideNext).toBeFalsy();
                });

                it("should set hideNext to true if image is undefined, currPhoto is defined, and currPhoto.fileObj is undefined", function () {
                    image = undefined;
                    scope.currPhoto = {};

                    controller = controller('RequiredPhotoController', {$scope: scope});
                    scope.$apply();

                    expect(scope.hideNext).toBeTruthy();
                });

                it("should set hideNext to true if image is undefined, currPhoto is defined, and currPhoto.fileObj is defined", function () {
                    image = undefined;
                    scope.currPhoto = {
                        fileObj: {}
                    };

                    controller = controller('RequiredPhotoController', {$scope: scope});
                    scope.$apply();

                    expect(scope.hideNext).toBeTruthy();
                });

                it("should set hideNext to false if image is defined, currPhoto is defined, and currPhoto.fileObj is undefined", function () {
                    image = {
                        imageSection: {
                            name: 'HEAD',
                            description: 'Head'
                        },
                        imageType: {
                            name: 'HEAD_FACE',
                            description: 'Head and Face'
                        }
                    };
                    scope.currPhoto = {};

                    controller = controller('RequiredPhotoController', {$scope: scope});
                    scope.$apply();

                    expect(scope.hideNext).toBeFalsy();
                });

                it("should set hideNext to true if image is defined, currPhoto is defined, and currPhoto.fileObj is defined", function () {
                    image = {
                        imageSection: {
                            name: 'HEAD',
                            description: 'Head'
                        },
                        imageType: {
                            name: 'HEAD_FACE',
                            description: 'Head and Face'
                        }
                    };
                    scope.currPhoto = {
                        fileObj: {}
                    };

                    controller = controller('RequiredPhotoController', {$scope: scope});
                    scope.$apply();

                    expect(scope.hideNext).toBeTruthy();
                });
            });
        });

        describe("acceptPhotoCallback function", function () {
            beforeEach(function () {
                controller = controller('RequiredPhotoController', {$scope: scope});
                scope.$apply();
            });

            it("should call the nextPage function on the mediaRequestNavigationService after uploading the image if currPhoto.fileObj is set", function () {
                scope.currPhoto = {
                    fileObj: {
                        name: 'file.jpg'
                    }
                };

                scope.acceptPhotoCallback();

                expect(imageUploadServiceMock.uploadImage).toHaveBeenCalled();
                expect(mediaRequestNavigationServiceMock.nextPage).toHaveBeenCalled();
            });

            it("should call uploadImage with new image data if currPhoto.fileObj is defined but image is not", function () {
                scope.image = null;
                scope.currPhoto = {
                    fileObj: {
                        name: 'file.jpg'
                    }
                };

                scope.acceptPhotoCallback();

                expect(imageUploadServiceMock.uploadImage).toHaveBeenCalledWith(scope.currPhoto.fileObj, {
                    imageSection: scope.imageSection,
                    imageType: scope.imageType,
                    imageClass: 'PRIMARY',
                    fileName: 'file.jpg'
                }, false);
            });

            it("should call uploadImage with existing image data if currPhoto.fileObj and image are defined", function () {
                scope.image = {
                    id: 'someId',
                    imageUrl: 'http://some-url/image-content-url',
                    imageSection: {
                        name: 'HEAD',
                        description: 'Head'
                    },
                    imageType: {
                        name: 'HEAD_FACE',
                        description: 'Head and Face'
                    }
                }
                scope.currPhoto = {
                    fileObj: {
                        name: 'file.jpg'
                    }
                };

                scope.acceptPhotoCallback();

                expect(imageUploadServiceMock.uploadImage).toHaveBeenCalledWith(scope.currPhoto.fileObj, {
                    id: 'someId',
                    imageUrl: 'http://some-url/image-content-url',
                    imageSection: {
                        name: 'HEAD',
                        description: 'Head'
                    },
                    imageType: {
                        name: 'HEAD_FACE',
                        description: 'Head and Face'
                    },
                    fileName: 'file.jpg'
                }, false);
            });
        });

        describe("tip function", function () {
            it("should open a modal with the correct options", function () {
                controller = controller('RequiredPhotoController', {$scope: scope});
                scope.tip();
    
                var modalObject = modalMock.open.calls.argsFor(0)[0];
    
                expect(modalObject.windowTemplateUrl).toEqual('modules/ui-components/modals/helper/modal-window_template.html');
                expect(modalObject.templateUrl).toEqual('modules/ui-components/modals/photo-tips/photo-tips_template.html');
                expect(modalObject.controller).toEqual('PhotoTipsController');
                expect(modalObject.backdrop).toEqual('static');
                expect(modalObject.keyboard).toEqual(false);
                expect(modalObject.resolve.modalInfo()).toEqual({value: null, dismissed: true, resendHash: []});
            });
        });

        it("should extend the PhotoInputController", function () {
            controller = controller('RequiredPhotoController', {$scope: scope});
            spyOn(scope, 'validate');

            scope.photoInputForm = {
                validationSummary: {
                    validate: function () {}
                }
            };
            spyOn(scope.photoInputForm.validationSummary, 'validate').and.returnValue({
                then: function (callback) {
                    callback();
                }
            });
            
            spyOn(scope, 'acceptPhotoCallback');

            scope.acceptPhoto();

            expect(scope.validate).toHaveBeenCalled();
            expect(scope.acceptPhotoCallback).toHaveBeenCalled();
        });
    });
});