define(['PhotoInputController'], function() {
    'use strict';

    describe("The Photo Input Controller", function () {
        var controller,
            scope,
            rootScope,
            httpMock,
            timeoutMock,
            timeoutCallback,
            focusServiceMock,
            imageRotationServiceMock,
            imageResponseServiceMock,
            mediaRequestNavigationServiceMock,
            fileValidationServiceMock,
            evaluation = {},
            imageResponse = {
                data: {
                    content: {
                        'contentType': 'image',
                        'data': 'image-content'
                    }
                }
            },
            userAgent = '',
            originalNavigator = navigator,
            fileTypes = ['png', 'jpg', 'jpeg', 'gif'],
            isfileSizeValid,
            isfileTypeValid,
            formatterMock,
            modalMock,
            fakeModal = {
                result: {
                    then: function(confirmCallback) {
                        this.confirmCallBack = confirmCallback;
                    }
                },
                close: function( item ) {
                    this.result.confirmCallBack( item );
                }
            };

        beforeEach(function () {
            module('angularTemplateApp');
            
            httpMock = jasmine.createSpyObj('httpMock', ['$http']);
            timeoutMock = jasmine.createSpyObj('timeoutMock', ['$timeout']);
            focusServiceMock = jasmine.createSpyObj('focusService', ['focusElement']);
            imageRotationServiceMock = jasmine.createSpyObj('imageRotationService', ['getRotateClass']);
            imageResponseServiceMock = jasmine.createSpyObj('imageResponseService', ['getEvaluation', 'initUploadedImagesBySection']);
            mediaRequestNavigationServiceMock = jasmine.createSpyObj('mediaRequestNavigationService', ['previousPage', 'nextPage']);
            fileValidationServiceMock = jasmine.createSpyObj('fileValidationServiceMock', ['validateFileSize', 'validateFileType']);
            formatterMock = jasmine.createSpyObj('formatter', ['base64toBlob']);
            modalMock = jasmine.createSpyObj('$modal', ['open', 'close', 'result']);

            httpMock.$http.and.returnValue({
                then: function (callback) {
                    callback(imageResponse);
                }
            });

            timeoutMock.$timeout.and.callFake(function (callback) {
                timeoutCallback = callback;
            });

            imageRotationServiceMock.getRotateClass.and.returnValue({
                then: function (callback) {
                    callback('rotate-180');
                }
            });

            imageResponseServiceMock.getEvaluation.and.callFake(function () {
                return evaluation;
            });

            fileValidationServiceMock.validateFileSize.and.callFake(function () {
                return isfileSizeValid;
            });

            fileValidationServiceMock.validateFileType.and.callFake(function () {
                return isfileTypeValid;
            });

            formatterMock.base64toBlob.and.callFake(function () {
                return 'image-content';
            });

            modalMock.open.and.callFake(function(){
                return fakeModal;
            });

            navigator = new Object();
            navigator.__proto__ = originalNavigator;
            navigator.__defineGetter__('userAgent', function () { 
                return userAgent; 
            });

            module(function ($provide) {
                $provide.value('$timeout', timeoutMock.$timeout);
                $provide.value('$http', httpMock.$http);
                $provide.value('focusService', focusServiceMock);
                $provide.value('imageRotationService', imageRotationServiceMock);
                $provide.value('imageResponseService', imageResponseServiceMock);
                $provide.value('mediaRequestNavigationService', mediaRequestNavigationServiceMock);
                $provide.value('fileValidationService', fileValidationServiceMock);
                $provide.value('formatter', formatterMock);
                $provide.value('$modal', modalMock);
            });

            inject(function($controller, $rootScope) {
                scope = $rootScope.$new();
                scope.imageFileSizeLimit = 3;
                controller = $controller;
                rootScope = $rootScope;

                spyOn(rootScope, '$broadcast');
            });
        });

        afterEach(function () {
            navigator = originalNavigator;
        });

		describe("initial state", function () {
            beforeEach(function () {
                evaluation = {};
            });

            it("should correctly set initial values", function() {
                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();
                expect(scope.photoComments).toEqual('');
            });

            describe("isIE11", function () {
                it("should set isIE11 to true if MSInputMethodContent and documentMode are defined", function () {
                    window.MSInputMethodContext = true;
                    document.documentMode = true;

                    controller = controller('PhotoInputController', {$scope: scope});
                    scope.$apply();

                    expect(scope.isIE11).toEqual(true);
                });

                it("should set isIE11 to false if MSInputMethodContext is defined, but documentMode is not", function () {
                    window.MSInputMethodContext = true;
                    document.documentMode = undefined;

                    controller = controller('PhotoInputController', {$scope: scope});
                    scope.$apply();

                    expect(scope.isIE11).toEqual(false);
                });

                it("should set isIE11 to false if documentMode is defined, but MSInputMethodContext is not", function () {
                    window.MSInputMethodContext = undefined;
                    document.documentMode = true;

                    controller = controller('PhotoInputController', {$scope: scope});
                    scope.$apply();

                    expect(scope.isIE11).toEqual(false);
                });

                it("should set isIE11 to false if both MSInputMethodContext and documentMode are undefined", function () {
                    window.MSInputMethodContext = undefined;
                    document.documentMode = undefined;

                    controller = controller('PhotoInputController', {$scope: scope});
                    scope.$apply();

                    expect(scope.isIE11).toEqual(false);
                });
            });

            describe("showTakeAnother", function () {
                it("should set showTakeAnother to false if imageNum is undefined", function () {
                    controller = controller('PhotoInputController', {$scope: scope});
                    scope.$apply();
                    expect(scope.showTakeAnother).toEqual(false);
                });

                it("should set showTakeAnother to true if imageNum is less than 3", function () {
                    scope.imageNum = 1;
                    controller = controller('PhotoInputController', {$scope: scope});
                    scope.$apply();
                    expect(scope.showTakeAnother).toEqual(true);
                });

                it("should set showTakeAnother to false if imageNum is equal to 3", function () {
                    scope.imageNum = 3;
                    controller = controller('PhotoInputController', {$scope: scope});
                    scope.$apply();
                    expect(scope.showTakeAnother).toEqual(false);
                });

                it("should set showTakeAnother to false if imageNum is greater than 3", function () {
                    scope.imageNum = 4;
                    controller = controller('PhotoInputController', {$scope: scope});
                    scope.$apply();
                    expect(scope.showTakeAnother).toEqual(false);
                });
            });

            it("should set currPhoto to a default value if image is not set", function () {
                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();
                expect(scope.currPhoto).toEqual({});
            });

            it("should have the correct errorHandling object set", function () {
                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();
                expect(scope.errorHandling).toEqual({
                    'File-Size-Check' : {
                        message : 'Image file can be no bigger than 3 MB.',
                        priority : 1
                    },
                    'File-Type-Check' : {
                        message : 'Images must be one of the following types: gif, jpg, jpeg, png',
                        priority : 1
                    }
                });
            });
        });

        describe("currPhoto.src on a non-android device", function () {
            beforeEach(function () {
                userAgent = '';
            });
            
            it("should not set src on currPhoto if image is not set", function () {
                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();

                expect(scope.currPhoto.src).toBeFalsy();
            });

            it("should not set src on currPhoto if imageUrl is not set on the image object", function () {
                scope.image = {};
                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();

                expect(scope.currPhoto.src).toBeFalsy();
            });

            it("should set src on currPhoto if imageUrl is set on the image object", function () {
                scope.image = {
                    imageUrl: 'file.jpg'
                };
                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();

                expect(scope.currPhoto.src).toContain('data:image;base64,image-content');
            });

            it("should not execute the functionality to auto-rotate images", function () {
                scope.image = {
                    imageUrl: 'file.jpg'
                };
                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();

                expect(imageRotationServiceMock.getRotateClass).not.toHaveBeenCalled();
            });
        });

        describe("auto-rotate images on android", function () {
            beforeEach(function () {
                userAgent = 'android';
            });
            
            it("should not set src on currPhoto if image is not set", function () {
                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();

                expect(scope.currPhoto.src).toBeFalsy();
                expect(httpMock.$http).not.toHaveBeenCalled();
            });

            it("should not set src on currPhoto if imageUrl is not set on the image object", function () {
                scope.image = {};
                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();

                expect(scope.currPhoto.src).toBeFalsy();
                expect(httpMock.$http).not.toHaveBeenCalled();
            });

            it("should make a call to get the blob object if imageUrl is set on the image object", function () {
                scope.image = {
                    imageUrl: 'file.jpg'
                };
                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();

                expect(httpMock.$http).toHaveBeenCalledWith({
                    url: 'file.jpg',
                    method: 'GET'
                });
            });

            it("should call getRotateClass on the imageRotationService with the blob object", function () {
                scope.image = {
                    imageUrl: 'file.jpg'
                };
                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();

                expect(imageRotationServiceMock.getRotateClass).toHaveBeenCalledWith('image-content');
            });

            it("should update the currPhoto object with the result of the getRotateClass function", function () {
                scope.image = {
                    imageUrl: 'file.jpg'
                };
                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();

                expect(scope.currPhoto.src).toContain('data:image;base64,image-content');
                expect(scope.currPhoto.rotateClass).toEqual('rotate-180');
            });
        });

        describe("showDescriptionRequired attribute", function () {
            it("should be set to false if showImageRequired is true and image is undefined", function () {
                scope.showImageRequired = true;

                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();

                expect(scope.showDescriptionRequired).toEqual(false);
            });

            it("should be set to false if showImageRequired is true and image has no id", function () {
                scope.showImageRequired = true;
                scope.image = {};

                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();

                expect(scope.showDescriptionRequired).toEqual(false);
            });

            it("should be set to false if showImageRequired is true and image has an id", function () {
                scope.showImageRequired = true;
                scope.image = {
                    id: 'someId'
                };

                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();

                expect(scope.showDescriptionRequired).toEqual(false);
            });

            it("should be set to false if showImageRequired is false and image is undefined", function () {
                scope.showImageRequired = false;

                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();

                expect(scope.showDescriptionRequired).toEqual(false);
            });

            it("should be set to false if showImageRequired is false and image has no id", function () {
                scope.showImageRequired = false;
                scope.image = {};

                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();

                expect(scope.showDescriptionRequired).toEqual(false);
            });

            it("should be set to true if showImageRequired is false and image has an id", function () {
                scope.showImageRequired = false;
                scope.image = {
                    id: 'someId'
                };

                controller = controller('PhotoInputController', {$scope: scope});
                scope.$apply();

                expect(scope.showDescriptionRequired).toEqual(true);
            });
        });

        describe("retakePhoto function", function () {
            beforeEach(function () {
                controller = controller('PhotoInputController', {$scope: scope});

                spyOn(scope, 'takePhoto');

                scope.imageAccepted = true;

                scope.retakePhoto();
            });

            it("should set imageAccepted to false", function () {
                expect(scope.imageAccepted).toEqual(false);
            });

            it("should call takePhoto", function () {
                expect(scope.takePhoto).toHaveBeenCalled();
            });
        });

        describe("takePhoto function", function () {
            it("should set imageSelected to true", function () {
                controller = controller('PhotoInputController', {$scope: scope});

                scope.takePhoto();
                expect(scope.imageSelected).toEqual(true);
            });

            it("should set hideNext to true", function () {
                controller = controller('PhotoInputController', {$scope: scope});

                scope.hideNext = false;
                scope.takePhoto();
                expect(scope.hideNext).toEqual(true);
            });

            it("should set showDescriptionRequired to the opposite of showImageRequired", function () {
                controller = controller('PhotoInputController', {$scope: scope});

                scope.showImageRequired = true;
                scope.takePhoto();
                expect(scope.showDescriptionRequired).toEqual(false);

                scope.showImageRequired = false;
                scope.takePhoto();
                expect(scope.showDescriptionRequired).toEqual(true);
            });

            it("should wait 700ms before setting focus on a non-android device", function () {
                userAgent = '';
                controller = controller('PhotoInputController', {$scope: scope});
                
                scope.takePhoto();
                expect(timeoutMock.$timeout.calls.argsFor(0)[1]).toEqual(700);

                timeoutCallback();

                expect(focusServiceMock.focusElement).toHaveBeenCalledWith('.media-container img');
            });

            it("should call getRotate on the imageRotationService on an android device", function () {
                userAgent = 'android';
                controller = controller('PhotoInputController', {$scope: scope});

                scope.currPhoto = {
                    fileObj: {
                        src: 'file.jpg'
                    }
                };

                scope.takePhoto();

                expect(imageRotationServiceMock.getRotateClass).toHaveBeenCalledWith({
                    src: 'file.jpg'
                });
            });

            it("should update the currPhoto object with the result of the getRotateClass function", function () {
                userAgent = 'android';
                controller = controller('PhotoInputController', {$scope: scope});

                scope.currPhoto = {
                    fileObj: {
                        src: 'file.jpg'
                    }
                };

                scope.takePhoto();

                expect(scope.currPhoto.rotateClass).toEqual('rotate-180');
            });

            it("should wait 700ms before setting focus on an android device", function () {
                userAgent = 'android';
                controller = controller('PhotoInputController', {$scope: scope});
                
                scope.takePhoto();
                expect(timeoutMock.$timeout.calls.argsFor(0)[1]).toEqual(700);

                timeoutCallback();

                expect(focusServiceMock.focusElement).toHaveBeenCalledWith('.media-container img');
            });
        });

        describe("validateFileSize function", function () {
            beforeEach(function () {
                controller = controller('PhotoInputController', {$scope: scope});

                scope.photoInputForm = {
                    $setValidity: function () {}
                };

                spyOn(scope.photoInputForm, '$setValidity');
            });

            it("should validate successfully file size validation passes", function () {
                isfileSizeValid = true;

                scope.validateFileSize();

                expect(scope.photoInputForm.$setValidity).toHaveBeenCalledWith('File-Size-Check', true);
            });

            it("should not validate successfully if the file size validation fails", function () {
                isfileSizeValid = false;

                scope.validateFileSize();

                expect(scope.photoInputForm.$setValidity).toHaveBeenCalledWith('File-Size-Check', false);
            });
        });

        describe("validateFileType function", function () {
            var fileExt = '';

            beforeEach(function () {
                controller = controller('PhotoInputController', {$scope: scope});

                scope.photoInputForm = {
                    $setValidity: function () {}
                };

                spyOn(scope.photoInputForm, '$setValidity');
            });

            it('should use the fileName from the image object, if present', function () {
                scope.image = {
                    fileName: 'file.jpg'
                };

                scope.validateFileType();

                expect(fileValidationServiceMock.validateFileType).toHaveBeenCalledWith(scope.image.fileName, fileTypes);
            });

            it('should use the name from the fileObj, if fileObj is present and image is not', function () {
                scope.image = undefined;
                scope.currPhoto.fileObj = {
                    name: 'fileObj.jpg'
                };

                scope.validateFileType();

                expect(fileValidationServiceMock.validateFileType).toHaveBeenCalledWith(scope.currPhoto.fileObj.name, fileTypes);
            });

            it('should use the name from the fileObj if image doesn\'t have a fileName', function () {
                scope.image = {};
                scope.currPhoto.fileObj = {
                    name: 'fileObj.jpg'
                };

                scope.validateFileType();

                expect(fileValidationServiceMock.validateFileType).toHaveBeenCalledWith(scope.currPhoto.fileObj.name, fileTypes);
            });

            it('should use an empty string if neither fileObj nor image are present', function () {
                scope.image = undefined;
                scope.currPhoto.fileObj = undefined;

                scope.validateFileType();

                expect(fileValidationServiceMock.validateFileType).toHaveBeenCalledWith('', fileTypes);
            });

            it('should validate successfully if the file extension matches one of the valid types', function () {
                isfileTypeValid = true;
                
                fileExt = 'jpg';

                scope.validateFileType();

                fileExt = 'jpeg';

                scope.validateFileType();

                fileExt = 'png';

                scope.validateFileType();

                fileExt = 'gif';

                scope.validateFileType();

                var calls = scope.photoInputForm.$setValidity.calls;

                expect(calls.argsFor(0)).toEqual(['File-Type-Check', true]);
                expect(calls.argsFor(1)).toEqual(['File-Type-Check', true]);
                expect(calls.argsFor(2)).toEqual(['File-Type-Check', true]);
                expect(calls.argsFor(3)).toEqual(['File-Type-Check', true]);
                
                expect(scope.photoInputForm.$setValidity).not.toHaveBeenCalledWith('File-Type-Check', false);
            });

            it('should not validate successfully if the file extension does not match one of the valid types', function () {
                isfileTypeValid = false;
                fileExt = 'bmp';

                scope.validateFileType();

                expect(scope.photoInputForm.$setValidity).toHaveBeenCalledWith('File-Type-Check', false);
            });
        });

        describe("validate function", function () {
            beforeEach(function () {
                controller = controller('PhotoInputController', {$scope: scope});

                spyOn(scope, 'validateFileSize');
                spyOn(scope, 'validateFileType');

                scope.validate();
            });

            it('should call the validateFileSize function', function () {
                expect(scope.validateFileSize).toHaveBeenCalled();
            });

            it('should call the validateFileType function', function () {
                expect(scope.validateFileType).toHaveBeenCalled();
            });
        });

        describe("acceptPhoto function", function () {
            beforeEach(function () {
                scope.image = {
                    imageDescription: 'test'
                }

                controller = controller('PhotoInputController', {$scope: scope});

                spyOn(scope, 'validate');

                scope.photoInputForm = {
                    validationSummary: {
                        validate: function () {}
                    }
                };
                spyOn(scope.photoInputForm.validationSummary, 'validate').and.returnValue({
                    then: function (callback) {
                        callback();
                    }
                });

                scope.acceptPhotoCallback = function() {};
                spyOn(scope, 'acceptPhotoCallback');
            });

            it("should call the validate function", function () {
                scope.acceptPhoto();

                expect(scope.validate).toHaveBeenCalled();
            });

            it("should call the acceptPhotoCallback function when validation is successful", function () {
                scope.acceptPhoto();

                expect(scope.acceptPhotoCallback).toHaveBeenCalled();
            });

            it("should not call the acceptPhotoCallback function when validation is unsuccessful", function () {
                scope.photoInputForm.validationSummary.validate.and.returnValue({
                    then: function (callback) {
                        //empty success block
                    }
                });

                scope.acceptPhoto();

                expect(scope.acceptPhotoCallback).not.toHaveBeenCalled();
            });
        });

        describe("anotherPhoto function", function () {
            beforeEach(function () {
                scope.anotherPhotoCallback = function () {};
                spyOn(scope, 'anotherPhotoCallback');

                scope.image = {
                    imageDescription: 'test'
                }

                controller = controller('PhotoInputController', {$scope: scope});

                spyOn(scope, 'validate');

                scope.photoInputForm = {
                    validationSummary: {
                        validate: function () {}
                    }
                };
                spyOn(scope.photoInputForm.validationSummary, 'validate').and.returnValue({
                    then: function (callback) {
                        callback();
                    }
                });

            });

            it("should call the validate function", function () {
                scope.anotherPhoto();

                expect(scope.validate).toHaveBeenCalled();
            });

            it("should call the anotherPhotoCallback function when validation is successful", function () {
                scope.anotherPhoto();

                expect(scope.anotherPhotoCallback).toHaveBeenCalled();
            });

            it("should call the nextPage function on the mediaRequestNavigationService, but not save the evaluation when validation is successful", function () {
                scope.anotherPhoto();

                expect(mediaRequestNavigationServiceMock.nextPage).toHaveBeenCalledWith(false);
            });

            it("should not call the anotherPhotoCallback function when validation is unsuccessful", function () {
                scope.photoInputForm.validationSummary.validate.and.returnValue({
                    then: function (callback) {
                        //empty success block
                    }
                });

                scope.acceptPhoto();

                expect(scope.anotherPhotoCallback).not.toHaveBeenCalled();
            });

            it("should not call the nextPage function when validation is unsuccessful", function () {
                scope.photoInputForm.validationSummary.validate.and.returnValue({
                    then: function (callback) {
                        //empty success block
                    }
                });

                scope.acceptPhoto();

                expect(mediaRequestNavigationServiceMock.nextPage).not.toHaveBeenCalled();
            });
        });

        describe("skipSection function", function () {
            it("should call the skipSectionCallback function", function () {
                scope.skipSectionCallback = function() {};
                spyOn(scope, 'skipSectionCallback');

                controller = controller('PhotoInputController', {$scope: scope});

                scope.skipSection();

                expect(scope.skipSectionCallback).toHaveBeenCalled();
            });
        });

        describe("$watch image.decription", function () {
            var oldVal,
                newVal;

            beforeEach(function () {
                oldVal = undefined;
                newVal = undefined;

                spyOn(scope, '$watch').and.callFake(function (variable, callback) {
                    callback(newVal, oldVal);
                });

                scope.image = {};
                scope.imageAccepted = true;
                scope.hideNext = false;
            });

            it("should leave imageAccepted and hideNext unchanged if there was no previous value for image.description", function () {
                newVal = "a description";

                controller = controller('PhotoInputController', {$scope: scope});

                expect(scope.imageAccepted).toBeTruthy();
                expect(scope.hideNext).toBeFalsy();
            });

            it("should leave imageAccepted and hideNext unchanged if the previous value for image.description matches the new value", function () {
                oldVal = "a description";
                newVal = "a description";

                controller = controller('PhotoInputController', {$scope: scope});

                expect(scope.imageAccepted).toBeTruthy();
                expect(scope.hideNext).toBeFalsy();
            });

            it("should update imageAccepted and hideNext if the value for image.description has changed", function () {
                oldVal = "a description";
                newVal = "a new description";

                controller = controller('PhotoInputController', {$scope: scope});

                expect(scope.imageAccepted).toBeFalsy();
                expect(scope.hideNext).toBeTruthy();
            });
        });

        describe("removeEmptyImages function", function () {
            beforeEach(function () {
                evaluation = {
                    images: [
                        {
                            id: 'has id',
                            imageSection: {
                                name: 'HEAD',
                                description: 'Head'
                            }
                        },
                        {
                            imageSection: {
                                name: 'OTHER',
                                description: 'Other'
                            }
                        },
                        {}
                    ]
                };

                controller = controller('PhotoInputController', {$scope: scope});

                scope.removeEmptyImages();
            });

            it("should remove images from the evaluation object that don't have an id", function () {
                expect(scope.evaluation.images).toEqual([
                    {
                        id: 'has id',
                        imageSection: {
                            name: 'HEAD',
                            description: 'Head'
                        }
                    }
                ]);
            });
        });

        describe("previous function", function () {
            it("should open Confirm Media Update modal", function () {
                controller = controller('PhotoInputController', {$scope: scope});
                spyOn(scope, 'removeEmptyImages');

                scope.imageSelected = true;
                scope.imageAccepted = false;

                scope.previous();

                expect(modalMock.open).toHaveBeenCalled();
            });

            it("should not open Confirm Media Update modal", function () {
                controller = controller('PhotoInputController', {$scope: scope});
                spyOn(scope, 'removeEmptyImages');

                scope.imageSelected = false;

                scope.previous();

                expect(modalMock.open).not.toHaveBeenCalled();
            });

            it("should remove empty images from the evaluation", function () {
                controller = controller('PhotoInputController', {$scope: scope});
                spyOn(scope, 'removeEmptyImages');

                scope.previous();

                expect(scope.removeEmptyImages).toHaveBeenCalled();
            });

            it("should call the previousPage function on the mediaRequestNavigationService", function () {
                controller = controller('PhotoInputController', {$scope: scope});
                spyOn(scope, 'removeEmptyImages');

                scope.previous();

                expect(mediaRequestNavigationServiceMock.previousPage).toHaveBeenCalled();
            });
        });

        describe("next function", function () {
            it("should call the nextPage function on the mediaRequestNavigationService", function () {
                controller = controller('PhotoInputController', {$scope: scope});
                scope.next();

                expect(mediaRequestNavigationServiceMock.nextPage).toHaveBeenCalled();
            });
        });

        describe("swipeLeft function", function () {
			it("should broadcast swipeLeft", function () {
                controller = controller('PhotoInputController', {$scope: scope});
                scope.swipeLeft();
				expect(rootScope.$broadcast).toHaveBeenCalledWith('swipeLeft');
			});
        });

        describe("swipeRight function", function () {
            beforeEach(function () {
                controller = controller('PhotoInputController', {$scope: scope});
            });

			it("should broadcast swipeRight if hideNext is falsy", function () {
                scope.hideNext = false;
                scope.swipeRight();
				expect(rootScope.$broadcast).toHaveBeenCalledWith('swipeRight');
            });
            
            it("should not broadcast swipeRight if hideNext is true", function () {
                scope.hideNext = true;
                scope.swipeRight();
				expect(rootScope.$broadcast).not.toHaveBeenCalled();
            });
        });
    });
});