define(['OptionalPhotoController'], function() {
    'use strict';

    describe("The Optional Photo Controller", function () {
        var controller,
            scope,
            timeout,
            imageResponseServiceMock,
            mediaRequestNavigationServiceMock,
			focusServiceMock,
			imageUploadServiceMock,
			image,
            stateParams = {
                imageNum: 1
            },
            configServiceValues = {
                imageFileSizeLimit: 3,
                suggestedVideoSize: 10,
                videoFileSizeLimit: 20
            },
            httpMock;

        beforeEach(function () {
            module('angularTemplateApp');

            imageResponseServiceMock = jasmine.createSpyObj('imageResponseService', ['getOptionalImage', 'createImagePlaceholder', 'getEvaluation','saveEvaluation']);
            mediaRequestNavigationServiceMock = jasmine.createSpyObj('mediaRequestNavigationService', ['skipToReviewPhotos']);
			focusServiceMock = jasmine.createSpyObj('focusService', ['focusElement']);
            imageUploadServiceMock = jasmine.createSpyObj('imageUploadService', ['uploadImage']);

			imageUploadServiceMock.uploadImage.and.returnValue({
				then: function (callback) {
					callback();
				}
			});

			imageResponseServiceMock.getOptionalImage.and.callFake(function (imageNum) {
                return imageNum == 0 ? image : null;
            });

            imageResponseServiceMock.getEvaluation.and.returnValue({});

			imageResponseServiceMock.saveEvaluation.and.returnValue({
				then: function (callback) {
					callback();
				}
            });

			module(function ($provide) {
                $provide.value('$stateParams', stateParams);
                $provide.value('configServiceValues', configServiceValues);
                $provide.value('$modal', {});
                $provide.value('imageResponseService', imageResponseServiceMock);
                $provide.value('mediaRequestNavigationService', mediaRequestNavigationServiceMock);
				$provide.value('focusService', focusServiceMock);
                $provide.value('imageUploadService', imageUploadServiceMock);
			});

            inject(function($controller, $rootScope, $timeout, $httpBackend) {
                scope = $rootScope.$new();
                controller = $controller;
                timeout = $timeout;
                httpMock = $httpBackend;
            });
        });

		describe("initial state", function () {
            beforeEach(function () {
                httpMock.expectGET('http://some-url/image-content-url').respond(200, 'image-content');
            });

            it("should correctly set initial values", function() {
                controller = controller('OptionalPhotoController', {$scope: scope});
                scope.$apply();

                expect(scope.picType).toEqual('Optional');
				expect(scope.isOptional).toEqual(true);
				expect(scope.imageNum).toEqual(1);
                expect(scope.imageAccepted).toEqual(false);
                expect(scope.additionalPicText).toEqual('Please take additional photos of the area of interest');
                expect(scope.imageTypeTemplate).toEqual('modules/applets/mytelederm/pages/photo-input/partials/_not-required-photo.html');
                expect(scope.showImageRequired).toEqual(false);
            });

            it("should call getOptionalImage to set the image value in the scope", function () {
                image = {
                    imageSection: {
                        name: 'OPTIONAL',
							description: 'Optional'
                    },
                    imageType: {
                        name: 'OPTIONAL',
							description: 'Optional'
                    },
                    imageClass: 'OPTIONAL'
                };

                controller = controller('OptionalPhotoController', {$scope: scope});
                scope.$apply();

                expect(imageResponseServiceMock.getOptionalImage).toHaveBeenCalledWith(0);
                expect(scope.image).toEqual(image);
            });

            it("should call getOptionalImage to get the next image", function () {
                image = {
                    imageSection: {
                        name: 'OPTIONAL',
						description: 'Optional'
                    },
                    imageType: {
                        name: 'OPTIONAL',
						description: 'Optional'
                    },
                    imageClass: 'OPTIONAL'
                };

                controller = controller('OptionalPhotoController', {$scope: scope});
                scope.$apply();

                expect(imageResponseServiceMock.getOptionalImage).toHaveBeenCalledWith(1);
                expect(scope.nextImage).toEqual(null);
            });

            it("should set imageSelected and imageAccepted to false if the image is undefined, hideNext is the opposite of imageAccepted", function () {
                image = undefined;

                controller = controller('OptionalPhotoController', {$scope: scope});
                scope.$apply();

                expect(scope.imageSelected).toBeFalsy();
                expect(scope.imageAccepted).toBeFalsy();
                expect(scope.hideNext).toBeTruthy();
            });

            it("should set imageSelected and imageAccepted to false if the image is defined but it doesn't have an id or imageUrl, hideNext is the opposite of imageAccepted", function () {
                image = {
                    imageSection: {
                        name: 'OPTIONAL',
						description: 'Optional'
                    },
                    imageType: {
                        name: 'OPTIONAL',
						description: 'Optional'
                    },
                    imageClass: 'OPTIONAL'
                };

                controller = controller('OptionalPhotoController', {$scope: scope});
                scope.$apply();

                expect(scope.imageSelected).toBeFalsy();
                expect(scope.imageAccepted).toBeFalsy();
                expect(scope.hideNext).toBeTruthy();
            });

            it("should set imageSelected and imageAccepted to false if the image is defined and has an id but not an imageUrl, hideNext is the opposite of imageAccepted", function () {
                image = {
                    id: 'someId',
                    imageSection: {
                        name: 'OPTIONAL',
						description: 'Optional'
                    },
                    imageType: {
                        name: 'OPTIONAL',
						description: 'Optional'
                    },
                    imageClass: 'OPTIONAL'
                };

                controller = controller('OptionalPhotoController', {$scope: scope});
                scope.$apply();

                expect(scope.imageSelected).toBeFalsy();
                expect(scope.imageAccepted).toBeFalsy();
                expect(scope.hideNext).toBeTruthy();
            });

            it("should set imageSelected to true if the image is defined and has an id and an imageUrl, hideNext is the opposite of imageAccepted", function () {
                image = {
                    id: 'someId',
                    imageUrl: 'http://some-url/image-content-url',
                    imageSection: {
                        name: 'OPTIONAL',
						description: 'Optional'
                    },
                    imageType: {
                        name: 'OPTIONAL',
						description: 'Optional'
                    },
                    imageClass: 'OPTIONAL'
                };

                controller = controller('OptionalPhotoController', {$scope: scope});
                scope.$apply();

                expect(scope.imageSelected).toBeTruthy();
                expect(scope.imageAccepted).toBeTruthy();
                expect(scope.hideNext).toBeFalsy();
            });
        });
		
        describe("acceptPhotoCallback function", function () {
            beforeEach(function () {
                httpMock.expectGET('http://some-url/image-content-url').respond(200, 'image-content');
                controller = controller('OptionalPhotoController', {$scope: scope});
                scope.$apply();

                scope.imageAccepted = false;
                scope.hideNext = true;
            });

			it("should call saveEvaluation if currPhoto.fileObj is not defined", function () {
				image =  null;
				scope.currPhoto = {
					fileObj: null
				};

				scope.acceptPhotoCallback();

				expect(imageResponseServiceMock.saveEvaluation).toHaveBeenCalled();
			});

            it("should set imageAccepted to true if currPhoto.fileobj is not defined", function () {
				image =  null;
				scope.currPhoto = {
					fileObj: null
                };
                
                scope.acceptPhotoCallback();

                expect(scope.imageAccepted).toBeTruthy();
            });

            it("should set hideNext to false if currPhoto.fileobj is not defined", function () {
				image =  null;
				scope.currPhoto = {
					fileObj: null
                };
                
                scope.acceptPhotoCallback();

                expect(scope.hideNext).toBeFalsy();
            });

            it("should not set focus if currPhoto.fileObj is undefined", function () {
				scope.currPhoto = {
					fileObj: null
                };
                
                scope.acceptPhotoCallback();

                expect(focusServiceMock.focusElement).not.toHaveBeenCalled();

                try {
                    timeout.flush();
                } catch (e) {
                    expect(focusServiceMock.focusElement).not.toHaveBeenCalled();
                }
            });

            it("should not call uploadImage if currPhoto.fileObj is undefined", function () {
				scope.currPhoto = {
					fileObj: null
                };
                
                scope.acceptPhotoCallback();

                expect(imageUploadServiceMock.uploadImage).not.toHaveBeenCalled();
            });

            it("should not call getOptionalImage if currPhoto.fileObj is undefined", function () {
                scope.currPhoto = {
					fileObj: null
                };
                
                imageResponseServiceMock.getOptionalImage.calls.reset();

                scope.acceptPhotoCallback();

                expect(imageResponseServiceMock.getOptionalImage).not.toHaveBeenCalled();
            });

			it("should call uploadImage with new data if currPhoto.fileObj is defined but image id is not", function () {
				image =  null;
				scope.image = { imageDescription: 'ImageDesc'};
				scope.currPhoto = {
					fileObj: {
						name: 'file.jpg'
					}
				};

				scope.acceptPhotoCallback();

				expect(imageUploadServiceMock.uploadImage).toHaveBeenCalledWith(scope.currPhoto.fileObj, {
					imageSection: {
						name: 'OPTIONAL',
						description: 'Optional'
					},
					imageType: {
						name: 'OPTIONAL',
						description: 'Optional'
					},
					imageClass: 'OPTIONAL',
					imageDescription: 'ImageDesc' ,
                    fileName: 'file.jpg',
                    imageNumber: 1
				});
			});

			it("should call uploadImage with existing image data if currPhoto.fileObj and image id are defined", function () {
				scope.image = {
					id: 'someId',
					imageUrl: 'http://some-url/image-url'
                };
                
				scope.currPhoto = {
					fileObj: {
						name: 'file.jpg'
					}
				};

				scope.acceptPhotoCallback();

				expect(imageUploadServiceMock.uploadImage).toHaveBeenCalledWith(scope.currPhoto.fileObj, {
					id: 'someId',
					imageUrl: 'http://some-url/image-url',
					fileName: 'file.jpg'
				});
			});

            it("should set imageAccepted to true if currPhoto.fileobj is defined", function () {
				scope.currPhoto = {
					fileObj: {
						name: 'file.jpg'
					}
				};
                
                scope.acceptPhotoCallback();

                expect(scope.imageAccepted).toBeTruthy();
            });

            it("should call getOptionalImage to set the image value in the scope if currPhoto.fileObj is defined", function () {
                image = {
                    imageSection: {
                        name: 'OPTIONAL',
						description: 'Optional'
                    },
                    imageType: {
                        name: 'OPTIONAL',
						description: 'Optional'
                    },
                    imageClass: 'Optional',
                    imageNumber: 0
                };

                scope.image = {
					id: 'someId',
                    imageUrl: 'http://some-url/image-url'
                };
                
				scope.currPhoto = {
					fileObj: {
						name: 'file.jpg'
					}
				};
                
                scope.acceptPhotoCallback();

                expect(imageResponseServiceMock.getOptionalImage).toHaveBeenCalledWith(0);
                expect(scope.image).toEqual(image);
            });

            it("should set hideNext to false if currPhoto.fileObj is defined", function () {
				scope.currPhoto = {
					fileObj: {
						name: 'file.jpg'
					}
				};
                
                scope.acceptPhotoCallback();

                expect(scope.hideNext).toBeFalsy();
            });

            it("should set focus on the photo-description field after a timeout if currPhoto.fileObj is defined", function () {
				scope.currPhoto = {
					fileObj: {
						name: 'file.jpg'
					}
                };
                
                scope.acceptPhotoCallback();

                expect(focusServiceMock.focusElement).not.toHaveBeenCalled();

                timeout.flush();

                expect(focusServiceMock.focusElement).toHaveBeenCalledWith('#photo-description');
            });

            it("should not call saveEvaluation if currPhoto.fileObj is defined", function () {
				scope.currPhoto = {
					fileObj: {
						name: 'file.jpg'
					}
                };
                
                scope.acceptPhotoCallback();

                expect(imageResponseServiceMock.saveEvaluation).not.toHaveBeenCalled();
            });
        });

        describe("anotherPhotoCallback function", function () {
            beforeEach(function () {
                httpMock.expectGET('http://some-url/image-content-url').respond(200, 'image-content');
                controller = controller('OptionalPhotoController', {$scope: scope});
                scope.$apply();
            });

            it("should create an placeholder optional image if nextImage is undefined", function () {
                scope.nextImage = undefined;

                scope.anotherPhotoCallback();

                expect(imageResponseServiceMock.createImagePlaceholder).toHaveBeenCalledWith({
                    imageSection: {
                        name: 'OPTIONAL',
						description: 'Optional'
                    },
                    imageType: {
                        name: 'OPTIONAL',
						description: 'Optional'
                    },
                    imageClass: 'OPTIONAL',
                    imageNumber: 2
                });
            });

            it("should create an placeholder optional image if nextImage is defined, but has no id", function () {
                scope.nextImage = {};

                scope.anotherPhotoCallback();

                expect(imageResponseServiceMock.createImagePlaceholder).toHaveBeenCalledWith({
                    imageSection: {
                        name: 'OPTIONAL',
						description: 'Optional'
                    },
                    imageType: {
                        name: 'OPTIONAL',
						description: 'Optional'
                    },
                    imageClass: 'OPTIONAL',
                    imageNumber: 2
                });
            });

            it("should not create a placeholder optional image if nextImage is defined and has an id", function () {
                scope.nextImage = {
                    id: 'someId'
                };

                scope.anotherPhotoCallback();

                expect(imageResponseServiceMock.createImagePlaceholder).not.toHaveBeenCalled();
            });
        });

        describe("skipSectionCallback function", function () {
            beforeEach(function () {
                httpMock.expectGET('http://some-url/image-content-url').respond(200, 'image-content');
                controller = controller('OptionalPhotoController', {$scope: scope});
                scope.$apply();

                scope.skipSectionCallback();
            });

            it("should skip to the next image section", function () {
                expect(mediaRequestNavigationServiceMock.skipToReviewPhotos).toHaveBeenCalled();
            });
        });

        it("should extend the PhotoInputController", function () {
            controller = controller('OptionalPhotoController', {$scope: scope});
            spyOn(scope, 'validate');

            scope.photoInputForm = {
                validationSummary: {
                    validate: function () {}
                }
            };
            spyOn(scope.photoInputForm.validationSummary, 'validate').and.returnValue({
                then: function (callback) {
                    callback();
                }
            });
            
            spyOn(scope, 'acceptPhotoCallback');

            scope.acceptPhoto();

            expect(scope.validate).toHaveBeenCalled();
            expect(scope.acceptPhotoCallback).toHaveBeenCalled();
        });
    });
});