define(['CloseUpPhotoController'], function() {
    'use strict';

    describe("The Close-Up Photo Controller", function () {
        var controller,
            scope,
            timeout,
            imageResponseServiceMock,
            mediaRequestNavigationServiceMock,
            focusServiceMock,
			imageUploadServiceMock,
			image,
            stateParams,
            configServiceValues,
            httpMock;

        beforeEach(function () {
            module('angularTemplateApp');

            imageResponseServiceMock = jasmine.createSpyObj('imageResponseService', ['getCloseUpImage', 'getEvaluation', 'createImagePlaceholder', 'saveEvaluation']);
            mediaRequestNavigationServiceMock = jasmine.createSpyObj('mediaRequestNavigationService', ['skipToNextSection']);
            focusServiceMock = jasmine.createSpyObj('focusService', ['focusElement']);
            imageUploadServiceMock = jasmine.createSpyObj('imageUploadService', ['uploadImage']);
            
            imageUploadServiceMock.uploadImage.and.returnValue({
				then: function (callback) {
					callback();
				}
			});

            imageResponseServiceMock.getCloseUpImage.and.callFake(function (sectionName, imageNum) {
                return imageNum == 0 ? image : null;
            });

            imageResponseServiceMock.getEvaluation.and.returnValue({});

            imageResponseServiceMock.saveEvaluation.and.returnValue({
				then: function (callback) {
					callback();
				}
            });
            
            stateParams = {
                imageNum: 1,
                imageSection: {
                    name: 'HEAD',
                    description: 'Head'
                }
            };

            configServiceValues = {
                imageFileSizeLimit: 3,
                suggestedVideoSize: 10,
                videoFileSizeLimit: 20
            };

            module(function ($provide) {
                $provide.value('$stateParams', stateParams);
                $provide.value('configServiceValues', configServiceValues);
                $provide.value('$modal', {});
                $provide.value('imageResponseService', imageResponseServiceMock);
                $provide.value('mediaRequestNavigationService', mediaRequestNavigationServiceMock);
                $provide.value('focusService', focusServiceMock);
                $provide.value('imageUploadService', imageUploadServiceMock);
            });

            inject(function($controller, $rootScope, $timeout, $httpBackend) {
                scope = $rootScope.$new();
                controller = $controller;
                timeout = $timeout;
                httpMock = $httpBackend;
            });
        });

		describe("initial state", function () {
            beforeEach(function () {
                httpMock.expectGET('http://some-url/image-content-url').respond(200, 'image-content');
            });

            it("should correctly set initial values", function() {
                controller = controller('CloseUpPhotoController', {$scope: scope});
                scope.$apply();

                expect(scope.picType).toEqual('Close up');
                expect(scope.isCloseup).toEqual(true);
                expect(scope.additionalPicText).toEqual('Please take a close up of the area of interest');
                expect(scope.imageSection).toEqual({
                    name: 'HEAD',
                    description: 'Head'
                });
                expect(scope.imageNum).toEqual(1);
                expect(scope.imageAccepted).toEqual(false);
                expect(scope.imageTypeTemplate).toEqual('modules/applets/mytelederm/pages/photo-input/partials/_not-required-photo.html');
            });

            it("should call getCloseUpImage to set the image value in the scope", function () {
                image = {
                    imageSection: {
                        name: 'HEAD',
                        description: 'Head'
                    },
                    imageType: {
                        name: 'CLOSE_UP',
                        description: 'Close up'
                    },
                    imageClass: 'CLOSE_UP'
                };

                controller = controller('CloseUpPhotoController', {$scope: scope});
                scope.$apply();

                expect(imageResponseServiceMock.getCloseUpImage).toHaveBeenCalledWith('HEAD', 0);
                expect(scope.image).toEqual(image);
            });

            it("should call getCloseUpImage to get the next image", function () {
                image = {
                    imageSection: {
                        name: 'HEAD',
                        description: 'Head'
                    },
                    imageType: {
                        name: 'CLOSE_UP',
                        description: 'Close up'
                    },
                    imageClass: 'CLOSE_UP'
                };

                controller = controller('CloseUpPhotoController', {$scope: scope});
                scope.$apply();

                expect(imageResponseServiceMock.getCloseUpImage).toHaveBeenCalledWith('HEAD', 1);
                expect(scope.nextImage).toEqual(null);
            });

            it("should set imageSelected and imageAccepted to false if the image is undefined, hideNext is the opposite of imageAccepted", function () {
                image = undefined;

                controller = controller('CloseUpPhotoController', {$scope: scope});
                scope.$apply();

                expect(scope.imageSelected).toBeFalsy();
                expect(scope.imageAccepted).toBeFalsy();
                expect(scope.hideNext).toBeTruthy();
            });

            it("should set imageSelected and imageAccepted to false if the image is defined but it doesn't have an id or imageUrl, hideNext is the opposite of imageAccepted", function () {
                image = {
                    imageSection: {
                        name: 'HEAD',
                        description: 'Head'
                    },
                    imageType: {
                        name: 'CLOSE_UP',
                        description: 'Close up'
                    },
                    imageClass: 'CLOSE_UP'
                };

                controller = controller('CloseUpPhotoController', {$scope: scope});
                scope.$apply();

                expect(scope.imageSelected).toBeFalsy();
                expect(scope.imageAccepted).toBeFalsy();
                expect(scope.hideNext).toBeTruthy();
            });

            it("should set imageSelected and imageAccepted to false if the image is defined and has an id but not an imageUrl, hideNext is the opposite of imageAccepted", function () {
                image = {
                    id: 'someId',
                    imageSection: {
                        name: 'HEAD',
                        description: 'Head'
                    },
                    imageType: {
                        name: 'CLOSE_UP',
                        description: 'Close up'
                    },
                    imageClass: 'CLOSE_UP'
                };

                controller = controller('CloseUpPhotoController', {$scope: scope});
                scope.$apply();

                expect(scope.imageSelected).toBeFalsy();
                expect(scope.imageAccepted).toBeFalsy();
                expect(scope.hideNext).toBeTruthy();
            });

            it("should set imageSelected to true if the image is defined and has an id and an imageUrl, hideNext is the opposite of imageAccepted", function () {
                image = {
                    id: 'someId',
                    imageUrl: 'http://some-url/image-content-url',
                    imageSection: {
                        name: 'HEAD',
                        description: 'Head'
                    },
                    imageType: {
                        name: 'CLOSE_UP',
                        description: 'Close up'
                    },
                    imageClass: 'CLOSE_UP'
                };

                controller = controller('CloseUpPhotoController', {$scope: scope});
                scope.$apply();

                expect(scope.imageSelected).toBeTruthy();
                expect(scope.imageAccepted).toBeTruthy();
                expect(scope.hideNext).toBeFalsy();
            });

            it("should set showImageRequired to true if the imageNum is 1", function () {
                stateParams.imageNum = 1;

                controller = controller('CloseUpPhotoController', {$scope: scope});
                scope.$apply();

                expect(scope.showImageRequired).toBeTruthy();
            });

            it("should set showImageRequired to false if the imageNum is not 1", function () {
                stateParams.imageNum = 2;

                controller = controller('CloseUpPhotoController', {$scope: scope});
                scope.$apply();

                expect(scope.showImageRequired).toBeFalsy();
            });
        });

        describe("acceptPhotoCallback function", function () {
            beforeEach(function () {
                httpMock.expectGET('http://some-url/image-content-url').respond(200, 'image-content');
                controller = controller('CloseUpPhotoController', {$scope: scope});
                scope.$apply();

                scope.imageAccepted = false;
                scope.hideNext = true;
            });

            it("should call saveEvaluation if currPhoto.fileObj is not defined", function () {
				image =  null;
				scope.currPhoto = {
					fileObj: null
				};

				scope.acceptPhotoCallback();

				expect(imageResponseServiceMock.saveEvaluation).toHaveBeenCalled();
            });
            
            it("should set imageAccepted to true if currPhoto.fileobj is not defined", function () {
				image =  null;
				scope.currPhoto = {
					fileObj: null
                };
                
                scope.acceptPhotoCallback();

                expect(scope.imageAccepted).toBeTruthy();
            });

            it("should set hideNext to false if currPhoto.fileobj is not defined", function () {
				image =  null;
				scope.currPhoto = {
					fileObj: null
                };
                
                scope.acceptPhotoCallback();

                expect(scope.hideNext).toBeFalsy();
            });

            it("should not set focus if currPhoto.fileObj is undefined", function () {
				scope.currPhoto = {
					fileObj: null
                };
                
                scope.acceptPhotoCallback();

                expect(focusServiceMock.focusElement).not.toHaveBeenCalled();

                try {
                    timeout.flush();
                } catch (e) {
                    expect(focusServiceMock.focusElement).not.toHaveBeenCalled();
                }
            });

            it("should not call uploadImage if currPhoto.fileObj is undefined", function () {
				scope.currPhoto = {
					fileObj: null
                };
                
                scope.acceptPhotoCallback();

                expect(imageUploadServiceMock.uploadImage).not.toHaveBeenCalled();
            });

            it("should not call getCloseUpImage if currPhoto.fileObj is undefined", function () {
                scope.currPhoto = {
					fileObj: null
                };
                
                imageResponseServiceMock.getCloseUpImage.calls.reset();

                scope.acceptPhotoCallback();

                expect(imageResponseServiceMock.getCloseUpImage).not.toHaveBeenCalled();
            });

            it("should call uploadImage with new data if currPhoto.fileObj is defined but image id is not", function () {
				image =  null;
				scope.image = { imageDescription: 'ImageDesc'};
				scope.currPhoto = {
					fileObj: {
						name: 'file.jpg'
					}
				};

				scope.acceptPhotoCallback();

				expect(imageUploadServiceMock.uploadImage).toHaveBeenCalledWith(scope.currPhoto.fileObj, {
					imageSection: {
                        name: 'HEAD',
                        description: 'Head'
                    },
                    imageType: {
                        name: 'CLOSE_UP',
                        description: 'Close up'
                    },
                    imageClass: 'CLOSE_UP',
					imageDescription: 'ImageDesc' ,
                    fileName: 'file.jpg',
                    imageNumber: 1
				});
            });
            
            it("should call uploadImage with existing image data if currPhoto.fileObj and image id are defined", function () {
				scope.image = {
					id: 'someId',
                    imageUrl: 'http://some-url/image-url'
                };
                
				scope.currPhoto = {
					fileObj: {
						name: 'file.jpg'
					}
				};

				scope.acceptPhotoCallback();

				expect(imageUploadServiceMock.uploadImage).toHaveBeenCalledWith(scope.currPhoto.fileObj, {
					id: 'someId',
					imageUrl: 'http://some-url/image-url',
					fileName: 'file.jpg'
				});
            });

            it("should call getCloseUpImage to set the image value in the scope if currPhoto.fileObj is defined", function () {
                image = {
                    imageSection: {
                        name: 'HEAD',
                        description: 'Head'
                    },
                    imageType: {
                        name: 'CLOSE_UP',
                        description: 'Close up'
                    },
                    imageClass: 'CLOSE_UP'
                };

                scope.image = {
					id: 'someId',
                    imageUrl: 'http://some-url/image-url'
                };
                
				scope.currPhoto = {
					fileObj: {
						name: 'file.jpg'
					}
				};
                
                scope.acceptPhotoCallback();

                expect(imageResponseServiceMock.getCloseUpImage).toHaveBeenCalledWith('HEAD', 0);
                expect(scope.image).toEqual(image);
            });
            
            it("should set imageAccepted to true if currPhoto.fileobj is defined", function () {
				scope.currPhoto = {
					fileObj: {
						name: 'file.jpg'
					}
				};
                
                scope.acceptPhotoCallback();

                expect(scope.imageAccepted).toBeTruthy();
            });

            it("should set hideNext to false if currPhoto.fileObj is defined", function () {
				scope.currPhoto = {
					fileObj: {
						name: 'file.jpg'
					}
				};
                
                scope.acceptPhotoCallback();

                expect(scope.hideNext).toBeFalsy();
            });

            it("should set focus on the photo-description field after a timeout if currPhoto.fileObj is defined", function () {
				scope.currPhoto = {
					fileObj: {
						name: 'file.jpg'
					}
                };
                
                scope.acceptPhotoCallback();

                expect(focusServiceMock.focusElement).not.toHaveBeenCalled();

                timeout.flush();

                expect(focusServiceMock.focusElement).toHaveBeenCalledWith('#photo-description');
            });

            it("should not call saveEvaluation if currPhoto.fileObj is defined", function () {
				scope.currPhoto = {
					fileObj: {
						name: 'file.jpg'
					}
                };
                
                scope.acceptPhotoCallback();

                expect(imageResponseServiceMock.saveEvaluation).not.toHaveBeenCalled();
            });
        });

        describe("anotherPhotoCallback function", function () {
            beforeEach(function () {
                httpMock.expectGET('http://some-url/image-content-url').respond(200, 'image-content');
                controller = controller('CloseUpPhotoController', {$scope: scope});
                scope.$apply();
            });

            it("should create an placeholder close-up image if nextImage is undefined", function () {
                scope.nextImage = undefined;

                scope.anotherPhotoCallback();

                expect(imageResponseServiceMock.createImagePlaceholder).toHaveBeenCalledWith({
                    imageSection: {
                        name: 'HEAD',
                        description: 'Head'
                    },
                    imageType: {
                        name: 'CLOSE_UP',
                        description: 'Close up'
                    },
                    imageClass: 'CLOSE_UP',
                    imageNumber: 2
                });
            });

            it("should create an placeholder close-up image if nextImage is defined, but has no id", function () {
                scope.nextImage = {};

                scope.anotherPhotoCallback();

                expect(imageResponseServiceMock.createImagePlaceholder).toHaveBeenCalledWith({
                    imageSection: {
                        name: 'HEAD',
                        description: 'Head'
                    },
                    imageType: {
                        name: 'CLOSE_UP',
                        description: 'Close up'
                    },
                    imageClass: 'CLOSE_UP',
                    imageNumber: 2
                });
            });

            it("should not create a placeholder close-up image if nextImage is defined and has an id", function () {
                scope.nextImage = {
                    id: 'someId'
                };

                scope.anotherPhotoCallback();

                expect(imageResponseServiceMock.createImagePlaceholder).not.toHaveBeenCalled();
            });
        });

        describe("skipSectionCallback function", function () {
            beforeEach(function () {
                httpMock.expectGET('http://some-url/image-content-url').respond(200, 'image-content');
                controller = controller('CloseUpPhotoController', {$scope: scope});
                scope.$apply();

                scope.skipSectionCallback();
            });

            it("should skip to the next image section", function () {
                expect(mediaRequestNavigationServiceMock.skipToNextSection).toHaveBeenCalled();
            });
        });

        it("should extend the PhotoInputController", function () {
            controller = controller('CloseUpPhotoController', {$scope: scope});
            spyOn(scope, 'validate');

            scope.photoInputForm = {
                validationSummary: {
                    validate: function () {}
                }
            };
            spyOn(scope.photoInputForm.validationSummary, 'validate').and.returnValue({
                then: function (callback) {
                    callback();
                }
            });
            
            spyOn(scope, 'acceptPhotoCallback');

            scope.acceptPhoto();

            expect(scope.validate).toHaveBeenCalled();
            expect(scope.acceptPhotoCallback).toHaveBeenCalled();
        });
    });
});