define(['MedicationsEntryController'], function() {
    'use strict';

    describe("The Medications Entry Controller", function () {
        var controller,
            scope,
            formatterObj,
            imageResponseServiceMock,
            mediaRequestNavigationServiceMock,
            focusServiceMock,
            medsByType = [],
            rootScope,
            timeout,
            medicationsEntryServiceMock,
            meds = [];

        beforeEach(function () {
            module('angularTemplateApp');

            imageResponseServiceMock = jasmine.createSpyObj('imageResponseService', ['getMedsByType', 'updateMedsByType']);
            mediaRequestNavigationServiceMock = jasmine.createSpyObj('mediaRequestNavigationService', ['previousPage', 'nextPage']);
            focusServiceMock = jasmine.createSpyObj('focusService', ['focusElement']);
            medicationsEntryServiceMock = jasmine.createSpyObj('medicationsEntryService', ['getMedications']);

            imageResponseServiceMock.getMedsByType.and.callFake(function () {
                return medsByType;
            });

            medicationsEntryServiceMock.getMedications.and.returnValue({
                then: function (callback) {
                    callback(
                        meds
                    );
                }
            });

            module(function ($provide) {
                $provide.value('$stateParams', {medType: 'TOPICAL'});
                $provide.value('imageResponseService', imageResponseServiceMock);
                $provide.value('mediaRequestNavigationService', mediaRequestNavigationServiceMock);
                $provide.value('focusService', focusServiceMock);
                $provide.value('medicationsEntryService', medicationsEntryServiceMock);
            });

            inject(function($controller, $rootScope, formatter, $timeout) {
                scope = $rootScope.$new();
                rootScope = $rootScope;
                controller = $controller;
                formatterObj = formatter;
                timeout = $timeout;
                spyOn(rootScope, '$broadcast');

                scope.medicationEntryForm = {
					validationSummary: {
						validate: function () {}
					}
                };
                
                spyOn(scope.medicationEntryForm.validationSummary, 'validate').and.returnValue({
					then: function (callback) {
						callback();
					}
				});
            });
        });

		describe("initial state", function () {
            it("should correctly set initial values", function() {
                controller = controller('MedicationsEntryController', {$scope: scope});
                scope.$apply();
                expect(scope.medType).toEqual('TOPICAL');
                expect(scope.freqOptions).toEqual(['a day', 'every other day', 'a week', 'every other week', 'a month', 'other']);
                expect(scope.medFrequencyItems).toEqual(["1 time", "2 times", "3 times", "4 times", "5 times", "6 times", "other"]);
                expect(scope.dosesMissedFrequencyItems).toEqual(["None", "1 time", "2 times", "3 times",  "4 times", "5 or more times"]);
				expect(scope.toTitleCase).toEqual(formatterObj.toTitleCase);
                expect(scope.medLabelValues).toEqual({
                    'TOPICAL': {
                        applicationMethod: 'apply to',
                        frequency: 'use',
                        doseHidden: 'Location',
                        dose: 'Location: Each time you use this medication, where do you use it?  (e.g., face, all over body)',
                        dosesMissed: 'week'
                    },
                    'ORAL': {
                        applicationMethod: 'take for',
                        frequency: 'take',
                        doseHidden: 'Dose',
                        dose: 'Dose: Each time you take this medication, how much do you take?',
                        dosesMissed: 'week'
                    },
                    'INJECTED': {
                        applicationMethod: 'inject for',
                        frequency: 'inject',
                        doseHidden: 'Dose',
                        dose: 'Dose: Each time you take this medication, how much do you currently inject?  (e.g., 2 syringes, 1 pen)',
                        dosesMissed: 'month'
                    }
                });
            });

            it("should have a default value for the medications list", function () {
                controller = controller('MedicationsEntryController', {$scope: scope});
                scope.$apply();
                expect(scope.medications).toEqual([
                    {
                        "type": "TOPICAL",
                        "name": "",
						"frequency": "",
						"frequencyOther": "",
						"timeUnit": "",
						"timeUnitOther": "",
						"dose": "",
						"missedDoses": ""
                    }
                ]);
            });

            it("should retrieve the medication list from the imageResponseService", function () {
                medsByType = [
                    {
                        "type": "TOPICAL",
                        "name": "Eucerin",
						"frequency": "3 times",
						"frequencyOther": "",
						"timeUnit": "a day",
						"timeUnitOther": "",
						"dose": "",
                        "missedDoses": "5"
                    }
                ];

                controller = controller('MedicationsEntryController', {$scope: scope});
                scope.$apply();
                expect(scope.medications).toEqual([
                    {
                        "type": "TOPICAL",
                        "name": "Eucerin",
						"frequency": "3 times",
						"frequencyOther": "",
						"timeUnit": "a day",
						"timeUnitOther": "",
						"dose": "",
                        "missedDoses": "5"
                    }
                ]);
            });
        });

        describe("addNew function", function () {
            it("should add a new medication to the medication object", function () {
                medsByType = [
                    {
                        "type": "TOPICAL",
                        "name": "Eucerin",
						"frequency": "3 times",
						"frequencyOther": "",
						"timeUnit": "a day",
						"timeUnitOther": "",
						"dose": "",
                        "missedDoses": "5"
                    }
                ];

                controller = controller('MedicationsEntryController', {$scope: scope});
                scope.$apply();
                scope.addNew();
                expect(scope.medications).toEqual([
                    {
                        "type": "TOPICAL",
                        "name": "Eucerin",
						"frequency": "3 times",
						"frequencyOther": "",
						"timeUnit": "a day",
						"timeUnitOther": "",
						"dose": "",
                        "missedDoses": "5"
                    },
                    {
                        "type": "TOPICAL",
                        "name": "",
						"frequency": "",
						"frequencyOther": "",
						"timeUnit": "",
						"timeUnitOther": "",
						"dose": "",
                        "missedDoses": ""
                    }
                ]);
            });

            it("should set focus to the last medication section header", function () {
                controller = controller('MedicationsEntryController', {$scope: scope});
                scope.addNew();
                timeout.flush();
                expect(focusServiceMock.focusElement).toHaveBeenCalledWith(angular.element('.medication-header:last'));
            });
        });

        describe("removeMedication function", function () {
            it("should remove a medication from the medication object", function () {
                medsByType = [
                    {
                        "type": "TOPICAL",
                        "name": "Eucerin",
                        "dose": "3",
                        "missedDoses": "5",
						"frequency": '1 time',
						"timeUnit": "a day"
                    },
                    {
                        "type": "TOPICAL",
                        "name": "",
                        "dose": "",
                        "missedDoses": "",
						"frequency": '1 time',
						"timeUnit": ""
                    }
                ];

                controller = controller('MedicationsEntryController', {$scope: scope});
                scope.$apply();
                scope.removeMedication(1);
                expect(scope.medications).toEqual([
                     {
                        "type": "TOPICAL",
                        "name": "Eucerin",
                        "dose": "3",
                        "missedDoses": "5",
						"frequency": '1 time',
						"timeUnit": "a day"
                    }
                ]);
            });

            it("should remove a medication from the medication object", function () {
                medsByType = [
                    {
                        "type": "TOPICAL",
                        "name": "Eucerin",
                        "dose": "3",
                        "missedDoses": "5",
						"frequency": '1 time',
						"timeUnit": "a day"
                    },
                    {
                        "type": "TOPICAL",
                        "name": "",
                        "dose": "",
                        "missedDoses": "",
						"frequency": '1 time',
						"timeUnit": ""
                    }
                ];

                controller = controller('MedicationsEntryController', {$scope: scope});
                scope.$apply();
                scope.removeMedication(1);
                expect(scope.medications).toEqual([
                    {
                        "type": "TOPICAL",
                        "name": "Eucerin",
                        "dose": "3",
                        "missedDoses": "5",
						"frequency": '1 time',
						"timeUnit": "a day"
                    }
                ]);
            });

            it("should set focus to the last medication section header", function () {
                controller = controller('MedicationsEntryController', {$scope: scope});
                scope.removeMedication(1);
                timeout.flush();
                expect(focusServiceMock.focusElement).toHaveBeenCalledWith(angular.element('.medication-header:last'));
            });
        });

        describe("medLabelValues", function () {
            it("should set the correct medication application method", function () {
                controller = controller('MedicationsEntryController', {$scope: scope});
                scope.$apply();
                expect(scope.medLabelValues[scope.medType].applicationMethod).toEqual("apply to");
            });
        });

        describe("items function", function () {
            it("should retrieve list of meds from medicationsEntryService.getMedications", function () {
                controller = controller('MedicationsEntryController', {$scope: scope});
                scope.$apply();

                scope.items('ben');

                expect(medicationsEntryServiceMock.getMedications).toHaveBeenCalledWith('TOPICAL', 'ben', 20);
            });

            it("should set search term to lower case to retrieve list of meds from medicationsEntryService.getMedications", function () {
                controller = controller('MedicationsEntryController', {$scope: scope});
                scope.$apply();

                scope.items('BEN');

                expect(medicationsEntryServiceMock.getMedications).toHaveBeenCalledWith('TOPICAL', 'ben', 20);
            });
        });

        describe("selectMedication function", function () {
            it("should set medications[index].name equal to selected autocomplete medication", function () {
                medsByType = [
                    {
                        "type": "TOPICAL",
                        "name": "medName1",
                        "dose": "3",
                        "missedDoses": "5",
						"frequency": '1 time',
						"timeUnit": "a day"
                    },
                    {
                        "type": "TOPICAL",
                        "name": "tempName",
                        "dose": "",
                        "missedDoses": "",
						"frequency": '1 time',
						"timeUnit": ""
                    },
                    {
                        "type": "TOPICAL",
                        "name": "medName3",
                        "dose": "",
                        "missedDoses": null,
						"frequency": '1 time',
						"timeUnit": ""
                    }
                ];

                controller = controller('MedicationsEntryController', {$scope: scope});
                scope.$apply();

                scope.selectMedication('medName2', 1);

                expect(scope.medications[1].name).toEqual("medName2");
            });
        });
		
        describe("previous function", function () {
            beforeEach(function () {
                controller = controller('MedicationsEntryController', {$scope: scope});
                scope.$apply();

                scope.medications = [
                    {
                        "type": "TOPICAL",
                        "name": "Eucerin",
                        "dose": "3",
                        "missedDoses": "5",
						"frequency": '1 time',
						"timeUnit": "a day"
                    }
                ];
            });

            it("should update the medications for the given type", function () {
                scope.previous();
                expect(imageResponseServiceMock.updateMedsByType).toHaveBeenCalledWith([
                    {
                        "type": "TOPICAL",
                        "name": "Eucerin",
                        "dose": "3",
                        "missedDoses": "5",
                        "frequency": '1 time',
                        "frequencyOther": '',
                        "timeUnit": "a day",
                        "timeUnitOther": ""
                    }
                ], 'TOPICAL');
            });

            it("should call the previousPage function on the mediaRequestNavigationService if validation passes", function () {
                scope.previous();
                expect(mediaRequestNavigationServiceMock.previousPage).toHaveBeenCalled();
            });

            it("should not call the previousPage function on the mediaRequestNavigationService if validation fails", function () {
                scope.medicationEntryForm.validationSummary.validate.and.returnValue({
					then: function (callback) {
						//empty success block
					}
				});
                scope.previous();
                expect(mediaRequestNavigationServiceMock.previousPage).not.toHaveBeenCalled();
            });
        });
        
        describe("next function", function () {
            beforeEach(function () {
                controller = controller('MedicationsEntryController', {$scope: scope});
                scope.$apply();

                scope.medications = [
                    {
                        "type": "TOPICAL",
                        "name": "Eucerin",
                        "dose": "3",
                        "missedDoses": "5",
						"frequency": '1 time',
						"timeUnit": "a day"
                    }
                ];
            });

            it("should update the medications for the given type", function () {
                scope.next();
                expect(imageResponseServiceMock.updateMedsByType).toHaveBeenCalledWith([
                    {
                        "type": "TOPICAL",
                        "name": "Eucerin",
                        "dose": "3",
                        "missedDoses": "5",
                        "frequency": '1 time',
                        "frequencyOther": '',
                        "timeUnit": "a day",
                        "timeUnitOther": ""
                    }
                ], 'TOPICAL');
            });

            it("should call the nextPage function on the mediaRequestNavigationService if validation passes", function () {
                scope.next();
                expect(mediaRequestNavigationServiceMock.nextPage).toHaveBeenCalled();
            });

            it("should not call the nextPage function on the mediaRequestNavigationService if validation fails", function () {
                scope.medicationEntryForm.validationSummary.validate.and.returnValue({
					then: function (callback) {
						//empty success block
					}
				});
                scope.next();
                expect(mediaRequestNavigationServiceMock.nextPage).not.toHaveBeenCalled();
            });
        });

        describe("swipeLeft function", function () {
			it("should broadcast swipeLeft", function () {
                controller = controller('MedicationsEntryController', {$scope: scope});
                scope.swipeLeft();
				expect(rootScope.$broadcast).toHaveBeenCalledWith('swipeLeft');
			});
        });
        
        describe("swipeRight function", function () {
			it("should broadcast swipeRight", function () {
                controller = controller('MedicationsEntryController', {$scope: scope});
                scope.swipeRight();
				expect(rootScope.$broadcast).toHaveBeenCalledWith('swipeRight');
			});
        });
    });
});