define(['MedicationsController'], function() {
    'use strict';

    describe("The Medications Controller", function () {
        var controller,
            scope,
            formatterObj,
            mediaRequestNavigationServiceMock,
			focusServiceMock,
            timeout,
            rootScope,
            mediaRequest = {},
            evaluation = {},
            medTypes = ['TOPICAL', 'ORAL', 'INJECTED'],
            medExamples = {
                'TOPICAL': 'lotion',
                'ORAL': 'pill, liquid',
                'INJECTED': 'syringe' 
            },
            imageResponseServiceMock;

        beforeEach(function () {
            module('angularTemplateApp');

            mediaRequestNavigationServiceMock = jasmine.createSpyObj('mediaRequestNavigationService', ['previousPage', 'nextPage']);
            focusServiceMock = jasmine.createSpyObj('focusService', ['focusElement']);
            imageResponseServiceMock = jasmine.createSpyObj('imageResponseService', ['getEvaluation']);

            imageResponseServiceMock.getEvaluation.and.callFake(function () {
                return evaluation;
            });

            module(function ($provide) {
				$provide.value('mediaRequestNavigationService', mediaRequestNavigationServiceMock);
                $provide.value('medicationTypes', medTypes);
                $provide.value('medicationExamples', medExamples);
                $provide.value('focusService', focusServiceMock);
                $provide.value('imageResponseService', imageResponseServiceMock);
			});

            inject(function($controller, $rootScope, formatter, $timeout) {
                scope = $rootScope.$new();
                controller = $controller;
                formatterObj = formatter;
                timeout = $timeout;
                rootScope = $rootScope;
                spyOn(rootScope, '$broadcast');
				
				scope.medicationSelectionForm = {
					validationSummary: {
						validate: function () {}
					}
				};

				spyOn(scope.medicationSelectionForm.validationSummary, 'validate').and.returnValue({
					then: function (callback) {
						callback();
					}
				});
                
            });
        });

		describe("initial state", function () {
            it ("should correctly set initial values", function() {
                evaluation = {};

                controller = controller('MedicationsController', {$scope: scope});
                scope.$apply();
                expect(scope.evaluation).toEqual({});
                expect(scope.medTypes).toEqual(medTypes);
                expect(scope.medExamples).toEqual(medExamples);
                expect(scope.toTitleCase).toEqual(formatterObj.toTitleCase);           
            });
        });

        describe("setMedTypes function", function () {
            beforeEach(function () {
                evaluation = {
                    medicationTypes: [
                        'TOPICAL',
                        'OTHER'
                    ],
                    otherMedication: 'Tylenol'
                };

                controller = controller('MedicationsController', {$scope: scope});
            });

            it ("should add the medication type to the list if not found", function () {
                scope.setMedTypes('ORAL');

                expect(evaluation.medicationTypes).toEqual([
                    'TOPICAL',
                    'OTHER',
                    'ORAL'
                ]);
            });

            it ("should remove the medication type if found", function () {
                scope.setMedTypes('TOPICAL');

                expect(evaluation.medicationTypes).toEqual([
                    'OTHER'
                ]);
            });

            it ("should maintain the otherMedication value if the medication type to remove is not 'OTHER'", function () {
                scope.setMedTypes('TOPICAL');

                expect(evaluation.otherMedication).toEqual('Tylenol');
            });

            it ("should clear the otherMedication value if the medication type to remove is 'OTHER'", function () {
                scope.setMedTypes('OTHER');

				expect(evaluation.otherMedication).toEqual('');
			});

			it("should set focus to the other medication details field if 'OTHER' is checked", function () {
				scope.evaluation = {
					medicationTypes: [
						'TOPICAL'
					]
				};
                scope.setMedTypes('OTHER');
                timeout.flush();
				expect(focusServiceMock.focusElement).toHaveBeenCalledWith('#other-medication');
			});

			it("should clear out N/A on other medication details", function () {
				scope.evaluation = {
					medicationTypes: [],
					otherMedication: 'N/A'
				};

				scope.setMedTypes('OTHER');
				expect(scope.evaluation.otherMedication).toEqual('');
			});
		});

        describe("isSelected function", function () {
            beforeEach(function () {
                evaluation = {
                    medicationTypes: [
                        'TOPICAL'
                    ]
                };

                controller = controller('MedicationsController', {$scope: scope});
            });

            it ("should return true if the medType is found in the list", function () {
                expect(scope.isSelected('TOPICAL')).toBeTruthy();
            });

            it ("should return false if the medType is not found in the list", function () {
                expect(scope.isSelected('ORAL')).toBeFalsy();
            });
        });

        describe("previous function", function () {
			beforeEach(function () {
				controller = controller('MedicationsController', {$scope: scope});
			});

			it ("should call previousPage on the mediaRequestNavigationService object", function () {
                scope.previous();

                expect(mediaRequestNavigationServiceMock.previousPage).toHaveBeenCalled();
            });
			
			it("should not call the previousPage function on the mediaRequestNavigationService if validate on the form is not successful", function () {
				scope.medicationSelectionForm.validationSummary.validate.and.returnValue({
					then: function (callback) {
						//empty success block
					}
				});

				scope.previous();

				expect(mediaRequestNavigationServiceMock.previousPage).not.toHaveBeenCalled();
			});

		});

        describe("next function", function () {
			beforeEach(function () {
				controller = controller('MedicationsController', {$scope: scope});
			});

			it ("should call nextPage on the mediaRequestNavigationService object", function () {
                scope.next();

                expect(mediaRequestNavigationServiceMock.nextPage).toHaveBeenCalled();
            });

			it("should call the nextPage function on the mediaRequestNavigationService if validate on the form is successful", function () {
				scope.next();

				expect(mediaRequestNavigationServiceMock.nextPage).toHaveBeenCalled();
			});

			it("should not call the nextPage function on the mediaRequestNavigationService if validate on the form is not successful", function () {
				scope.medicationSelectionForm.validationSummary.validate.and.returnValue({
					then: function (callback) {
						//empty success block
					}
				});

				scope.next();

				expect(mediaRequestNavigationServiceMock.nextPage).not.toHaveBeenCalled();
			});
        });

        describe("swipeLeft function", function () {
			it("should broadcast swipeLeft", function () {
                controller = controller('MedicationsController', {$scope: scope});
                scope.swipeLeft();
				expect(rootScope.$broadcast).toHaveBeenCalledWith('swipeLeft');
			});
        });
        
        describe("swipeRight function", function () {
			it("should broadcast swipeRight", function () {
                controller = controller('MedicationsController', {$scope: scope});
                scope.swipeRight();
				expect(rootScope.$broadcast).toHaveBeenCalledWith('swipeRight');
			});
        });
    });
});