define(['AdditionalCommentsController'], function() {
    'use strict';

    describe("The Additional Comments Controller", function () {
        var controller,
            rootScope,
            scope,
            timeout,
            focusServiceMock,
            mediaRequestNavigationServiceMock,
            mediaRequest = {},
            evaluation = {},
            imageResponseServiceMock;

        beforeEach(function () {
            module('angularTemplateApp');

            focusServiceMock = jasmine.createSpyObj('focusService', ['focusElement']);
            mediaRequestNavigationServiceMock = jasmine.createSpyObj('mediaRequestNavigationService', ['previousPage', 'nextPage']);
            imageResponseServiceMock = jasmine.createSpyObj('imageResponseService', ['getEvaluation']);
            
            imageResponseServiceMock.getEvaluation.and.callFake(function () {
                return evaluation;
            });

            module(function ($provide) {
                $provide.value('focusService', focusServiceMock);
                $provide.value('mediaRequestNavigationService', mediaRequestNavigationServiceMock);
                $provide.value('imageResponseService', imageResponseServiceMock);
            });

            inject(function($controller, $rootScope, $timeout) {
                scope = $rootScope.$new();
                controller = $controller;
                rootScope = $rootScope;
                timeout = $timeout;

                scope.additionalCommentsForm = {
                    $setValidity: function () {},
                    validationSummary: {
                        validate: function () {}
                    }
                };

                spyOn(scope.additionalCommentsForm, '$setValidity');
                spyOn(scope.additionalCommentsForm.validationSummary, 'validate').and.returnValue({
                    then: function (callback) {
                        callback();
                    }
                });
            });
        });

		describe("initial state", function () {
            it("should correctly set initial values", function() {
                evaluation = {};

                controller = controller('AdditionalCommentsController', {$scope: scope});
                scope.$apply();
                expect(scope.evaluation).toEqual({});
            });

            it("should set hasAdditionalComments to null by default", function () {
                evaluation = {};

                controller = controller('AdditionalCommentsController', {$scope: scope});
                scope.$apply();
                expect(scope.hasAdditionalComments).toEqual(null);
            });

            it("should set hasAdditionalComments to true if the additionalComments field on the evaluation exists", function () {
                evaluation = {
                    additionalComments: 'comments'
                };

                controller = controller('AdditionalCommentsController', {$scope: scope});
                scope.$apply();
                expect(scope.hasAdditionalComments).toEqual(true);
            });

            it("should set hasAdditionalComments to false if additionalComments on the evaluation equals 'N/A'", function () {
                evaluation = {
                    additionalComments: 'N/A'
                };

                controller = controller('AdditionalCommentsController', {$scope: scope});
                scope.$apply();
                expect(scope.hasAdditionalComments).toEqual(false);
            });
        });

        describe("updateComments function", function () {
            beforeEach(function () {
                evaluation = {};

                controller = controller('AdditionalCommentsController', {$scope: scope});
                scope.$apply();
            });

            it("should set additionalComments on the evaluation to 'N/A' if 'false' is selected", function () {
                scope.updateComments(false);

                expect(evaluation.additionalComments).toEqual('N/A');
            });

            it("should clear additionalComments on the evaluation if 'true' is selected", function () {
                scope.updateComments(true);

                expect(evaluation.additionalComments).toEqual('');
            });

            it("should set focus to the additional comments field if 'true' is selected", function () {
                scope.updateComments(true);

                timeout.flush();

                expect(focusServiceMock.focusElement).toHaveBeenCalledWith('#additional-comments');
            });
        });

        describe("validate function", function () {
            beforeEach(function () {
                controller = controller('AdditionalCommentsController', {$scope: scope});
            });

            it("should set Additional-Comments-Required to false if hasAdditionalComments is null", function () {
                scope.hasAdditionalComments = null;

                scope.validate();

                expect(scope.additionalCommentsForm.$setValidity).toHaveBeenCalledWith('Additional-Comments-Required', false);
            });

            it("should set Additional-Comments-Required to true if hasAddtionalComments is false", function () {
                scope.hasAdditionalComments = false;

                scope.validate();

                expect(scope.additionalCommentsForm.$setValidity).toHaveBeenCalledWith('Additional-Comments-Required', true);
            });

            it("should set Additional-Comments-Required to true if hasAddtionalComments is true", function () {
                scope.hasAdditionalComments = true;

                scope.validate();

                expect(scope.additionalCommentsForm.$setValidity).toHaveBeenCalledWith('Additional-Comments-Required', true);
            });
        });

        describe("previous function", function () {
            beforeEach(function () {
                controller = controller('AdditionalCommentsController', {$scope: scope});

                spyOn(scope, 'validate');
            });

            it("should call the validate function", function () {
                scope.previous();

                expect(scope.validate).toHaveBeenCalled();
            });

            it("should call the previousPage function on the mediaRequestNavigationService if validate on the form is successful", function () {
                scope.previous();
                
                expect(mediaRequestNavigationServiceMock.previousPage).toHaveBeenCalled();
            });

            it("should not call the previousPage function on the mediaRequestNavigationService if validate on the form is not successful", function () {
                scope.additionalCommentsForm.validationSummary.validate.and.returnValue({
                    then: function (callback) {
                        //empty success block
                    }
                });

                scope.previous();

                expect(mediaRequestNavigationServiceMock.previousPage).not.toHaveBeenCalled();
            });
        });

        describe("next function", function () {
            beforeEach(function () {
                controller = controller('AdditionalCommentsController', {$scope: scope});

                spyOn(scope, 'validate');
            });

            it("should call the nextPage function on the mediaRequestNavigationService", function () {
                scope.next();

                expect(mediaRequestNavigationServiceMock.nextPage).toHaveBeenCalled();
            });

            it("should call the validate function", function () {
                scope.next();

                expect(scope.validate).toHaveBeenCalled();
            });

            it("should call the nextPage function on the mediaRequestNavigationService if validate on the form is successful", function () {
                scope.next();
                
                expect(mediaRequestNavigationServiceMock.nextPage).toHaveBeenCalled();
            });

            it("should not call the nextPage function on the mediaRequestNavigationService if validate on the form is not successful", function () {
                scope.additionalCommentsForm.validationSummary.validate.and.returnValue({
                    then: function (callback) {
                        //empty success block
                    }
                });

                scope.next();

                expect(mediaRequestNavigationServiceMock.nextPage).not.toHaveBeenCalled();
            });
        });

        describe("swipeLeft function", function () {
			it("should broadcast swipeLeft", function () {
                spyOn(rootScope, '$broadcast');
                controller = controller('AdditionalCommentsController', {$scope: scope});
                scope.swipeLeft();
				expect(rootScope.$broadcast).toHaveBeenCalledWith('swipeLeft');
			});
        });
        
        describe("swipeRight function", function () {
			it("should broadcast swipeRight", function () {
                spyOn(rootScope, '$broadcast');
                controller = controller('AdditionalCommentsController', {$scope: scope});
                scope.swipeRight();
				expect(rootScope.$broadcast).toHaveBeenCalledWith('swipeRight');
			});
        });
    });
});