require_relative '../../../globalized'
include Globalized

describe "[Story #JRNL-2482: Hospitalization List View and Filters]", :regression => true  do

  before(:all) do
    @HOSPITALIZATION_FILTER_SCREEN_CONTENT = [
        "Date Range:",
        "Start Date:",
        "MM/DD/YYYY",
        "This date selection control is not optimized for screen readers. "+
            "Please directly enter the date in the text box provided. Open Date Picker",
        "End Date:",
        "MM/DD/YYYY",
        "This date selection control is not optimized for screen readers. "+
            "Please directly enter the date in the text box provided. Open Date Picker",
        "Filter Reset"
    ]

    initializeConfigurations(BASE_URL)
    resetCollection("hospitalization", "medicalhistorydb", "hospitalization")

    initializeAllObjects

    @common.loginEndsOnLandingPage("zztest.patient01", "patient01, zztest")

    @main.clickNavMenu()
    !45.times{ break if (@nav_menu.getNavMenuHeading() == "App Options"); sleep 1 }
    expect(@nav_menu.getNavMenuHeading()).to eq("App Options")

    @nav_menu.clickMyMedical()

    !45.times{ break if (getSecondaryHeader() == "My Medical"); sleep 1 }
    expect(getSecondaryHeader()).to eq("My Medical")

    @mymedical.clickHospitalizations()
    !45.times{ break if (getPrimaryHeader() == "Hospitalizations"); sleep 1 }
    expect(getPrimaryHeader()).to eq("Hospitalizations")
  end

  after(:all) do
    quitDriver()
  end

  it "AC #JRNL-3304|TC #JRNL-2482: (a) Filter Feature (i) Filter defaults to an expanded view. Filter button alt text reads 'Collapse filter' when expanded." do
    expect(@hospitalizations_filter.isFilterAccordionExpanded()).to be true
    expect(isElementVisible(:id,'startDate')).to be true
    expect(isElementVisible(:id,'endDate')).to be true

    expect(@hospitalizations_filter.isStartDatePresent()).to be true
    expect(@hospitalizations_filter.isEndDatePresent()).to be true
    expect(@hospitalizations_filter.getFilterAccordionAltText()).to eq("Collapse filter options")
  end

  it "AC #JRNL-3304|TC #JRNL-2482: (a) Filter Feature (i) Users can collapse the filter list by clicking on the Collapse filter icon.  When collapsed, the filter icon alt text reads Expand Filter.", :broken => true do
    @hospitalizations_filter.clickFilterAccordion()
    expect(@hospitalizations_filter.isFilterAccordionExpanded()).to be false
    expect(@hospitalizations_filter.getFilterAccordionAltText()).to eq("Expand filter options")

    expect(isElementVisible(:id,'startDate')).to be false
    expect(isElementVisible(:id,'endDate')).to be false

    @hospitalizations_filter.clickFilterAccordion()
    expect(@hospitalizations_filter.isFilterAccordionExpanded()).to be true

    expect(isElementVisible(:id,'startDate')).to be true
    expect(isElementVisible(:id,'endDate')).to be true

    expect(@hospitalizations_filter.isStartDatePresent()).to be true
    expect(@hospitalizations_filter.isEndDatePresent()).to be true
  end

  it "AC #JRNL-3304|TC #JRNL-2482: (b) Filter Options (iv) Default view is null for start and end date, resulting in all hospitalizations shown by default.", :broken => true do
    expect(@hospitalizations_filter.getStartDate()).to eq("")
    expect(@hospitalizations_filter.getEndDate()).to eq("")

    #Verify list count on screen is same as count in DB
    hospitalizationList = getAllHospitalizationsInDBSorted()
    !45.times{ break if (getCount() == hospitalizationList.size ); sleep 1 }

    verifyHospitalizationListSame(@hospitalizations_list.getHospitalizationsList(), getAllHospitalizationsInDBSorted())
  end

  it "AC #JRNL-3304|TC #JRNL-2482: (b) Filter Options (all are optional)", :broken => true do
    #Verify 'required field' is not on screen
    expect(@hospitalizations_filter.getScreenContent()).to_not include("required")

    #Verify able to search with all fields empty
    @hospitalizations_filter.clearFromDateFiled()
    @hospitalizations_filter.clearEndDateField()

    expect(@hospitalizations_filter.getStartDate()).to be_empty
    expect(@hospitalizations_filter.getEndDate()).to be_empty

    @hospitalizations_filter.clickFilterBtn()

    verifyHospitalizationListSame(@hospitalizations_list.getHospitalizationsList(), getAllHospitalizationsInDBSorted())
  end

  it "AC #JRNL-3304|TC #JRNL-2482: (b) Filter Options (i) Date Range field is shown with Start Date and End date fields.", :broken => true do
    #Verify Start Date and End Date fields are on the screen
    expect(@hospitalizations_filter.isStartDatePresent()).to be true
    expect(@hospitalizations_filter.isEndDatePresent()).to be true

    #Verify Start Date and End Date labels are on the screen
    expect(@hospitalizations_filter.getStartDateLabel()).to include("Start Date:")
    expect(@hospitalizations_filter.getEndDateLabel()).to include("End Date:")

    hospitalizationsFilterForm = @hospitalizations_filter.getScreenContent().split("\n")
    @HOSPITALIZATION_FILTER_SCREEN_CONTENT.each_with_index do | expectedScreenRow, i |
      #DEBUG puts "checking [#{hospitalizationForm[i]}] with [#{expectedScreenRow}]"
      expect(hospitalizationsFilterForm[i]).to eq(expectedScreenRow)
    end

    expect(hospitalizationsFilterForm.length).to eq(@HOSPITALIZATION_FILTER_SCREEN_CONTENT.length)
  end

  it "AC #JRNL-3304|TC #JRNL-2482: (b) Filter Options (ii) Date range field are date picker controls and formatted as mm/dd/yyyy", :broken => true do
    expect(@hospitalizations_filter.getStartDatePlaceHolder()).to eq("MM/DD/YYYY")
    expect(@hospitalizations_filter.getEndDatePlaceHolder()).to eq("MM/DD/YYYY")

    #Verify date picker for Start and End dates are on the screen
    @hospitalizations_filter.clickStartDateCalendarBtn()
    !45.times{ break if (@date_picker.getSetButtonText == "SET"); sleep 1 }
    @date_picker.setDateFromGUI(3, 24, 2014)
    expect(@hospitalizations_filter.getStartDate()).to eq("03/24/2014")

    @hospitalizations_filter.clickEndDateCalendarBtn()
    !45.times{ break if (@date_picker.getSetButtonText == "SET"); sleep 1 }
    @date_picker.setDateFromGUI(4, 25, 2014)
    expect(@hospitalizations_filter.getEndDate()).to eq("04/25/2014")
  end

  it "AC #JRNL-3304|TC #JRNL-2482: (b) Filter Options (iii) Start date must be before or equal to end date," do
    # otherwise an error message is shown when the Filter button is selected.
    # The error message is located in the standard location, at the top of the detail screen." do

    validateError("01/01/2014", "09/23/2014", [])  #Start Date < End Date should be no error
    validateError("09/19/2014", "07/17/2010", ["Start Date must occur before End Date."])    #Start Date > End Date should be error
    validateError("09/23/2014", "09/23/2014", [])  #Start Date == End Date should be no error
  end

  it "AC #JRNL-3304|TC #JRNL-2482: (b) Filter Options (v) If start date is valued, end date is required, and vice versa." do
    validateError("", "05/21/1931", ["Please provide both a Start Date and End Date to filter by Date Range."] ) #if Start Date == blank and End Date != blank, should be error
    validateError("", "", [])                    #if Start Date == blank and End Date == blank, should be no error
    validateError("09/10/2000", "", ["Please provide both a Start Date and End Date to filter by Date Range."])  #if Start Date != blank and End Date == blank, should be error
    validateError("09/10/2000", "09/10/2014", []) #if Start Date != blank and End Date != blank, should be no error
  end

  it "AC #JRNL-3304|TC #JRNL-2482: (c) Filter Button (alt text = Filter)", :broken => true do
    expect(@hospitalizations_filter.getFilterButtonLabel()).to eq("Filter")
  end

  it "AC #JRNL-3304|TC #MVAH-1220: (c) Filter Button (i) Adding a value to any of the filter fields and selecting Filter button updates the table view to show records that match the filter parameters.", :broken => true do
    #Various range values returned
    @hospitalizations_filter.setDateRangeAndFilter("", "")

    verifyHospitalizationListSame(@hospitalizations_list.getHospitalizationsList(), getSortedHospitalizationsByDateRange("", ""))

    @hospitalizations_filter.setDateRangeAndFilter("01/01/2014", "06/01/2014")

    verifyHospitalizationListSame(@hospitalizations_list.getHospitalizationsList(), getSortedHospitalizationsByDateRange("01/01/2014", "06/01/2014"))
  end

  it "AC #JRNL-3304|TC #MVAH-1220: (c) Filter Button (ii) If no records are found when a filter is applied the message 'No results found.' Is displayed in the screen area in place of the table view." do
    #None found
    @hospitalizations_filter.setDateRangeAndFilter("01/01/1990", "12/31/1990")
    !10.times{ break if (getFocusedElementText().include?("No results were found with the current filters. Use the Add button above to create a new record.")); sleep 1 }
    expect(getNoResultsFoundMessage()).to eq("No results were found with the current filters. Use the Add button above to create a new record.")
  end

  it "AC #JRNL-3304|TC #JRNL-2482: (c) Filter Button (iii) After the filter button is selected, the focus moves to the returned results, first row, or the No results found message.", :broken => true do
    #Not found and check where focus is on
    @hospitalizations_filter.setDateRangeAndFilter("01/01/1991", "09/17/1991")
    !10.times{ break if (getFocusedElementText().include?("No results were found with the current filters. Use the Add button above to create a new record.")); sleep 1 }
    expect(getFocusedElementText()).to eq("No results were found with the current filters. Use the Add button above to create a new record.")

    #Find results and check where focus is on
    @hospitalizations_filter.setDateRangeAndFilter("10/01/2013", "12/31/2013")
    !10.times{ break if (getFocusedElementText().include?("12/12/2013")); sleep 1 }
    expect(getFocusedElementText()).to match("12/12/2013")
  end

  it "AC #JRNL-3304|TC #JRNL-2482: d.	Hospitalization entries are displayed in a table format, sorted in reverse chronological order by Start Date, End Date, and Facility.", :broken => true do
    expect(getTableHeaders()).to eq(["Start Date", "End Date", "Facility", "Source"])

    verifyHospitalizationListSame(@hospitalizations_list.getHospitalizationsList(), getSortedHospitalizationsByDateRange("10/01/2013", "12/31/2013"))
  end

  it "Story JRNL-1696|AC JRNL-1583: Reset Button", :broken => true do
    expect(@hospitalizations_filter.isResetButtonPresent()).to eq(true)
    expect(@hospitalizations_filter.getResetButtonLabel()).to eq("Reset")

    @hospitalizations_filter.clickResetBtn()
    !30.times { break if (@hospitalizations_list.getFocusedElementText().include?("Name")); sleep 1 }
    expect(@hospitalizations_list.getFocusedElementText()).to eq(getTextForRow(1) + "\n ")
    expect(getCount()).to eq(23)
    verifyHospitalizationListSame(@hospitalizations_list.getHospitalizationsList(), getAllHospitalizationsInDBSorted())

    # DEFAULTS
    expect(@hospitalizations_filter.getStartDate()).to eq("")
    expect(@hospitalizations_filter.getEndDate()).to eq("")
  end

  #######################
  ### Local Functions ###
  #######################

  def getAllHospitalizationsInDBSorted

    def sortHospitalizationKeys(x,y)
      #Sort startDate (descend), endDate(descend), facility(ascend)
      returnval = -(x["admissionDate"] <=> y["admissionDate"])
      if returnval == 0
        returnval = -(x["dischargeDate"] <=> y["dischargeDate"])
        if returnval == 0
          return x["facility"] <=> y["facility"]
        else
          return returnval
        end
      else
        return returnval
      end
    end

    return retreiveAllDocumentsInCollection("hospitalization", "medicalhistorydb").sort { |x, y| sortHospitalizationKeys(x,y) }
  end

  def getSortedHospitalizationsByDateRange(startDate, endDate)
    #Given start and end date parameters, extract the hospitalizations that are in the db
    selectedHospitalizations = []
    filterStartDate = "#{startDate[6..9]}-#{startDate[0..1]}-#{startDate[3..4]}"
    filterEndDate = "#{endDate[6..9]}-#{endDate[0..1]}-#{endDate[3..4]}"
    getAllHospitalizationsInDBSorted.each_with_index do |hospitalization_event, rowIndex|
      if filterStartDate == "--" and filterEndDate == "--"
        selectedHospitalizations << hospitalization_event
      #elsif filterStartDate == "--" and filterEndDate != "--"
      #  if (hospitalization_event["admissionDate"].strftime('%Y-%m-%d') <= filterEndDate)
      #    selectedHospitalizations << hospitalization_event
      #  end
      #elsif filterStartDate != "--" and filterEndDate == "--"
      #  if (hospitalization_event["dischargeDate"].strftime('%Y-%m-%d') >= filterStartDate)
      #    selectedHospitalizations << hospitalization_event
      #  end
      elsif ((hospitalization_event["admissionDate"].strftime('%Y-%m-%d') >= filterStartDate) or
          (hospitalization_event["dischargeDate"].strftime('%Y-%m-%d') >= filterStartDate)) and
         ((hospitalization_event["admissionDate"].strftime('%Y-%m-%d') <= filterEndDate) or
          (hospitalization_event["dischargeDate"].strftime('%Y-%m-%d') <= filterEndDate))
        selectedHospitalizations << hospitalization_event
      end
    end

    return selectedHospitalizations
  end

  def verifyHospitalizationListSame(screenList, dbList)
    #expect(screenList.length).to eq(dbList.length)
    dbList.each_with_index do |dbrow, i|
      screenRow = screenList[i]
      expect(screenRow[0]).to eq(dbrow['admissionDate'].strftime('%m/%d/%Y'))
      expect(screenRow[1]).to eq(dbrow['dischargeDate'].strftime('%m/%d/%Y'))
      expect(screenRow[2]).to eq(dbrow['facility'])
      expect(screenRow[3]).to eq(dbrow['source'])
    end
  end

  def validateError(startDate, endDate, errors)
    @hospitalizations_filter.setDateRangeAndFilter(startDate, endDate)

    expectedValidationSummaryHeader = errors.length > 0 ? "The following errors were found:" : ""

    sleep 1

    if expectedValidationSummaryHeader == ""
      #Expecting no errors
      !3.times{ break if (@errorSummary.isErrorSummaryElementPresent("filterForm")); sleep 1 }
      expect(@errorSummary.isErrorSummaryElementPresent("filterForm")).to eq(false)
    else
      !45.times{ break if (@errorSummary.getValidationSummaryHeader("filterForm") == expectedValidationSummaryHeader); sleep 1 }
      expect(@errorSummary.getValidationSummaryHeader("filterForm")).to eq(expectedValidationSummaryHeader)
      if(expectedValidationSummaryHeader != "")
        for i in 0..(errors.length - 1)
          if errors[i] != nil
            expect(@errorSummary.isErrorMessageDisplayed("filterForm", errors[i])).to eq(true)
          else
            expect(@errorSummary.isErrorMessageDisplayed("filterForm", errors[i])).to eq(false)
          end
        end
      else
        expect(@errorSummary.isErrorSummaryElementPresent("filterForm")).to eq(false)
      end
    end
  end

end