require_relative '../../globalized'
include Globalized

describe "[Story #JRNL-291: Date Selection]
          [Story #JRNL-2509: Move Diet to the grouped Personal Tracker View]
          [Story #JRNL-2508: Move Exercise to the grouped Personal Tracker View]
          [Story #JRNL-2507: Move Mood to the grouped Personal Tracker View]
          [Story #JRNL-2506: Move Pain to the grouped Personal Tracker View]
          [Story #JRNL-2505: Move Daily Events to the grouped Personal Tracker View]", :regression =>true do

  before(:all) do
    initializeConfigurations(BASE_URL)
    deleteDailyEventTable("D123401")
    initializeAllObjects

    START_DATE_LABEL = "Start Date"
    END_DATE_LABEL = "End Date"
    POSSIBLE_ERRORS_TEMPLATE = Hash["required" => " field is required.",
                                    "format" => " must be formatted MM/DD/YYYY.",
                                    "invalid" => " is an invalid date.",
                                    "range" => " must be between <min-date> and <max-date>.",
                                    "endBeforeStart" => "<start-label> must occur before <end-label>."]

    @common.loginEndsOnLandingPage("zztest.patient01", "patient01, zztest")
    expect(@main.getFooterText()).to eq("Logged in as patient01, zztest")
  end

  after(:all) do
    quitDriver()
  end

  describe "Journal Sections" do

    before(:all) do
      maxDate = getDateNthDaysFromNow(0, "%m/%d/%Y")
      minDate = "01/01/1900"

      POSSIBLE_ERRORS = Hash[];

      POSSIBLE_ERRORS_TEMPLATE.each {|key, value|
        errorString = value.gsub("<min-date>", minDate)
        errorString.gsub!("<max-date>", maxDate)
        errorString.gsub!("<start-label>", START_DATE_LABEL)
        errorString.gsub!("<end-label>", END_DATE_LABEL)
        if key != "endBeforeStart"
          POSSIBLE_ERRORS["start-" + key] = START_DATE_LABEL + errorString
          POSSIBLE_ERRORS["end-" + key] = END_DATE_LABEL + errorString
        else
          POSSIBLE_ERRORS[key] = errorString
        end
      }
    end

    describe "#Date Selection Validation for Daily Events" do

      before(:all) do
        @main.clickNavMenu
        @navMenu.clickPersonalTrackers
        @personalTrackers.clickDailyEvents
      end

      it "#AC #JRNL-1898|TC #JRNL-1810: Date Range for Move Daily Events to the grouped Personal Tracker View
          AC #JRNL-349: The user will be able to enter a date manually" do
        validateUserCanEnterDate()
      end

      it "#AC #JRNL-1898|TC #JRNL-1810: Date Range for Move Daily Events to the grouped Personal Tracker View
          AC #JRNL-350|TC #JRNL-222: The date will be displayed in mm/dd/yyyy format", :broken => true do
        validateDateFormat()
      end

      it "#AC #JRNL-1898|TC #JRNL-1810: Date Range for Move Daily Events to the grouped Personal Tracker View
          AC #JRNL-361|TC #JRNL-222: Date Validation", :broken => true do
        validateErrorMessages()
      end
    end

  end

  def validateUserCanEnterDate
    @date_filter.setDateRange("01/02/2013", "06/04/2014")
    expect(@date_filter.getStartDate()).to eq("01/02/2013")
    expect(@date_filter.getEndDate()).to eq("06/04/2014")

    @date_filter.setDateRange("04/01/2014", "05/04/2014")
    expect(@date_filter.getStartDate()).to eq("04/01/2014")
    expect(@date_filter.getEndDate()).to eq("05/04/2014")

    @date_filter.setDateRange("04/01/2014", "04/01/2014")
    expect(@date_filter.getStartDate()).to eq("04/01/2014")
    expect(@date_filter.getEndDate()).to eq("04/01/2014")
  end

  def validateDateFormat()
    @date_filter.clearFromDateFiled
    @date_filter.clearEndDateField

    expect(@date_filter.getStartDatePlaceHolder).to eq("MM/DD/YYYY")
    expect(@date_filter.getEndDatePlaceHolder).to eq("MM/DD/YYYY")

    validateError("", "", [POSSIBLE_ERRORS["start-required"], POSSIBLE_ERRORS["end-required"]])
    validateError("1", "1", [POSSIBLE_ERRORS["start-format"], POSSIBLE_ERRORS["end-format"]])
    validateError("03/01/2014", "05/01/2014", [])

    expect(@date_filter.getStartDate()).to eq("03/01/2014")
    expect(@date_filter.getEndDate()).to eq("05/01/2014")
  end

  def validateErrorMessages()
    # The date will be validated (day, month and year should be valid)
    # POSSIBLE_ERRORS = [
    # "Start Date field is required.",
    # "End Date field is required.",
    # "Start Date must be formatted MM/DD/YYYY.",
    # "End Date must be formatted MM/DD/YYYY.",
    # "Start Date cannot be a future date.",
    # "End Date cannot be a future date.",
    # "Start Date must occur before End Date."]

    #  Month values (01 - 12)
    validateError("01/24/2013", "02/21/2014", [] )
    validateError("03/24/2013", "04/21/2014", [] )
    validateError("05/24/2013", "06/21/2014", [] )
    validateError("07/24/2013", "08/21/2014", [] )
    validateError("09/24/2013", "10/21/2014", [] )
    validateError("11/24/2013", "12/21/2014", [] )
    validateError("13/24/2013", "00/21/2014", [POSSIBLE_ERRORS["start-format"], POSSIBLE_ERRORS["end-format"]] )

    # Day values:
    # 01-31 for Jan, March, May, July, Aug, Oct, Dec
    # 01-30 for Sept, April, June, Nov
    # 01-29 for Feb
    validateError("01/31/2013", "05/21/2014", [] )
    validateError("02/29/2008", "02/29/2009", [POSSIBLE_ERRORS["end-invalid"]])
    validateError("03/31/2013", "04/30/2013", [] )
    validateError("05/31/2013", "06/30/2013", [] )
    validateError("07/31/2013", "08/31/2013", [] )
    validateError("09/30/2013", "10/31/2013", [] )
    validateError("11/30/2013", "12/31/2013", [] )
    validateError("01/33/2013", "04/32/2013", [POSSIBLE_ERRORS["start-format"], POSSIBLE_ERRORS["end-format"]] )

    # Year value: Must begin with 19 or 20
    validateError("01/31/1813", "05/21/0024", [POSSIBLE_ERRORS["start-format"], POSSIBLE_ERRORS["end-format"]] )

    # Invalid format/characters
    validateError("ab/bb/cccc", "gg/hh/iiii", [POSSIBLE_ERRORS["start-required"], POSSIBLE_ERRORS["end-required"]])
    validateError("0/24/2014", "06/72", [POSSIBLE_ERRORS["start-format"], POSSIBLE_ERRORS["end-format"]])
    # Defect year 201 is accepted
    validateError("02/24/2014", "05/21/201", [POSSIBLE_ERRORS["end-format"]])

    # The date cannot be future date
    tomorrow = getDateNthDaysFromNow(1, "%m/%d/%Y")
    dayAfterThat = getDateNthDaysFromNow(2, "%m/%d/%Y")
    year = Time.new.year.to_s
    if tomorrow[-4..-1] != year
      validateError(tomorrow, dayAfterThat, [POSSIBLE_ERRORS["start-format"], POSSIBLE_ERRORS["end-format"]])
    elsif dayAfterThat[-4..-1] != year
      validateError(tomorrow, dayAfterThat, [POSSIBLE_ERRORS["start-range"], POSSIBLE_ERRORS["end-format"]])
    else
      validateError(tomorrow, dayAfterThat, [POSSIBLE_ERRORS["start-range"], POSSIBLE_ERRORS["end-range"]])
    end
    validateError("05/01/2014", "03/01/2014", [POSSIBLE_ERRORS["endBeforeStart"]])
    validateError("1", "06/30/2018", [POSSIBLE_ERRORS["start-format"], POSSIBLE_ERRORS["end-format"]])
  end

  def validateError(startDate, endDate, errors)
    # puts" validate error(start date=" + startDate + ", end date=" + endDate + ")"
    @date_filter.setDateRangeAndFilter(startDate, endDate)
    expectedValidationSummaryHeader = errors.length > 0 ? "The following errors were found:" : ""

    !45.times{ break if (@error.getValidationSummaryHeader("filterForm") == expectedValidationSummaryHeader); sleep 1 }
    expect(@error.getValidationSummaryHeader("filterForm")).to eq(expectedValidationSummaryHeader)

    if(expectedValidationSummaryHeader != "")
      POSSIBLE_ERRORS.each_value {|value|
        # UNCOMMENT TO HELP DEBUG
        # puts "Checking error " + value
        if errors.index(value) != nil
          expect(@error.isErrorMessageDisplayed("filterForm", value)).to eq(true)
        else
          expect(@error.isErrorMessageDisplayed("filterForm", value)).to eq(false)
        end
      }
    else
      # puts "No errors displayed"
      expect(@error.isErrorSummaryElementPresent("filterForm")).to eq(false)
    end
  end
end
