package com.agilex.va.mpi.model;

import static org.junit.Assert.assertEquals;

import org.junit.Ignore;
import org.junit.Test;

import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import com.agilex.va.mpi.cache.CacheService;
import com.agilex.va.mpi.cache.LastInCacheService;

public class PatientCorrelationIdTest {

    public static final PatientCorrelationId CACHE_KEY = new PatientCorrelationId("42", "NI", "200DOD", "USDOD");
    public static final String CTK_123_UNIQUE_ID = "CTK-123";
    public static final String W_000001_UNIQUE_ID = "W000001";
    public static final String ICN_ASSIGNING_AUTHORITY = "ICN";
    public static final String MOCK_ASSIGNING_AUTHORITY = "mock-aa";
    public static final String VISTA_ASSIGNING_AUTHORITY = "dfn-123";

    @Test(expected = IllegalArgumentException.class)
    public void badlyFormatedHl7v2Id() {
        PatientCorrelationId correlationId = new PatientCorrelationId("1^2^3");
    }

    @Test
    public void createIdExtension() {
        PatientCorrelationId correlationId = createPatientIdExtension();

        assertEquals("A^B^C^D", correlationId.createHl7Id());
    }

    @Test
    public void createIdExtensionWithNull() {
        PatientCorrelationId correlationId = createNullSourcePatientIdExtension();

        assertEquals("A^B^D", correlationId.createHl7Id());
    }

    @Test
    public void toStringWithNull() {
        PatientCorrelationId correlationId = createNullSourcePatientIdExtension();

        assertEquals("A B D", correlationId.toString());
    }

    private PatientCorrelationId createNullSourcePatientIdExtension() {
        return new PatientCorrelationId("A", "B", null, "D");
    }

    @Test
    public void createFromHl7v2Id() {
        PatientCorrelationId compare = createPatientIdExtension();
        PatientCorrelationId correlationId = new PatientCorrelationId("A^B^C^D");

        assertEquals(compare, correlationId);
    }

    private PatientCorrelationId createPatientIdExtension() {
        return new PatientCorrelationId("A", "B", "C", "D");
    }

	//FIXME Test data issue with system identifier
    @Ignore
    public void getAllIdTranslations() {
        CacheService cache = createCache();

        PatientIdentifiers patientIdentifiers = CACHE_KEY.getAllIdTranslations(cache);

        assertEquals(3, patientIdentifiers.size());

        assertEquals(createPatientIdentifier(CTK_123_UNIQUE_ID, ICN_ASSIGNING_AUTHORITY), patientIdentifiers.get(0));
        assertEquals(createPatientIdentifier(W_000001_UNIQUE_ID, MOCK_ASSIGNING_AUTHORITY), patientIdentifiers.get(1));
        assertEquals(createPatientIdentifier(W_000001_UNIQUE_ID, VISTA_ASSIGNING_AUTHORITY), patientIdentifiers.get(2));
    }

    @Test
    public void getAllIdTranslationsNoneFound() {
        CacheService cache = new LastInCacheService();
        PatientIdentifiers patientIdentifiers = CACHE_KEY.getAllIdTranslations(cache);

        assertEquals(0, patientIdentifiers.size());
    }

    @Test
    public void getIdTranslation() {
        CacheService cache = createCache();

        PatientIdentifiers patientIdentifiers = CACHE_KEY.getIdTranslation(MOCK_ASSIGNING_AUTHORITY, cache);

        assertEquals(createPatientIdentifier(W_000001_UNIQUE_ID, MOCK_ASSIGNING_AUTHORITY), patientIdentifiers.get(0));
    }



    @Test
    public void getIdTranslationNotFound() {
        CacheService cache = createCache();

        PatientIdentifiers patientIdentifiers = CACHE_KEY.getIdTranslation("empire", cache);

        assertEquals(0, patientIdentifiers.size());
    }

    private PatientIdentifier createPatientIdentifier(String uniqueId, String assigningAuthority) {
        PatientIdentifier patientIdentifier = new PatientIdentifier();

        patientIdentifier.setUniqueId(uniqueId);
        patientIdentifier.setAssigningAuthority(assigningAuthority);

        return patientIdentifier;
    }

    public CacheService createCache() {

        PatientCorrelationId[] ids = new PatientCorrelationId[3];
        ids[0] = PatientCorrelationId.createId(CTK_123_UNIQUE_ID, ICN_ASSIGNING_AUTHORITY);
        ids[1] = PatientCorrelationId.createId(W_000001_UNIQUE_ID, MOCK_ASSIGNING_AUTHORITY);
        ids[2] = PatientCorrelationId.createId(W_000001_UNIQUE_ID, VISTA_ASSIGNING_AUTHORITY);

        CacheService service = new LastInCacheService();
        service.add(CACHE_KEY, ids);

        return service;
    }
}
