package com.agilex.va.mpi.model;

import com.agilex.soap.*;
import org.junit.*;

import java.util.LinkedList;

import static junit.framework.Assert.*;
import static org.mockito.Mockito.*;

public class GetIdsResponseTest {

    private static final String UNKNOWN_ID_ERR_MSG = "The correlation does not exist for the given id.";
    private static final String INVALID_ID_TYPE_ERR_MSG = "The given id has an invalid identifier type.";
    private static final String INVALID_ID_FORMAT_ERR_MSG = "The given id has an invalid vpid format.";
    private static final String UNKNOWN_ERR_MSG = "An unknown error occurred.";

    private final static String[] ackDetailTextNodeNames = new String[] {"acknowledgement", "acknowledgementDetail", "text"};
    private final static String[] patientIdsNodeNames = new String[] {"controlActProcess", "subject", "registrationEvent", "subject1", "patient"};
    public static final String PREFIX = "ns1";
    public static final String NAME_SPACE = "urn:hl7-org:v3";

    private Message message;
    private GetIdsResponse response;

    @Before
    public void setup() {
        message = mock(Message.class);
        response = new GetIdsResponse(message);
    }

    @Test(expected = RuntimeException.class)
    public void emptyMessageTest() {
        GetIdsResponse response = new GetIdsResponse(new Message());
        response.processMessage();

        assertNotNull(response.getIds());
        assertNull(response.getMessageException());
    }

    @Test
    public void unknownIdExceptionReturned() {
        addMockAcknowledgementDetails(UNKNOWN_ID_ERR_MSG);

        assertTrue(response.getMessageException() instanceof IllegalArgumentException);
        assertEquals(UNKNOWN_ID_ERR_MSG, response.getMessageException().getMessage());
    }

    @Test
    public void invalidIdTypeExceptionReturned() {
        addMockAcknowledgementDetails(INVALID_ID_TYPE_ERR_MSG);

        assertTrue(response.getMessageException() instanceof IllegalArgumentException);
        assertEquals(INVALID_ID_TYPE_ERR_MSG, response.getMessageException().getMessage());
    }

    @Test
    public void invalidIdFormatExceptionReturned() {
        addMockAcknowledgementDetails(INVALID_ID_FORMAT_ERR_MSG);

        assertTrue(response.getMessageException() instanceof IllegalArgumentException);
        assertEquals(INVALID_ID_FORMAT_ERR_MSG, response.getMessageException().getMessage());
    }

    @Test
    public void unknownMessageExceptionReturned() {
        addMockAcknowledgementDetails(UNKNOWN_ERR_MSG);

        assertTrue(response.getMessageException() instanceof RuntimeException);
        assertEquals(UNKNOWN_ERR_MSG, response.getMessageException().getMessage());
    }

    private void addMockAcknowledgementDetails(String errorMessage) {
        Element mockElement = getMockRoot();
        when(mockElement.getChild(ackDetailTextNodeNames, PREFIX, NAME_SPACE)).thenReturn(mockElement);
        when(mockElement.getElementValue()).thenReturn(errorMessage);
     }

    @Test
    public void noIdsInMessage() {
        addPatient();

        assertNotNull(response.getIds());
        assertEquals(0, response.getIds().length);
    }

    @Test
    public void getIdsFromMessage() {
        Elements ids = addPatient();
        LinkedList linkedList = new LinkedList();
        linkedList.add("id^idtype^site^issuer");
        when(ids.getAttributeValue("extension")).thenReturn(linkedList);

        assertNotNull(response.getIds());
        assertEquals(1, response.getIds().length);
    }

    private Elements addPatient() {
        Element mockElement = getMockRoot();
        when(mockElement.getChild(patientIdsNodeNames, PREFIX, NAME_SPACE)).thenReturn(mockElement);

        Elements mockElements = mock(Elements.class);
        when(mockElement.getChildren("id", PREFIX, NAME_SPACE)).thenReturn(mockElements);

        return mockElements;
    }

    private Element getMockRoot() {
        Body mockBody = mock(Body.class);
        Element mockElement = mock(Element.class);

        when(message.getBody()).thenReturn(mockBody);
        when(mockBody.getRootElement()).thenReturn(mockElement);
        return mockElement;
    }
}
