package com.agilex.va.mpi.model;

import com.agilex.soap.*;
import org.junit.*;

import java.text.*;
import java.util.Date;

import static org.hamcrest.CoreMatchers.*;
import static org.junit.Assert.*;

public class GetIdsRequestBuilderTest {

    private static final String ID = "id";
    private static final String PREFIX = "ns1";
    private static final String NAME_SPACE = "urn:hl7-org:v3";
    private static final String EXTENSION = "extension";
    private static final String ROOT = "root";
    private static final String TYPE_CODE = "typeCode";
    private static final String CLASS_CODE = "classCode";
    private static final String DETERMINER_CODE = "determinerCode";
    private static final String DEV = "DEV";
    private static final String INSTANCE = "INSTANCE";
    private static final String DEVICE = "device";
    private static final String CODE = "code";
    private static final String VALUE = "value";
    private final static String[] queryIdNodeNames = new String[] {"controlActProcess", "queryByParameter", "queryId"};

    private GetIdsRequestBuilder builder;

    @Before
    public void setup() {
        builder = new GetIdsRequestBuilder(ROOT, EXTENSION);
    }
    @Test
    public void buildMessage() throws ParseException {
        Message message = builder.build();

        Element root = message.getBody().getRootElement();

        assertEquals("XML_1.0", root.getAttributeValue("ITSVersion"));
        assertEquals("2.16.840.1.113883.4.349", root.getChild(ID, PREFIX, NAME_SPACE).getAttributeValue(ROOT));

        isValidDate(root.getChild("creationTime", PREFIX, NAME_SPACE).getAttributeValue(VALUE));

        assertEquals("2.16.840.1.113883.1.6", root.getChild("interactionId", PREFIX, NAME_SPACE).getAttributeValue(ROOT));
        assertEquals("PRPA_IN201309UV02", root.getChild("interactionId", PREFIX, NAME_SPACE).getAttributeValue(EXTENSION));
        assertEquals("T", root.getChild("processingCode", PREFIX, NAME_SPACE).getAttributeValue(CODE));
        assertEquals("T", root.getChild("processingModeCode", PREFIX, NAME_SPACE).getAttributeValue(CODE));
        assertEquals("AL", root.getChild("acceptAckCode", PREFIX, NAME_SPACE).getAttributeValue(CODE));

        Element element = root.getChild("receiver", PREFIX, NAME_SPACE);
        assertEquals("RCV", element.getAttributeValue(TYPE_CODE));

        element = element.getChild(DEVICE, PREFIX, NAME_SPACE);
        assertEquals(DEV, element.getAttributeValue(CLASS_CODE));
        assertEquals(INSTANCE, element.getAttributeValue(DETERMINER_CODE));
        assertEquals("2.16.840.1.113883.4.349", element.getChild(ID, PREFIX, NAME_SPACE).getAttributeValue(ROOT));
        assertNull(element.getChild(ID, PREFIX, NAME_SPACE).getAttributeValue(EXTENSION));

        element = root.getChild("sender", PREFIX, NAME_SPACE);
        assertEquals("SND", element.getAttributeValue(TYPE_CODE));

        element = element.getChild(DEVICE, PREFIX, NAME_SPACE);
        assertEquals(DEV, element.getAttributeValue(CLASS_CODE));
        assertEquals(INSTANCE, element.getAttributeValue(DETERMINER_CODE));
        assertEquals("2.16.840.1.113883.3.933", element.getChild("id", PREFIX, NAME_SPACE).getAttributeValue(ROOT));
        assertEquals("200CIH", element.getChild(ID, PREFIX, NAME_SPACE).getAttributeValue(EXTENSION));

        element = root.getChild("controlActProcess", PREFIX, NAME_SPACE);
        assertEquals("EVN", element.getAttributeValue("moodCode"));
        assertEquals("PRPA_TE201309UV02", element.getChild(CODE, PREFIX, NAME_SPACE).getAttributeValue(CODE));
        assertEquals("2.16.840.1.113883.1.6", element.getChild(CODE, PREFIX, NAME_SPACE).getAttributeValue("codeSystem"));

        element = element.getChild("queryByParameter", PREFIX, NAME_SPACE);
        assertEquals("new", element.getChild("statusCode", PREFIX, NAME_SPACE).getAttributeValue(CODE));
        assertEquals("I", element.getChild("responsePriorityCode", PREFIX, NAME_SPACE).getAttributeValue(CODE));
        assertEquals("2.16.840.1.113883.3.933", element.getChild("queryId", PREFIX, NAME_SPACE).getAttributeValue(ROOT));

        element = element.getChild(new String[] {"parameterList", "patientIdentifier"}, PREFIX, NAME_SPACE);
        assertEquals("Patient.Id", element.getChild("semanticsText", PREFIX, NAME_SPACE).getElementValue());
        assertEquals(ROOT, element.getChild(VALUE, PREFIX, NAME_SPACE).getAttributeValue(ROOT));
        assertEquals("extension", element.getChild(VALUE, PREFIX, NAME_SPACE).getAttributeValue(EXTENSION));
    }

    private void isValidDate(String value) throws ParseException {
        DateFormat df = new SimpleDateFormat("yyyyMMddHHmmss");
        Date now = df.parse(value);
    }

    @Test
    public void idsAreUnique() {
        Message message1 = builder.build();
        Message message2 = builder.build();

        assertThat(getMessageIdExtension(message1), not(equalTo(getMessageIdExtension(message2))));
        assertThat(getQueryIdExtension(message1), not(equalTo(getQueryIdExtension(message2))));
    }

    private String getMessageIdExtension(Message message) {
        return message.getBody().getRootElement().getChild(ID, PREFIX, NAME_SPACE).getAttributeValue(EXTENSION);
    }
    private String getQueryIdExtension(Message message) {
        return message.getBody().getRootElement().getChild(queryIdNodeNames, PREFIX, NAME_SPACE).getAttributeValue(EXTENSION);
    }
}