package com.agilex.va.mpi.model;

import com.agilex.healthcare.mobilehealthplatform.domain.*;
import com.agilex.va.mpi.cache.CacheService;
import com.google.common.base.Joiner;
import com.google.common.base.Strings;

public class PatientCorrelationId {
    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientCorrelationId.class);

    private final String id;
    private final String idType;
    private final SystemContext context;

    public static PatientCorrelationId createId(String uniqueId, String assigningAuthority) {
        return new PatientCorrelationId(uniqueId,
                                      IdTransformer.getIdType(assigningAuthority),
                                      IdTransformer.getSource(assigningAuthority),
                                      IdTransformer.getIssuer(assigningAuthority));
    }

    public PatientCorrelationId(String hl7v2FormatId) {
        String[] parts = hl7v2FormatId.split("\\^");

        if (parts == null || parts.length < 4)
            throw new IllegalArgumentException("Cannot process id.  Patient Id is in an invalid format.  Format should be: x^x^x^x");

        id = parts[0];
        idType = parts[1];

        context = new SystemContext(parts[2], parts[3]);
    }

    public PatientCorrelationId(String id, String idType, String source, String issuer) {
        this.id = id;
        this.idType = idType;

        context = new SystemContext(source, issuer);
     }

    public String createHl7Id() {
        return combine("^");
    }

    public PatientIdentifiers getIdTranslation(String toAssigningAuthority, CacheService cache) {
        if (logger.isDebugEnabled()) logger.debug("Looking up id translation");

        PatientIdentifiers patientIdentifiers = new PatientIdentifiers();

        PatientCorrelationId correlationId = matchOnAssigningAuthority(toAssigningAuthority, cache);

        if (correlationId != null && isValidAssigningAuthority(correlationId)) {
            PatientIdentifier patientIdentifier = createPatientIdentifier(correlationId);
            patientIdentifiers.add(patientIdentifier);
        }

        return patientIdentifiers;
    }

    private PatientCorrelationId matchOnAssigningAuthority(String toAssigningAuthority, CacheService cache) {
        PatientCorrelationId[] correlationIds = cache.match(this);

        if (correlationIds.length == 0)
            return null;

        SystemContext lookupContext = new SystemContext(toAssigningAuthority);
        PatientCorrelationId matchId = null;

        for (PatientCorrelationId correlationId : correlationIds) {
            if (correlationId.context.equals(lookupContext)) {
                matchId = correlationId;
                break;
            }
        }

        if (logger.isDebugEnabled() && matchId != null) logger.debug("Found Match");

        return matchId;
    }

    public PatientIdentifiers getAllIdTranslations(CacheService cache) {
        if (logger.isDebugEnabled()) logger.debug("Looking up id translations");

        PatientIdentifiers patientIdentifiers = new PatientIdentifiers();
        PatientCorrelationId[] correlationIds = cache.match(this);

        for (PatientCorrelationId correlationId: correlationIds) {
            if (isValidAssigningAuthority(correlationId)) {
                PatientIdentifier patientIdentifier = createPatientIdentifier(correlationId);
                patientIdentifiers.add(patientIdentifier);
            }
        }

        if (logger.isDebugEnabled() && patientIdentifiers.size() > 0) logger.debug("Found Matches ");

        return patientIdentifiers;
    }

    private boolean isValidAssigningAuthority(PatientCorrelationId correlationId) {
        return !Strings.isNullOrEmpty(correlationId.context.getAssigningAuthority());
    }
    //correlationId is not this instance
    private PatientIdentifier createPatientIdentifier(PatientCorrelationId correlationId) {
        PatientIdentifier patientIdentifier = new PatientIdentifier();

        patientIdentifier.setUniqueId(correlationId.id);
        patientIdentifier.setAssigningAuthority(correlationId.context.getAssigningAuthority());

        return patientIdentifier;
    }

    @Override
    public boolean equals(Object obj) {
        return obj instanceof PatientCorrelationId && id.equals(((PatientCorrelationId) obj).id) &&
                idType.equals(((PatientCorrelationId) obj).idType) && ((PatientCorrelationId) obj).context.equals(context);
    }

    @Override
    public int hashCode() {
        return id.hashCode() + idType.hashCode() + context.hashCode();
    }

    @Override
    public String toString() {
        return combine(" ");
    }

    private String combine(String delimiter) {
        Joiner joiner = Joiner.on(delimiter).skipNulls();
        return joiner.join(id, idType, context.combine(joiner));
    }
}
