package com.agilex.va.mpi;

import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import com.agilex.healthcare.mobilehealthplatform.patientcorrelation.PatientCorrelationService;
import com.agilex.soap.exceptions.SoapMessageException;
import com.agilex.system.health.SmokeTest;
import com.agilex.system.health.SmokeTestStatus;
import com.agilex.va.mpi.cache.CacheFactory;
import com.agilex.va.mpi.cache.CacheService;
import com.agilex.va.mpi.config.MpiConfig;
import com.agilex.va.mpi.model.IdTransformer;
import com.agilex.va.mpi.model.PatientCorrelationId;
import com.agilex.va.mpi.service.VaMpiService;

import java.util.UUID;

public class PatientCorrelationGateway implements PatientCorrelationService, SmokeTest {

    private static final org.apache.commons.logging.Log logger = org.apache.commons.logging.LogFactory.getLog(PatientCorrelationGateway.class);

    private final CacheService cacheService;
    private final MpiConfig config;

    public PatientCorrelationGateway(MpiConfig config) {
        this(new CacheFactory().getCacheService(), config);
    }

    public PatientCorrelationGateway(CacheService cacheService, MpiConfig config) {
        if (logger.isDebugEnabled()) logger.debug(String.format("Cache service used: %s", cacheService.getClass().getName()));
        this.cacheService = cacheService;
        this.config = config;
        IdTransformer.initVistaFacilities(config.getVistaFacilities());
    }

    @Override
    public PatientIdentifiers getCorrespondIdentifiers(PatientIdentifier patientIdentifier, String assigningAuthority) {
        if (logger.isDebugEnabled()) logger.debug("Getting patient identifiers and assigning authority");

        PatientCorrelationId searchId = PatientCorrelationId.createId(patientIdentifier.getUniqueId(), patientIdentifier.getAssigningAuthority());

        queryForCorrelationIds(patientIdentifier.getAssigningAuthority(), searchId);

        return searchId.getIdTranslation(assigningAuthority, cacheService);
    }

    @Override
    public PatientIdentifiers getCorrespondIdentifiers(PatientIdentifier patientIdentifier) {
        if (logger.isDebugEnabled()) logger.debug("Getting patient identifiers");

        PatientCorrelationId searchId = PatientCorrelationId.createId(patientIdentifier.getUniqueId(), patientIdentifier.getAssigningAuthority());

        queryForCorrelationIds(patientIdentifier.getAssigningAuthority(), searchId);

        return searchId.getAllIdTranslations(cacheService);
    }

    private void queryForCorrelationIds(String assigningAuthority, PatientCorrelationId id) {
        if (cacheService.contains(id))
            return;

        if (logger.isDebugEnabled()) logger.debug("Querying for correlating ids");

        VaMpiService service = new VaMpiService(config);

        PatientCorrelationId[] correlationIds = null;

        try{
            correlationIds = service.getPatientIds(assigningAuthority, id);
        } catch (IllegalArgumentException e) {
            //Don't want to throw errors if there is an issue with patient correlation just return an empty collection.
            logger.error(e);
            correlationIds = CacheService.EMPTY_ID_LIST;
        }  catch (SoapMessageException s) {
            //Don't want to throw errors if there is an issue with patient correlation just return an empty collection.
            logger.error(s);
            correlationIds = CacheService.EMPTY_ID_LIST;
        }

        cacheService.add(id, correlationIds);
    }

    @Override
    public SmokeTestStatus run() {
        SmokeTestStatus status = new SmokeTestStatus();

        try {
            //Use random id to force query (ie. not a cached value)
            getCorrespondIdentifiers(new PatientIdentifier("EDIPI", UUID.randomUUID().toString()));
            status.setSuccess(true);
        } catch (IllegalArgumentException sme) {
            //Do not expect id to be found.  Expect to get a correlation not found message.
            status.setSuccess(true);
        } catch (SoapMessageException sme) {
            status.setSuccess(true);
        } catch (Exception e) {
            status.setException(e);
        }

        status.setConfiguration(config.toString());

        return status;
    }
}
