package com.agilex.va.mpi;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifiers;
import com.agilex.system.health.SmokeTest;
import com.agilex.system.health.SmokeTestStatus;
import com.agilex.va.mpi.cache.CacheFactory;
import com.agilex.va.mpi.config.MpiConfig;
import com.agilex.va.mpi.model.PatientCorrelationId;
import com.agilex.va.mpi.service.VaMpiService;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

import static junit.framework.Assert.assertEquals;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.not;
import static org.junit.Assert.*;

public class PatientCorrelationTest {
    private final String SOURCE_ASSIGNING_AUTHORITY = "EDIPI";
    private final PatientCorrelationId INVALID_ID_TYPE_ID = new PatientCorrelationId("0000000006","NI","IS","USDOD");
    private final PatientCorrelationId INVALID_FORMAT_ID = new PatientCorrelationId("0000000002","NI","200DOD","UDD");

    private final PatientIdentifier KNOWN_PATIENT = new PatientIdentifier(SOURCE_ASSIGNING_AUTHORITY, "000000001");
    private final PatientIdentifier KNOWN_PATIENT_DFN = new PatientIdentifier("DFN-688", "100844");
    private final PatientIdentifier UNKNOWN_PATIENT = new PatientIdentifier(SOURCE_ASSIGNING_AUTHORITY, "1111111111");

    private PatientCorrelationGateway patientCorrelationGateway;
    private MpiConfig config;

    @Rule
    public ExpectedException thrown = ExpectedException.none();

    @Before
    public void init() {
        CacheFactory.defaultCacheType = CacheFactory.CacheType.LastIn;
        String MVI_URL = "";

        Properties p = new Properties();
        try {
          // note: properties file is loading from the classpath; if the int tests are being run through gradle,
          // this file can be placed at the root + /intTest/spring/properties/va-veteran-dev.int.properties
          InputStream is = ClassLoader.getSystemResourceAsStream("spring/properties/va-veteran-dev.int.properties");
          p.load(is);
          MVI_URL = p.getProperty("mvi.endPointUrl");
        } catch (IOException e) {
          MVI_URL = "http://172.16.45.55:8089/mvi/mockVAIdMPort";
        }


//        config = new MpiConfig("http://localhost:8080/mvi/mockVAIdMPort", 5000, 60000, "");
        config = new MpiConfig(MVI_URL, 5000, 60000, "");

       patientCorrelationGateway = new PatientCorrelationGateway(config);
    }

    @Test
    public void translatePatientId() {
        PatientIdentifiers correlatedPatientIdentifiers = patientCorrelationGateway.getCorrespondIdentifiers(KNOWN_PATIENT, "ICN");
        assertEquals(1, correlatedPatientIdentifiers.size());

        PatientIdentifier expectedIcnPatientIdentifier = new PatientIdentifier("ICN", "1008593069V747998");
        assertEquals(expectedIcnPatientIdentifier, correlatedPatientIdentifiers.get(0));
    }

    @Test
    public void translatePatientIdDfn() {
        PatientIdentifiers correlatedPatientIdentifiers = patientCorrelationGateway.getCorrespondIdentifiers(KNOWN_PATIENT_DFN, "ICN");
        assertEquals(1, correlatedPatientIdentifiers.size());

        PatientIdentifier expectedIcnPatientIdentifier = new PatientIdentifier("ICN", "1008593069V747998");
        assertEquals(expectedIcnPatientIdentifier, correlatedPatientIdentifiers.get(0));
    }

    @Test
    public void translatePatientIdDfn2() {
        PatientIdentifiers correlatedPatientIdentifiers = patientCorrelationGateway.getCorrespondIdentifiers(new PatientIdentifier("DFN-688",  "100848"), "ICN");
        assertEquals(1, correlatedPatientIdentifiers.size());

        PatientIdentifier expectedIcnPatientIdentifier = new PatientIdentifier("ICN", "0010010010V123456");
        assertEquals(expectedIcnPatientIdentifier, correlatedPatientIdentifiers.get(0));
    }

    @Test
    public void patientIdValidButNoTranslationFound() {
        PatientIdentifiers correlatedPatientIdentifiers = patientCorrelationGateway.getCorrespondIdentifiers(KNOWN_PATIENT, "ICN");

        PatientIdentifier expectedIcnPatientIdentifier = new PatientIdentifier("ICN", "notvalid");
        assertThat(expectedIcnPatientIdentifier, not(equalTo(correlatedPatientIdentifiers.get(0))));
    }

    @Test
    public void getCorrespondIdentifiersForValidPatient() {
        PatientIdentifiers correlatedPatientIdentifiers = patientCorrelationGateway.getCorrespondIdentifiers(KNOWN_PATIENT);
        assertEquals(3, correlatedPatientIdentifiers.size());

    }

    @Test()
    public void getCorrespondIdentifiersForInvalidPatient() {
        PatientIdentifier patientIdentifier = new PatientIdentifier(SOURCE_ASSIGNING_AUTHORITY, "invalid");
        PatientIdentifiers correlatedPatientIdentifiers = patientCorrelationGateway.getCorrespondIdentifiers(patientIdentifier);
        assertEquals(0, correlatedPatientIdentifiers.size());
    }

    @Test
    public void unknownPatient() {
        PatientIdentifiers correlatedPatientIdentifiers = patientCorrelationGateway.getCorrespondIdentifiers(UNKNOWN_PATIENT, "ICN");
        assertEquals(0, correlatedPatientIdentifiers.size());
    }

    @Test
    public void invalidIdType() {
        thrown.expect(IllegalArgumentException.class);
        thrown.expectMessage("invalid identifier type");

        VaMpiService service = new VaMpiService(config);
        service.getPatientIds(SOURCE_ASSIGNING_AUTHORITY, INVALID_ID_TYPE_ID);
    }

    @Test
    public void invalidInFormat() {
        thrown.expect(IllegalArgumentException.class);
        thrown.expectMessage("Invalid VPID format");

        VaMpiService service = new VaMpiService(config);
        service.getPatientIds(SOURCE_ASSIGNING_AUTHORITY, INVALID_FORMAT_ID);
    }

    @Test
    public void smokeTestSuccess() {
        SmokeTest test = patientCorrelationGateway;
        SmokeTestStatus status = test.run();

        assertTrue(status.isSuccess());
    }


    @Test
    public void smokeTestFailure() {
        MpiConfig config = new MpiConfig("http://hadev.invalid.com:8089/mvi/mockVAIdMPort", 100, 30000, "");
        SmokeTest test = new PatientCorrelationGateway(config);
        SmokeTestStatus status = test.run();

        assertFalse(status.isSuccess());
        assertEquals("java.net.SocketTimeoutException", status.getExceptionType());
    }

}
