package com.agilex.healthcare.mobilehealthplatform.timinglog;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Date;
import java.text.ParseException;

import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.agilex.healthcare.mobilehealthplatform.timinglog.domain.Event;
import com.agilex.healthcare.mobilehealthplatform.timinglog.domain.EventData;
import com.agilex.healthcare.utility.DateHelper;
import com.agilex.healthcare.utility.StreamHelper;
import com.agilex.healthcare.utility.XmlHelper;
import com.agilex.healthcare.utility.XpathHelper;

public class Parser {
	private static final String XML_TEMPLATE_RESOURCE_NAME = "template.xml";
	private Storer storer = new Storer();

	public String echo(String m) {
		System.out.println(String.format("%s -> echo: %s", m, m));
		return m;
	}

	public void processFileContent(String filebody) throws IOException, ParseException {
		System.out.println(String.format("processing request from file content"));
		NodeList events = extractEvents(filebody);
		processEvents(events);
	}

	public void processFile(String filename) throws IOException, ParseException {
		InputStream fstream = new FileInputStream(filename);

		try {
			NodeList events = extractEvents(fstream);
			processEvents(events);
		} finally {
			fstream.close();
		}
	}

	private void processEvents(NodeList events) throws ParseException {
		System.out.println(String.format("found %s event(s)", events.getLength()));
		for (int i = 0; i < events.getLength(); i++) {
			System.out.println(String.format("processing event (%s/%s)", i + 1, events.getLength()));
			Node event = events.item(i);
			readSingleEvent(event);
		}
	}

	private NodeList extractEvents(InputStream fstream) {
		System.out.println("begin gathering events from file stream");
		return extractEvents(StreamHelper.streamToString(fstream));
	}

	private NodeList extractEvents(String body) {
		System.out.println("begin gathering events from file body");
		String template = StreamHelper.streamToString(this.getClass().getResourceAsStream(XML_TEMPLATE_RESOURCE_NAME));
		String formattedLogFile = String.format(template, body);
		Document doc = XmlHelper.loadXml(formattedLogFile);
		NodeList events = XpathHelper.getNodeList(doc, "//log4j:eventSet/log4j:event");
		return events;
	}

	public void readSingleEvent(Node event) throws ParseException {
		Event e = new Event();

		NodeList datatags = extractDataTagsFromEventNode(event);

		setLogDate(event, e);
		setLogThread(event, e);

		for (int i = 0; i < datatags.getLength(); i++) {
			Node eventMessageTag = datatags.item(i);
			processTag(e, eventMessageTag);
		}

		if (e.getId() != null) {
			storer.StoreEvent(e);
		}
	}

	private void setLogThread(Node event, Event e) {
		e.setLogThread(XpathHelper.getString(event, "@thread"));
	}

	private void setLogDate(Node event, Event e) {
		Date logDate = new Date(XpathHelper.getLong(event, "@timestamp"));
		e.setLogDate(DateHelper.formatDateTimeWithMilliseconds(logDate));
	}

	private NodeList extractDataTagsFromEventNode(Node event) {
		String eventMessageBody = XpathHelper.getString(event, "log4j:message");
		Document eventDoc = XmlHelper.loadXml(eventMessageBody);
		NodeList datatags = XpathHelper.getNodeList(eventDoc, "event/data");
		return datatags;
	}

	private void processTag(Event e, Node eventMessageTag) {
		String tag = XpathHelper.getString(eventMessageTag, "@tag");
		String value = XmlHelper.getNodeValue(eventMessageTag);
		processTag(e, tag, value);
	}

	private void processTag(Event e, String tag, String value) {
		if (tag.contentEquals("id"))
			e.setId(value);
		else if (tag.contentEquals("pid"))
			e.setParentEventId(value);
		else if (tag.contentEquals("pid"))
			e.setParentEventId(value);
		else if (tag.contentEquals("start-date"))
			e.setStartDate(DateHelper.parseDateTimeWithMilliseconds(value));
		else if (tag.contentEquals("end-date"))
			e.setEndDate(DateHelper.parseDateTimeWithMilliseconds(value));
		else if (tag.contentEquals("start-ticks"))
			e.setStartTicks(Long.parseLong(value));
		else if (tag.contentEquals("end-ticks"))
			e.setEndTicks(Long.parseLong(value));
		else if (tag.contentEquals("e"))
			e.setEventName(value);
		else if (tag.contentEquals("t-ms"))
			e.setTiming(Integer.parseInt(value));
		else if (tag.contentEquals("threadId"))
			e.setThreadId(value);
		else if (tag.contentEquals("status") || tag.contentEquals("result"))
			e.setResult(value);
		else {
			EventData data = new EventData();
			data.setTag(tag);
			if (value != null && value.length() >= 100)
				value = value.substring(0, 100);
			data.setValue(value);
			e.getEventDatas().add(data);
		}
	}
}
