package com.agilex.soap;

import com.agilex.soap.exceptions.SoapMessageException;
import org.junit.*;
import org.junit.rules.ExpectedException;

import javax.xml.soap.*;
import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

public class ElementAddChildTest {

    private static final String LOCAL_NAME = "local";
    private static final String PREFIX = "pre";
    private static final String NAMESPACE = "ns";
    private static final String ATTRIBUTE_ONE = "a1";
    private static final String ATTRIBUTE_TWO = "a2";
    private static final String VALUE_ONE = "v1";
    private static final String VALUE_TWO = "v2";
    private static final String[] ATTRIBUTES = new String[] {ATTRIBUTE_ONE, ATTRIBUTE_TWO};
    private static final String[] ATTRIBUTE_VALUES = new String[] {VALUE_ONE, VALUE_TWO};

    private Element element;
    private SOAPElement soapElement;

    @Rule
    public ExpectedException thrown = ExpectedException.none();

    @Before
    public void setup() {
        soapElement = mock(SOAPElement.class);
        element = new Element(soapElement);
    }

    @Test
    public void addChildLocalNameOnly() throws SOAPException {
        when(soapElement.addChildElement(LOCAL_NAME)).thenReturn(soapElement);
        Element newElement = element.addChild(LOCAL_NAME);

        assertNotNull(newElement);
        verify(soapElement).addChildElement(LOCAL_NAME);
    }

    @Test
    public void addChild() throws SOAPException {
        when(soapElement.addChildElement(LOCAL_NAME, PREFIX, NAMESPACE)).thenReturn(soapElement);
        Element newElement = element.addChild(LOCAL_NAME, PREFIX, NAMESPACE);

        assertNotNull(newElement);
        verify(soapElement).addChildElement(any(Name.class));
    }

    @Test
    public void addChildWithAttribute() throws SOAPException {
        when(soapElement.addChildElement(LOCAL_NAME)).thenReturn(soapElement);
        Element newElement = element.addChildWithAttribute(LOCAL_NAME, ATTRIBUTE_ONE, VALUE_ONE);

        assertNotNull(newElement);
        verify(soapElement).addChildElement(LOCAL_NAME);
        verify(soapElement).setAttribute(ATTRIBUTE_ONE, VALUE_ONE);
    }

    @Test
    public void addChildWithAttributes() throws SOAPException {
        when(soapElement.addChildElement(LOCAL_NAME)).thenReturn(soapElement);
        Element newElement = element.addChildWithAttributes(LOCAL_NAME, ATTRIBUTES, ATTRIBUTE_VALUES);

        assertNotNull(newElement);
        verify(soapElement).addChildElement(LOCAL_NAME);
        verify(soapElement).setAttribute(ATTRIBUTE_ONE, VALUE_ONE);
        verify(soapElement).setAttribute(ATTRIBUTE_TWO, VALUE_TWO);
    }

    @Test
    public void addChildWithNullAttributes() {
        thrown.expect(SoapMessageException.class);
        thrown.expectMessage("Attributes and attribute values must not be null.");

        element.addChildWithAttributes(LOCAL_NAME, null, ATTRIBUTE_VALUES);
    }

    @Test
    public void addChildWithNullAttributeValues() {
        thrown.expect(SoapMessageException.class);
        thrown.expectMessage("Attributes and attribute values must not be null.");

        element.addChildWithAttributes(LOCAL_NAME, ATTRIBUTES, null);
    }

    @Test
    public void addChildWithAttributesSizeDifference() {
        thrown.expect(SoapMessageException.class);
        thrown.expectMessage("There must be an equal number of attributes and attribute values.");

        element.addChildWithAttributes(LOCAL_NAME, ATTRIBUTES, new String[] {VALUE_ONE});
    }
}
